#include "vimeoaccountslistmodel.h"
#include "session.h"
#include <QSqlRecord>
#include <QSqlField>

VimeoAccountsListModel::VimeoAccountsListModel(Session *session, QObject *parent) :
    QSqlTableModel(parent, QSqlDatabase::database()),
    m_session(session)
{
    QHash<int, QByteArray> roles;
    roles[DisplayNameRole] = "displayName";
    roles[UsernameRole] = "username";
    roles[AccessTokenRole] = "accessToken";
    roles[TokenSecretRole] = "tokenSecret";
    roles[CookiesRole] = "cookies";
    roles[ActiveRole] = "active";
    this->setRoleNames(roles);
    this->setTable("vimeoAccounts");
    this->setEditStrategy(QSqlTableModel::OnManualSubmit);
    this->select();

    if (this->session()) {
        this->connect(this->session()->vimeo(), SIGNAL(signedIn(QString,QString,QString)), this, SLOT(addAccount(QString,QString,QString)));
        this->connect(this, SIGNAL(accountSelected(QString,QString,QString)), this->session()->vimeo(), SLOT(setAccount(QString,QString,QString)));
        this->connect(this, SIGNAL(accountAdded(QString,QString,QString)), this->session()->vimeo(), SLOT(setAccount(QString,QString,QString)));
        this->connect(this, SIGNAL(accountDeleted()), this->session()->vimeo(), SLOT(setAccount()));
        this->connect(this, SIGNAL(gotCookies(QByteArray)), this->session()->cookieJar(), SLOT(setVimeoCookies(QByteArray)));
        this->connect(this, SIGNAL(accountAdded(QString,QString,QString)), this->session()->cookieJar(), SLOT(clearVimeoCookies()));
        this->connect(this, SIGNAL(accountDeleted()), this->session()->cookieJar(), SLOT(clearVimeoCookiesFromDB()));
        this->connect(this, SIGNAL(info(QString)), this->session(), SLOT(onInfo(QString)));
        this->connect(this, SIGNAL(error(QString)), this->session(), SLOT(onError(QString)));
    }
}

VimeoAccountsListModel::~VimeoAccountsListModel() {
    this->clear();
}

void VimeoAccountsListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        this->connect(this->session()->vimeo(), SIGNAL(signedIn(QString,QString,QString)), this, SLOT(addAccount(QString,QString,QString)));
        this->connect(this, SIGNAL(accountSelected(QString,QString,QString)), this->session()->vimeo(), SLOT(setAccount(QString,QString,QString)));
        this->connect(this, SIGNAL(accountAdded(QString,QString,QString)), this->session()->vimeo(), SLOT(setAccount(QString,QString,QString)));
        this->connect(this, SIGNAL(accountDeleted()), this->session()->vimeo(), SLOT(setAccount()));
        this->connect(this, SIGNAL(gotCookies(QByteArray)), this->session()->cookieJar(), SLOT(setVimeoCookies(QByteArray)));
        this->connect(this, SIGNAL(accountAdded(QString,QString,QString)), this->session()->cookieJar(), SLOT(clearVimeoCookies()));
        this->connect(this, SIGNAL(accountDeleted()), this->session()->cookieJar(), SLOT(clearVimeoCookiesFromDB()));
        this->connect(this, SIGNAL(info(QString)), this->session(), SLOT(onInfo(QString)));
        this->connect(this, SIGNAL(error(QString)), this->session(), SLOT(onError(QString)));
    }
}

void VimeoAccountsListModel::setActiveAccount(int row) {
    this->database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField secretField("secret", QVariant::String);
    QSqlField cookiesField("cookies", QVariant::ByteArray);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(this->data(this->index(row, 0), DisplayNameRole));
    usernameField.setValue(this->data(this->index(row, 0), UsernameRole));
    tokenField.setValue(this->data(this->index(row, 0), AccessTokenRole));
    secretField.setValue(this->data(this->index(row, 0), TokenSecretRole));
    cookiesField.setValue(this->data(this->index(row, 0), CookiesRole));
    activeField.setValue(1);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(secretField);
    rec.append(cookiesField);
    rec.append(activeField);

    this->setRecord(row, rec);

    for (int i = 0; i < this->rowCount(); i++) {
        if (i != row) {
            nameField.setValue(this->data(this->index(i, 0), DisplayNameRole));
            usernameField.setValue(this->data(this->index(i, 0), UsernameRole));
            tokenField.setValue(this->data(this->index(i, 0), AccessTokenRole));
            secretField.setValue(this->data(this->index(i, 0), TokenSecretRole));
            cookiesField.setValue(this->data(this->index(i, 0), CookiesRole));
            activeField.setValue(0);

            rec.clear();
            rec.append(nameField);
            rec.append(usernameField);
            rec.append(tokenField);
            rec.append(secretField);
            rec.append(cookiesField);
            rec.append(activeField);

            this->setRecord(i, rec);
        }
    }

    this->submitAll();
}

void VimeoAccountsListModel::switchAccount(int row) {
    this->setActiveAccount(row);

    this->database().open();
    QString user = this->data(this->index(row, 0), UsernameRole).toString();
    QString token = this->data(this->index(row, 0), AccessTokenRole).toString();
    QString secret = this->data(this->index(row, 0), TokenSecretRole).toString();
    QByteArray cookieString = this->data(this->index(row, 0), CookiesRole).toByteArray();

    emit accountSelected(user, token, secret);
    emit gotCookies(cookieString);
}

QVariant VimeoAccountsListModel::data(const QModelIndex &idx, int role) const {
    return QSqlTableModel::data(this->index(idx.row(), role - Qt::DisplayRole));
}

QVariant VimeoAccountsListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

void VimeoAccountsListModel::addAccount(const QString &displayName, const QString &token, const QString &secret) {
    this->database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField secretField("secret", QVariant::String);
    QSqlField cookiesField("cookies", QVariant::ByteArray);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(displayName);
    usernameField.setValue(QVariant::String);
    tokenField.setValue(token);
    secretField.setValue(secret);
    cookiesField.setValue(QVariant::ByteArray);
    activeField.setValue(0);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(secretField);
    rec.append(cookiesField);
    rec.append(activeField);

    this->insertRecord(-1, rec);
    this->setActiveAccount(this->rowCount() - 1);

    emit accountAdded(QString(), token, secret);
    emit countChanged(this->rowCount());
}

void VimeoAccountsListModel::deleteAccount(int row) {
    this->database().open();
    bool signOut = this->data(this->index(row, 0), ActiveRole).toBool();

    if ((this->removeRow(row)) && (this->submitAll())) {
        if (signOut) {
            emit accountDeleted();
        }

        emit countChanged(this->rowCount());
        emit info(tr("Account deleted. Please visit the Vimeo website to revoke access"));
    }
    else {
        emit error(tr("Database error. Unable to delete account"));
    }
}
