#include "dailymotionvideolistmodel.h"
#include "session.h"
#include "feedurls.h"
#include "json.h"
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

DailymotionVideoListModel::DailymotionVideoListModel(Session *session, QObject * parent) :
    AbstractVideoListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1),
    m_refresh(false)
{
}

void DailymotionVideoListModel::setSession(Session *session) {
    m_session = session;
}

void DailymotionVideoListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        if (feed == DAILYMOTION_UPLOADS_FEED) {
            this->connect(this->session()->dailymotion(), SIGNAL(deletedFromUploads(QString)), this, SLOT(onDeletedFromUploads(QString)));
        }
        else if (feed == DAILYMOTION_FAVOURITES_FEED) {
            this->connect(this->session()->dailymotion(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        }
        else if (feed.startsWith("https://api.dailymotion.com/playlist")) {
            this->setPlaylistId(feed.section('/', -2, -2));
            this->connect(this->session()->dailymotion(), SIGNAL(addedToPlaylist(QString,QString)), this, SLOT(onAddedToPlaylist(QString,QString)));
            this->connect(this->session()->dailymotion(), SIGNAL(deletedFromPlaylist(QString,QString)), this, SLOT(onDeletedFromPlaylist(QString,QString)));
        }
        else if (feed.startsWith("https://api.dailymotion.com/group")) {
            this->setGroupId(feed.section('/', -2, -2));
        }

        emit feedChanged(feed);
    }
}

void DailymotionVideoListModel::clear() {
    AbstractVideoListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void DailymotionVideoListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreVideos();
    }
}

void DailymotionVideoListModel::getMoreVideos() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getVideos();
        }
    }
}

void DailymotionVideoListModel::getVideos(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(DAILYMOTION_VIDEO_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void DailymotionVideoListModel::getRelatedVideos(const QString &id) {
    this->setFeed(QString("https://api.dailymotion.com/video/%1/related?limit=30").arg(id));
    this->getVideos();
}

void DailymotionVideoListModel::getPlaylistVideos(const QString &id) {
    this->setFeed(QString("https://api.dailymotion.com/playlist/%1/videos?limit=100").arg(id));
    this->getVideos();
}

void DailymotionVideoListModel::getGroupVideos(const QString &id) {
    this->setFeed(QString("https://api.dailymotion.com/group/%1/videos?limit=30").arg(id));
    this->getVideos();
}

void DailymotionVideoListModel::search(const QString &query, int order, const QString &language) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
        this->setLanguageFilter(language);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createSearchReply(Queries::Videos, this->query(), this->offset(), this->order(), this->languageFilter());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void DailymotionVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideos()));
        this->connect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                VideoItem *video = new VideoItem;
                video->loadDailymotionVideo(entries.at(i).toMap(), this->feed() == DAILYMOTION_FAVOURITES_FEED);
                this->connect(this->session()->dailymotion(), SIGNAL(favouriteChanged(QString,bool)), video, SLOT(onFavouriteChanged(QString,bool)));
                this->appendVideo(QSharedPointer<VideoItem>(video));
            }

            this->setLoading(false);
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);

            if ((!this->playlistId().isEmpty()) && (this->moreResults())) {
                this->getVideos();
            }
        }

        this->disconnect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideos()));
        this->disconnect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionVideoListModel::onDeletedFromUploads(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void DailymotionVideoListModel::onFavouriteChanged(const QString &videoId, bool favourite) {
    if (favourite) {
        this->setRefreshRequired(true);
    }
    else if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void DailymotionVideoListModel::onAddedToPlaylist(const QString &videoId, const QString &playlistId) {
    Q_UNUSED(videoId)

    if (playlistId == this->playlistId()) {
        this->setRefreshRequired(true);
    }
}

void DailymotionVideoListModel::onDeletedFromPlaylist(const QString &videoId, const QString &playlistId) {
    if ((playlistId == this->playlistId()) && (this->removeVideo(videoId))) {
        emit countChanged(this->rowCount());
    }
}

void DailymotionVideoListModel::deleteSelectedVideosFromUploads() {
    if (this->feed() == DAILYMOTION_UPLOADS_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->dailymotion()->deleteFromUploads(videoIds);
    }

    this->selectNone();
}

void DailymotionVideoListModel::addSelectedVideosToFavourites() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->dailymotion()->addToFavourites(videoIds);
    this->selectNone();
}

void DailymotionVideoListModel::deleteSelectedVideosFromFavourites() {
    if (this->feed() == DAILYMOTION_FAVOURITES_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->dailymotion()->deleteFromFavourites(videoIds);
    }

    this->selectNone();
}

void DailymotionVideoListModel::addSelectedVideosToPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->dailymotion()->addToPlaylist(videoIds, playlistId);
    this->selectNone();
}

void DailymotionVideoListModel::addSelectedVideosToNewPlaylist(const QVariantMap &playlist) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->dailymotion()->createPlaylist(playlist, videoIds);
    this->selectNone();
}

void DailymotionVideoListModel::deleteSelectedVideosFromPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->dailymotion()->deleteFromPlaylist(videoIds, playlistId);
    this->selectNone();
}

void DailymotionVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    this->session()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void DailymotionVideoListModel::queueSelectedVideos() {
    this->session()->playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}

VideoItem* DailymotionVideoListModel::getFromQML(int row) const {
    if (VideoItem *video = AbstractVideoListModel::getFromQML(row)) {
        this->connect(this->session()->dailymotion(), SIGNAL(favouriteChanged(QString,bool)), video, SLOT(onFavouriteChanged(QString,bool)));

        return video;
    }

    return 0;
}
