#include "dailymotionuserlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

DailymotionUserListModel::DailymotionUserListModel(Session *session, QObject *parent) :
    AbstractUserListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionUserListModel::setSession(Session *session) {
    m_session = session;
}

void DailymotionUserListModel::clear() {
    AbstractUserListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionUserListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreUsers();
    }
}

void DailymotionUserListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void DailymotionUserListModel::getUsers(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(DAILYMOTION_USER_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void DailymotionUserListModel::search(const QString &query) {
    if (!query.isEmpty()) {
        this->setQuery(query);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createSearchReply(Queries::Users, this->query(), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void DailymotionUserListModel::addUsers() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getUsers()));
        this->connect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                UserItem *user = new UserItem;
                user->loadDailymotionUser(entries.at(i).toMap());
                this->connect(this->session()->dailymotion(), SIGNAL(subscriptionChanged(QString,bool)), user, SLOT(onSubscriptionChanged(QString,bool)));
                this->appendUser(QSharedPointer<UserItem>(user));
            }

            this->setLoading(false);
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getUsers()));
        this->disconnect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionUserListModel::getMoreUsers() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getUsers();
        }
    }
}

UserItem* DailymotionUserListModel::getFromQML(int row) const {
    if (UserItem *user = AbstractUserListModel::getFromQML(row)) {
        this->connect(this->session()->dailymotion(), SIGNAL(subscriptionChanged(QString,bool)), user, SLOT(onSubscriptionChanged(QString,bool)));

        return user;
    }

    return 0;
}
