#include "abstractuserlistmodel.h"
#include "thumbnailcache.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

AbstractUserListModel::AbstractUserListModel(ThumbnailCache *cache, QObject *parent) :
    QAbstractListModel(parent),
    m_cache(cache)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[SubscriptionIdRole] = "subscriptionId";
    roles[UsernameRole] = "username";
    roles[ChannelNameRole] = "channelName";
    roles[AvatarRole] = "avatar";
    roles[AvatarUrlRole] = "avatarUrl";
    roles[VideoCountRole] = "videoCount";
    roles[DescriptionRole] = "description";
    roles[ViewCountRole] = "viewCount";
    roles[SubscriberCountRole] = "subscriberCount";
    roles[SubscriptionRole] = "subscription";
    this->setRoleNames(roles);

#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

AbstractUserListModel::~AbstractUserListModel() {
    m_list.clear();

    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void AbstractUserListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
}

void AbstractUserListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

int AbstractUserListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant AbstractUserListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case SubscriptionIdRole:
        return m_list.at(index.row()).data()->subscriptionId();
    case UsernameRole:
        return m_list.at(index.row()).data()->username();
    case ChannelNameRole:
        return m_list.at(index.row()).data()->channelName();
#ifndef QML_USER_INTERFACE
    case AvatarRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(m_list.at(index.row()).data()->avatarUrl(), QSize(60, 60));
#endif
    case VideoCountRole:
        return m_list.at(index.row()).data()->videoCount();
    case SubscriberCountRole:
        return m_list.at(index.row()).data()->subscriberCount();
    case SubscriptionRole:
        return m_list.at(index.row()).data()->subscription();
    default:
        return QVariant();
    }
}

QVariant AbstractUserListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<UserItem> AbstractUserListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<UserItem>(m_list.at(row));
    }

    return QSharedPointer<UserItem>();
}

UserItem* AbstractUserListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    if ((row >= 0) && (row < m_list.size())) {
        UserItem *user = new UserItem(m_list.at(row).data());
        QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
        return user;
    }
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void AbstractUserListModel::insertUser(int row, QSharedPointer<UserItem> user) {
    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, user);
    this->endInsertRows();
}

void AbstractUserListModel::appendUser(QSharedPointer<UserItem> user) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(user);
    this->endInsertRows();
}

bool AbstractUserListModel::removeUser(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool AbstractUserListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

void AbstractUserListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
