#include "abstractgrouplistmodel.h"
#include "thumbnailcache.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

AbstractGroupListModel::AbstractGroupListModel(ThumbnailCache *cache, QObject *parent) :
    QAbstractListModel(parent),
    m_cache(cache)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[VideoCountRole] = "videoCount";
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[DescriptionRole] = "description";
    this->setRoleNames(roles);

#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

AbstractGroupListModel::~AbstractGroupListModel() {
    m_list.clear();

    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void AbstractGroupListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
}

void AbstractGroupListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

int AbstractGroupListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant AbstractGroupListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case VideoCountRole:
        return m_list.at(index.row()).data()->videoCount();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(120, 90));
#endif
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
    case DescriptionRole:
        return m_list.at(index.row()).data()->description();
    default:
        return QVariant();
    }
}

QVariant AbstractGroupListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<GroupItem> AbstractGroupListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<GroupItem>(m_list.at(row));
    }

    return QSharedPointer<GroupItem>();
}

GroupItem* AbstractGroupListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    if ((row >= 0) && (row < m_list.size())) {
        GroupItem *group = new GroupItem(m_list.at(row).data());
        QDeclarativeEngine::setObjectOwnership(group, QDeclarativeEngine::JavaScriptOwnership);
        return group;
    }
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void AbstractGroupListModel::insertGroup(int row, QSharedPointer<GroupItem> group) {
    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, group);
    this->endInsertRows();
}

void AbstractGroupListModel::appendGroup(QSharedPointer<GroupItem> group) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(group);
    this->endInsertRows();
}

bool AbstractGroupListModel::removeGroup(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool AbstractGroupListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

void AbstractGroupListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
