#include "youtubevideolisttab.h"
#include "youtubevideoinfowindow.h"
#include "youtubeplaylistdialog.h"
#include "videolistdelegate.h"
#include "youtubevideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/session.h"
#include "../base/youtubevideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>
#include <QTimer>

YouTubeVideoListTab::YouTubeVideoListTab(Session *session, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_deleteFromUploadsAction(new QAction(tr("Delete from uploads"), this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_deleteFromFavouritesAction(new QAction(tr("Delete from favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_deleteFromFavouritesContextAction(m_contextMenu->addAction(tr("Delete from favourites"), this, SLOT(deleteVideoFromFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_watchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterPlaylist()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_deleteFromUploadsAction);
    this->addAction(m_addToFavouritesAction);
    this->addAction(m_deleteFromFavouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_watchLaterAction);
    this->addAction(m_addToQueueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_deleteFromUploadsAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromUploads()));
    this->connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    this->connect(m_deleteFromFavouritesAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromFavourites()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    this->connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideosToWatchLaterPlaylist()));
    this->connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    this->onOrientationChanged();
}

void YouTubeVideoListTab::showEvent(QShowEvent *event) {
    AbstractTab::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void YouTubeVideoListTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeVideoListTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeVideoListTab::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    this->setMenuActions();
}

void YouTubeVideoListTab::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->feed() == YOUTUBE_UPLOADS_FEED));
    m_addToFavouritesAction->setEnabled((signedIn) && (m_model->feed() != YOUTUBE_FAVOURITES_FEED));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->feed() == YOUTUBE_FAVOURITES_FEED));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled((signedIn) && (m_model->feed() != YOUTUBE_WATCH_LATER_FEED));
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->feed() == YOUTUBE_UPLOADS_FEED));
    m_addToFavouritesContextAction->setVisible((signedIn) && (m_model->feed() != YOUTUBE_FAVOURITES_FEED));
    m_deleteFromFavouritesContextAction->setVisible((signedIn) && (m_model->feed() == YOUTUBE_FAVOURITES_FEED));
    m_addToPlaylistContextAction->setVisible(signedIn);
    m_watchLaterContextAction->setVisible((signedIn) && (m_model->feed() != YOUTUBE_WATCH_LATER_FEED));
}

void YouTubeVideoListTab::onLoadingChanged(bool loading) {
    m_session->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void YouTubeVideoListTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void YouTubeVideoListTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeVideoListTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, m_session->currentWindow());
    window->show();
}

void YouTubeVideoListTab::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), m_session->currentWindow());
        window->show();
    }
    else {
        m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}

void YouTubeVideoListTab::deleteVideosFromUploads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideosToFavourites() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::deleteVideosFromFavourites() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideosToPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideosToWatchLaterPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideoDownloads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addVideoDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addAudioDownloads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addAudioDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::queueVideos() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Add to queue"), SLOT(queueVideos()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void YouTubeVideoListTab::deleteVideoFromUploads() {
    m_session->youtube()->deleteFromUploads(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListTab::addVideoToFavourites() {
    m_session->youtube()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListTab::deleteVideoFromFavourites() {
    m_session->youtube()->deleteFromFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::FavouriteIdRole).toString()));
}

void YouTubeVideoListTab::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void YouTubeVideoListTab::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void YouTubeVideoListTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    m_session->youtube()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListTab::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListTab::addVideoDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListTab::addAudioDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void YouTubeVideoListTab::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}
