#include "youtubemetadatadialog.h"
#include "../base/session.h"
#include "youtubeuploadcategoryselector.h"
#include "youtubepermissionselector.h"
#include <QLineEdit>
#include <QTextEdit>
#include <QPushButton>
#include <QCheckBox>
#include <QDialogButtonBox>
#include <QScrollArea>
#include <QGridLayout>
#include <QLabel>

YouTubeMetadataDialog::YouTubeMetadataDialog(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_video(video),
    m_titleEdit(new QLineEdit(this)),
    m_descriptionEdit(new QTextEdit(this)),
    m_tagsEdit(new QLineEdit(this)),
    m_categorySelector(new YouTubeUploadCategorySelector(tr("Category"), this)),
    m_responsesSelector(new YouTubePermissionSelector(tr("Allow video repsonses?"), this)),
    m_commentsSelector(new YouTubePermissionSelector(tr("Allow comments?"), this)),
    m_commentVoteCheckbox(new QCheckBox(tr("Allow voting on comments?"), this)),
    m_ratingsCheckbox(new QCheckBox(tr("Allow ratings?"), this)),
    m_privateCheckbox(new QCheckBox(tr("Private?"), this)),
    m_listingsCheckbox(new QCheckBox(tr("Appear in listings?"), this)),
    m_syndicationCheckbox(new QCheckBox(tr("Syndicate to mobile devices?"), this)),
    m_autoPlayCheckbox(new QCheckBox(tr("Allow automatic playback?"), this)),
    m_embedCheckbox(new QCheckBox(tr("Allow embedding?"), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Done"), QDialogButtonBox::AcceptRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Edit metadata"));

    QLabel *titleLabel = new QLabel(tr("Title"), this);
    QLabel *descriptionLabel = new QLabel(tr("Description"), this);
    QLabel *tagsLabel = new QLabel(tr("Tags"), this);
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(titleLabel);
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(descriptionLabel);
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(tagsLabel);
    vbox->addWidget(m_tagsEdit);
    vbox->addWidget(m_categorySelector);
    vbox->addWidget(m_responsesSelector);
    vbox->addWidget(m_commentsSelector);
    vbox->addWidget(m_commentVoteCheckbox);
    vbox->addWidget(m_ratingsCheckbox);
    vbox->addWidget(m_autoPlayCheckbox);
    vbox->addWidget(m_privateCheckbox);
    vbox->addWidget(m_listingsCheckbox);
    vbox->addWidget(m_syndicationCheckbox);
    vbox->addWidget(m_embedCheckbox);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_doneButton->setEnabled(false);
    m_descriptionEdit->setAcceptRichText(false);

    this->connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleTextChanged(QString)));
    this->connect(m_privateCheckbox, SIGNAL(toggled(bool)), this, SLOT(onPrivateToggled(bool)));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(submitMetadata()));
    this->connect(m_session->youtube(), SIGNAL(gotVideoMetadata(QVariantMap)), this, SLOT(setMetadata(QVariantMap)));

    m_session->youtube()->getVideoMetadata(m_video.data()->videoId());

    this->toggleBusy(true);
    this->onOrientationChanged();
}

void YouTubeMetadataDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void YouTubeMetadataDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void YouTubeMetadataDialog::onTitleTextChanged(const QString &text) {
    m_doneButton->setEnabled(!text.isEmpty());
}

void YouTubeMetadataDialog::onPrivateToggled(bool isPrivate) {
    m_listingsCheckbox->setEnabled(!isPrivate);
    m_syndicationCheckbox->setEnabled(!isPrivate);
    m_embedCheckbox->setEnabled(!isPrivate);

    if (isPrivate) {
        m_listingsCheckbox->setChecked(false);
        m_syndicationCheckbox->setChecked(false);
        m_embedCheckbox->setChecked(false);
    }
}

void YouTubeMetadataDialog::setMetadata(const QVariantMap &metadata) {
    this->toggleBusy(false);

    m_titleEdit->setText(metadata.value("title").toString());
    m_descriptionEdit->setText(metadata.value("description").toString());
    m_tagsEdit->setText(metadata.value("tags").toString());
    m_categorySelector->setValue(metadata.value("category").toString());
    m_responsesSelector->setValue(metadata.value("responsesPermission").toString());
    m_commentsSelector->setValue(metadata.value("commentsPermission").toString());
    m_commentVoteCheckbox->setEnabled(metadata.value("commentsPermission").toString() != "denied");
    m_commentVoteCheckbox->setChecked(metadata.value("commentVotingPermission").toString() == "allowed");
    m_ratingsCheckbox->setChecked(metadata.value("ratingsPermission").toString() == "allowed");
    m_privateCheckbox->setChecked(metadata.value("isPrivate").toBool());
    m_listingsCheckbox->setEnabled(!metadata.value("isPrivate").toBool());
    m_listingsCheckbox->setChecked(metadata.value("listingsPermission").toString() == "allowed");
    m_syndicationCheckbox->setEnabled(!metadata.value("isPrivate").toBool());
    m_syndicationCheckbox->setChecked(metadata.value("syndicationPermission").toString() == "allowed");
    m_autoPlayCheckbox->setChecked(metadata.value("autoPlayPermission").toString() == "allowed");
    m_embedCheckbox->setEnabled(!metadata.value("isPrivate").toBool());
    m_embedCheckbox->setChecked(metadata.value("syndicationPermission").toString() == "allowed");

    this->disconnect(m_session->youtube(), SIGNAL(gotVideoMetadata(QVariantMap)), this, SLOT(setMetadata(QVariantMap)));
}

void YouTubeMetadataDialog::submitMetadata() {
    QVariantMap metadata;
    metadata.insert("videoId", m_video.data()->videoId());
    metadata.insert("title", m_titleEdit->text());
    metadata.insert("description", m_descriptionEdit->toPlainText());
    metadata.insert("tags", m_tagsEdit->text());
    metadata.insert("category", m_categorySelector->currentValue());
    metadata.insert("responsesPermission", m_responsesSelector->currentValue());
    metadata.insert("commentsPermission", m_commentsSelector->currentValue());
    metadata.insert("commentVotingPermission", m_commentVoteCheckbox->isChecked() ? "allowed" : "denied");
    metadata.insert("ratingsPermission", m_ratingsCheckbox->isChecked() ? "allowed" : "denied");
    metadata.insert("listingsPermission", m_listingsCheckbox->isChecked() ? "allowed" : "denied");
    metadata.insert("syndicationPermission", m_syndicationCheckbox->isChecked() ? "allowed" : "denied");
    metadata.insert("autoPlayPermission", m_autoPlayCheckbox->isChecked() ? "allowed" : "denied");
    metadata.insert("embedPermission", m_embedCheckbox->isChecked() ? "allowed" : "denied");
    metadata.insert("isPrivate", m_privateCheckbox->isChecked());

    m_session->youtube()->updateVideoMetadata(metadata);
    this->accept();
}
