#include "sharing.h"
#include "json.h"
#include "utils.h"
#include "oauth.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QApplication>
#include <QClipboard>
#include <QDateTime>
#include <QDesktopServices>
#include <qplatformdefs.h>
#ifdef MEEGO_EDITION_HARMATTAN
#include <MDataUri>
#include <maemo-meegotouch-interfaces/shareuiinterface.h>
#endif

using namespace QtJson;
using namespace QtOAuth;

Sharing::Sharing(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_facebookClientId("385476614847489"),
    m_facebookClientSecret("711d38a113d298af9348386957c524e5"),
    m_twitterClientId("YmfKe14OCdVxn9FCL1S77A"),
    m_twitterClientSecret("evSW7txyhlvvPXkezKvJgaZ2lC97iXZc4ZLXLZxfsfc"),
    m_redirectUri("https://sites.google.com/site/marxodian/home/cutetube"),
    m_facebookTimestamp(0),
    m_videoCache(new QList< QSharedPointer<VideoItem> >),
    m_busy(false)
{
}

Sharing::~Sharing() {
    this->clearCache();
}

void Sharing::clearCache() {
    m_videoCache->clear();
}

void Sharing::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            emit busy(message, numberOfOperations);
        }
        else {
            emit busyProgressChanged(1);
        }

        emit busyChanged(isBusy);
    }
}

QUrl Sharing::facebookAuthUrl() const {
    QUrl url("https://www.facebook.com/dialog/oauth");
    url.addQueryItem("client_id", m_facebookClientId);
    url.addQueryItem("redirect_uri", "http://www.facebook.com/connect/login_success.html");
    url.addQueryItem("response_type", "token");
    url.addQueryItem("display", "touch");
    url.addQueryItem("scope", "publish_stream,read_stream");

    return url;
}

void Sharing::signInToFacebook(const QUrl &response) {
    QUrl url(response.toString().replace('#', '?'));
    QString token = url.queryItemValue("access_token");

    if (token.isEmpty()) {
        QString errorString = url.queryItemValue("error");

        if (errorString == "access_denied") {
            emit info(tr("You have denied access to your facebook account"));
        }
        else {
            emit error(tr("Error obtaining facebook authorisation"));
        }
    }
    else {
        emit signedInToFacebook(token);
        emit alert(tr("You are signed in to your facebook account"));
    }
}

QByteArray Sharing::getTwitterOAuthHeader(const QString &method, const QUrl &url, QMap<QString, QString> params) {
    params.insert("oauth_consumer_key", m_twitterClientId);
    params.insert("oauth_consumer_secret", m_twitterClientSecret);

    if (this->userSignedInToTwitter()) {
        params.insert("oauth_token", this->twitterToken());
        params.insert("oauth_token_secret", this->twitterTokenSecret());
    }

    return OAuth::createOAuthHeader(method, url, params);
}

void Sharing::getTwitterAuthUrl() {
    QUrl url("https://api.twitter.com/oauth/request_token");
    QMap<QString, QString> params;
    params.insert("oauth_callback", m_redirectUri);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Authorization", this->getTwitterOAuthHeader("GET", url, params));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTwitterAuthUrl()));
}

void Sharing::checkTwitterAuthUrl() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl url("http://twitter.com?" + reply->readAll());
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");

    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit error(tr("Error obtaining twitter authorisation URL"));
    }
    else {
        this->setTwitterRequestToken(token);
        this->setTwitterRequestSecret(secret);
        QUrl authUrl("http://api.twitter.com/oauth/authorize");
        authUrl.addQueryItem("oauth_token", token);
        emit gotTwitterAuthUrl(authUrl);
    }

    reply->deleteLater();
}

void Sharing::signInToTwitter(const QUrl &response) {
    QString verifier = response.queryItemValue("oauth_verifier");

    if (verifier.isEmpty()) {
        emit error(tr("Error obtaining twitter authorisation verifier"));
    }
    else {
        this->setBusy(true, tr("Signing in"));
        QUrl url("https://api.twitter.com/oauth/access_token");
        QMap<QString, QString> params;
        params.insert("oauth_token", this->twitterRequestToken());
        params.insert("oauth_token_secret", this->twitterRequestSecret());
        params.insert("oauth_verifier", verifier);
        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
        request.setRawHeader("Authorization", this->getTwitterOAuthHeader("GET", url, params));
        QNetworkReply *reply = this->networkAccessManager()->get(request);
        this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedInToTwitter()));
    }
}

void Sharing::checkIfSignedInToTwitter() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl url("http://twitter.com?" + reply->readAll());
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");

    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit error(tr("Error obtaining twitter authorisation"));
    }
    else {
        this->setTwitterRequestToken(QString());
        this->setTwitterRequestSecret(QString());
        emit signedInToTwitter(token, secret);
        emit alert(tr("You are signed in to your twitter account"));
    }

    reply->deleteLater();
}

void Sharing::setFacebookAccessToken(const QString &token) {
    m_facebookToken = token;
    emit userSignedInToFacebookChanged();
}

void Sharing::signOutOfFacebook() {
    this->setFacebookAccessToken(QString());
    emit signedOutOfFacebook();
    emit info(tr("Access token deleted. Please visit the facebook website to revoke access"));
}

void Sharing::setTwitterAccount(const QString &token, const QString &secret) {
    this->setTwitterAccessToken(token);
    this->setTwitterTokenSecret(secret);
    emit userSignedInToTwitterChanged();
}

void Sharing::signOutOfTwitter() {
    this->setTwitterAccount(QString(), QString());
    emit signedOutOfTwitter();
    emit info(tr("Access token deleted. Please visit the twitter website to revoke access"));
}

void Sharing::postToFacebook(const QVariantMap &post) {
    m_post = post;
    this->postToFacebook();
}

void Sharing::postToFacebook() {
    QByteArray postData = "access_token=" + this->facebookToken().toUtf8()
            + "&message=" + m_post.value("body").toString().toUtf8()
            + "&link=" + m_post.value("videoUrl").toString().toUtf8()
            + "&source=" + m_post.value("embedUrl").toString().toUtf8()
            + "&picture=" + m_post.value("thumbnailUrl").toString().toUtf8()
            + "&name=" + m_post.value("title").toString().toUtf8()
            + "&description=" + m_post.value("description").toString().toUtf8();

    QNetworkRequest request(QUrl("https://graph.facebook.com/me/feed"));
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, postData);
    this->connect(reply, SIGNAL(finished()), this, SLOT(facebookPostFinished()));
}

void Sharing::facebookPostFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 400) && (userSignedInToFacebook())) {
        this->setFacebookAccessToken(QString());
        emit signedOutOfFacebook();
        emit reauthenticateForFacebook();
        this->connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            emit alert(tr("Video shared to facebook"));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit error(statusText);
        }

        this->disconnect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
    }

    reply->deleteLater();
}

void Sharing::postToTwitter(const QVariantMap &post) {
    m_post = post;
    this->postToTwitter();
}

void Sharing::postToTwitter() {
    QByteArray body = m_post.value("body").toString().toUtf8().toPercentEncoding();
    QUrl url("http://api.twitter.com/1/statuses/update.json");
    QMap<QString, QString> params;
    params.insert("status", QString(body));
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("Authorization", this->getTwitterOAuthHeader("POST", url, params));
    QNetworkReply *reply = this->networkAccessManager()->post(request, "status=" + body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(twitterPostFinished()));
}

void Sharing::twitterPostFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (this->userSignedInToTwitter())) {
        this->setTwitterAccount(QString(), QString());
        emit signedOutOfTwitter();
        emit reauthenticateForTwitter();
        connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            emit alert(tr("Video shared to twitter"));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit error(statusText);
        }

        this->disconnect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
    }

    reply->deleteLater();
}

void Sharing::shareViaHarmattanShareUI(VideoItem *video) {
#ifdef MEEGO_EDITION_HARMATTAN
    MDataUri duri;
    duri.setMimeType("text/x-url");
    duri.setTextData(video->url().toString());
    duri.setAttribute("title", video->title());
    duri.setAttribute("picture", video->largeThumbnailUrl().toString());
    duri.setAttribute("source", video->embedUrl().toString());

    if (!video->description().isEmpty()) {
        duri.setAttribute("description", video->description());
    }

    if (duri.isValid()) {
        ShareUiInterface shareIf("com.nokia.ShareUi");

        if (shareIf.isValid()) {
            shareIf.share(QStringList(duri.toString()));
        }
        else {
            emit error(tr("Unable to share video"));
        }
    }
    else {
        emit error(tr("Unable to share video"));
    }
#else
    Q_UNUSED(video)
#endif
}

void Sharing::shareViaEmail(const QString &title, const QUrl &url) {
    QDesktopServices::openUrl(QUrl(QString("mailto:?subject=%1&body=%2").arg(title).arg(url.toString())));
}

void Sharing::copyToClipboard(const QUrl &url) {
    QApplication::clipboard()->setText(url.toString());
    emit alert(tr("Copied to clipboard"));
}

void Sharing::getVideos(bool forceRefresh) {
    if ((forceRefresh) || (this->videoRefreshIsDue())) {
        if (this->userSignedInToFacebook()) {
            this->getVideosFromFacebook();
        }
        else if (userSignedInToTwitter()) {
            this->getVideosFromTwitter();
        }
    }
    else if (this->userSignedInToFacebook() && (this->facebookTimeStamp() == 0)) {
        this->getVideosFromFacebook();
    }
    else if (this->userSignedInToTwitter() && (this->lastTweet().isEmpty())) {
        this->getVideosFromTwitter();
    }
}

void Sharing::getVideosFromFacebook() {
    this->setBusy(true, tr("Retrieving videos from facebook"));
    QUrl url("https://graph.facebook.com/me/feed");

    if (this->facebookTimeStamp() > 0) {
        url.addQueryItem("since", QString::number(this->facebookTimeStamp()));
    }

    url.addQueryItem("limit", "100");
    url.addQueryItem("access_token", this->facebookToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFacebookVideos()));
}

void Sharing::checkFacebookVideos() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit error("Network error");
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 400) {
        this->setFacebookAccessToken(QString());
        emit signedOutOfFacebook();
        emit reauthenticateForFacebook();
        this->connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(getVideosFromFacebook()));
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            emit error("Error parsing server response");
        }
        else {
            this->setFacebookTimeStamp(QDateTime::currentMSecsSinceEpoch() / 1000);
            QVariantList results = result.value("data").toList();
            QStringList youtubeIds;
            QStringList dailymotionIds;
            QRegExp ytre("(youtube|youtu.be)", Qt::CaseInsensitive);
            QRegExp dmre("(dailymotion|dai.ly)", Qt::CaseInsensitive);

            for (int i = 0; i < results.size(); i++) {
                QString link = results.at(i).toMap().value("link").toString();

                if (!link.isEmpty()) {
                    if (link.contains(ytre)) {
                        youtubeIds.append(link.section(QRegExp("(v=|/)"), -1).left(11));
                    }
                    else if (link.contains(dmre)) {
                        dailymotionIds.append(link.section("/", -1).left(6));
                    }
                }
            }

            if (!youtubeIds.isEmpty()) {
                youtubeIds.removeDuplicates();
                emit gotYouTubeIds(youtubeIds);
            }
            if (!dailymotionIds.isEmpty()) {
                dailymotionIds.removeDuplicates();
                emit gotDailymotionIds(dailymotionIds);
            }
        }

        if (userSignedInToTwitter()) {
            this->getVideosFromTwitter();
        }
        else {
            this->resetVideoRefreshTime();
        }

        this->disconnect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(getVideosFromFacebook()));
    }

    reply->deleteLater();
}

void Sharing::getVideosFromTwitter() {
    this->setBusy(true, tr("Retreiving videos from twitter"));
    QUrl url("http://api.twitter.com/1/statuses/home_timeline.json");
    url.addQueryItem("count", "200");
    url.addQueryItem("include_entities", "true");
    url.addQueryItem("trim_user", "true");
    url.addQueryItem("include_rts", "false");

    if (!lastTweet().isEmpty()) {
        url.addQueryItem("since_id", this->lastTweet());
    }

    QMap<QString, QString> params;
    params.insert("count", "200");
    params.insert("include_entities", "true");
    params.insert("trim_user", "true");
    params.insert("include_rts", "false");

    if (!lastTweet().isEmpty()) {
        params.insert("since_id", lastTweet());
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Authorization", this->getTwitterOAuthHeader("GET", QUrl("http://api.twitter.com/1/statuses/home_timeline.json"), params));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTwitterVideos()));
}

void Sharing::checkTwitterVideos() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (this->userSignedInToTwitter())) {
        this->setTwitterAccount(QString(), QString());
        emit signedOutOfTwitter();
        emit reauthenticateForTwitter();
        this->connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(getVideosFromTwitter()));
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantList results = Json::parse(response, ok).toList();

        if (!ok) {
            emit error("Error parsing server response");
        }
        else {
            QStringList youtubeIds;
            QStringList dailymotionIds;
            QRegExp ytre("(youtube|youtu.be)", Qt::CaseInsensitive);
            QRegExp dmre("(dailymotion|dai.ly)", Qt::CaseInsensitive);

            if (!results.isEmpty()) {
                this->setLastTweet(results.first().toMap().value("id_str").toString());
            }

            while (!results.isEmpty()) {
                QVariantMap tweet = results.takeFirst().toMap();
                QVariantList urls = tweet.value("entities").toMap().value("urls").toList();

                while (!urls.isEmpty()) {
                    QString url = urls.takeFirst().toMap().value("expanded_url").toString();

                    if (!url.isEmpty()) {
                        if (url.contains(ytre)) {
                            youtubeIds.append(url.section(QRegExp("(v=|/)"), -1).left(11));
                        }
                        else if (url.contains(dmre)) {
                            dailymotionIds.append(url.section("/", -1).left(6));
                        }
                    }
                }
            }

            if (!youtubeIds.isEmpty()) {
                youtubeIds.removeDuplicates();
                emit gotYouTubeIds(youtubeIds);
            }
            if (!dailymotionIds.isEmpty()) {
                dailymotionIds.removeDuplicates();
                emit gotDailymotionIds(dailymotionIds);
            }
        }

        this->disconnect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(getVideosFromTwitter()));
    }

    this->resetVideoRefreshTime();
    reply->deleteLater();
}

void Sharing::addVideosToCache(QList< QSharedPointer<VideoItem> > videos) {
    while (!videos.isEmpty()) {
        m_videoCache->insert(0, videos.takeFirst());
        emit videoAddedToCache(0);
    }

    emit videosAdded();
}
