#include "session.h"
#include <QThread>
#include <QDir>
#include <QApplication>
#include <QClipboard>
#include <QWidget>
#include <QDebug>
#ifndef QML_USER_INTERFACE
#include "thumbnaildownloader.h"
#include <QProgressDialog>
#endif
#ifdef Q_WS_MAEMO_5
#ifndef QML_USER_INTERFACE
#include <QMaemo5InformationBox>
#endif
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QX11Info>
#endif
#ifdef MEEGO_EDITION_HARMATTAN
#include <QTimer>
#endif

#ifdef MEEGO_EDITION_HARMATTAN
using namespace TransferUI;
#endif

Session::Session(QObject *parent) :
    QObject(parent),
    m_settings(new Settings(this)),
    m_database(new Database(this)),
    m_launcher(new VideoLauncher(this)),
    m_youtube(new YouTube(this)),
    m_dailymotion(new Dailymotion(this)),
    m_vimeo(new Vimeo(this)),
    m_sharing(new Sharing(this)),
    m_grabber(new UrlGrabber(UrlGrabber::PlaybackMode, this)),
    m_jar(new CookieJar(CookieJar::DownloadMode, this)),
    m_thumbnailDownloader(0),
    m_cache(0),
    m_thumbnailCache(0),
    m_nam(0),
    m_thread(0),
    m_queue(new PlaybackListModel(this, this)),
    m_progressDialog(0),
    m_transfers(new QList< QSharedPointer<TransferItem> >),
    m_clipboardTimer(0),
    m_transferUI(0)
{
#ifdef MEEGO_EDITION_HARMATTAN
    m_clipboardTimer = new QTimer(this);
    m_clipboardTimer->setInterval(3000);
    m_clipboardTimer->setSingleShot(true);
    m_transferUI = new TransferUI::Client(this);
    this->harmattanTransferUI()->init();
#endif
#ifndef QML_USER_INTERFACE
    m_thread = new QThread(this);
    m_thread->start(QThread::LowestPriority);
    m_thumbnailDownloader = new ThumbnailDownloader(this);
    m_cache = new QCache<QUrl, Thumbnail>();
    m_thumbnailCache = new ThumbnailCache(m_cache);
    m_thumbnailCache->moveToThread(m_thread);
    m_queue = new PlaybackListModel(this, this);
#endif
    this->connect(this->database(), SIGNAL(gotDailymotionAccount(QString,QString,QString)), this->dailymotion(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this->database(), SIGNAL(gotYouTubeAccount(QString,QString,QString)), this->youtube(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this->database(), SIGNAL(gotVimeoAccount(QString,QString,QString)), this->vimeo(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this->database(), SIGNAL(gotFacebookAccount(QString)), this->sharing(), SLOT(setFacebookAccessToken(QString)));
    this->connect(this->database(), SIGNAL(gotTwitterAccount(QString,QString)), this->sharing(), SLOT(setTwitterAccount(QString,QString)));
    this->connect(this->sharing(), SIGNAL(signedInToFacebook(QString)), this->database(), SLOT(storeFacebookAccount(QString)));
    this->connect(this->sharing(), SIGNAL(signedInToTwitter(QString,QString)), this->database(), SLOT(storeTwitterAccount(QString,QString)));
    this->connect(this->sharing(), SIGNAL(signedOutOfFacebook()), this->database(), SLOT(deleteFacebookAccount()));
    this->connect(this->sharing(), SIGNAL(signedOutOfTwitter()), this->database(), SLOT(deleteTwitterAccount()));
    this->connect(this->youtube(), SIGNAL(usernameChanged(QString)), this->database(), SLOT(setYouTubeUsername(QString)));
    this->connect(this->youtube(), SIGNAL(accessTokenRefreshed(QString)), this->database(), SLOT(setYouTubeAccessToken(QString)));
    this->connect(this->dailymotion(), SIGNAL(usernameChanged(QString)), this->database(), SLOT(setDailymotionUsername(QString)));
    this->connect(this->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this->database(), SLOT(setDailymotionAccessToken(QString,QString)));
    this->connect(this->vimeo(), SIGNAL(usernameChanged(QString)), this->database(), SLOT(setVimeoUsername(QString)));
    this->connect(this->settings(), SIGNAL(youtubePlaybackQualityChanged(QSet<int>)), this->urlGrabber(), SLOT(setYouTubeQualitySet(QSet<int>)));
    this->connect(this->settings(), SIGNAL(dailymotionPlaybackQualityChanged(QSet<QByteArray>)), this->urlGrabber(), SLOT(setDailymotionQualitySet(QSet<QByteArray>)));
    this->connect(this->settings(), SIGNAL(vimeoPlaybackQualityChanged(QSet<QByteArray>)), this->urlGrabber(), SLOT(setVimeoQualitySet(QSet<QByteArray>)));
    this->connect(this->settings(), SIGNAL(safeSearchChanged(bool)), this->youtube(), SLOT(setSafeSearch(bool)));
    this->connect(this->settings(), SIGNAL(safeSearchChanged(bool)), this->dailymotion(), SLOT(setSafeSearch(bool)));
    this->connect(this->settings(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
    this->connect(this->settings(), SIGNAL(mediaPlayerChanged(QString)), this->videoLauncher(), SLOT(setMediaPlayer(QString)));
    this->connect(this->settings(), SIGNAL(mediaPlayerCommandChanged(QString)), this->videoLauncher(), SLOT(setCustomCommand(QString)));
    this->connect(this->sharing(), SIGNAL(gotYouTubeIds(QStringList)), this->youtube(), SLOT(getVideosFromIds(QStringList)));
    this->connect(this->sharing(), SIGNAL(gotDailymotionIds(QStringList)), this->dailymotion(), SLOT(getVideosFromIds(QStringList)));
    this->connect(this->youtube(), SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), this->sharing(), SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));
    this->connect(this->dailymotion(), SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), this->sharing(), SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));
    this->connect(this->database(), SIGNAL(gotYouTubeCookies(QByteArray)), this->cookieJar(), SLOT(setYouTubeCookies(QByteArray)));
    this->connect(this->database(), SIGNAL(gotDailymotionCookies(QByteArray)), this->cookieJar(), SLOT(setDailymotionCookies(QByteArray)));
    this->connect(this->database(), SIGNAL(gotVimeoCookies(QByteArray)), this->cookieJar(), SLOT(setVimeoCookies(QByteArray)));
    this->connect(this->cookieJar(), SIGNAL(gotYouTubeCookies(QByteArray)), this->database(), SLOT(setYouTubeCookies(QByteArray)));
    this->connect(this->cookieJar(), SIGNAL(gotDailymotionCookies(QByteArray)), this->database(), SLOT(setDailymotionCookies(QByteArray)));
    this->connect(this->cookieJar(), SIGNAL(gotVimeoCookies(QByteArray)), this->database(), SLOT(setVimeoCookies(QByteArray)));
    this->connect(this->sharing(), SIGNAL(signedInToTwitter(QString,QString)), this->cookieJar(), SLOT(clearTwitterCookies()));

    this->connect(this->database(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));
    this->connect(this->database(), SIGNAL(error(QString)), this, SLOT(onError(QString)));
    this->connect(this->database(), SIGNAL(info(QString)), this, SLOT(onInfo(QString)));
    this->connect(this->sharing(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));
    this->connect(this->sharing(), SIGNAL(error(QString)), this, SLOT(onError(QString)));
    this->connect(this->sharing(), SIGNAL(info(QString)), this, SLOT(onInfo(QString)));
    this->connect(this->youtube(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));
    this->connect(this->youtube(), SIGNAL(error(QString)), this, SLOT(onError(QString)));
    this->connect(this->youtube(), SIGNAL(info(QString)), this, SLOT(onInfo(QString)));
    this->connect(this->dailymotion(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));
    this->connect(this->dailymotion(), SIGNAL(error(QString)), this, SLOT(onError(QString)));
    this->connect(this->dailymotion(), SIGNAL(info(QString)), this, SLOT(onInfo(QString)));
    this->connect(this->vimeo(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));
    this->connect(this->vimeo(), SIGNAL(error(QString)), this, SLOT(onError(QString)));
    this->connect(this->vimeo(), SIGNAL(info(QString)), this, SLOT(onInfo(QString)));
    this->connect(this->urlGrabber(), SIGNAL(error(QString)), this, SLOT(onError(QString)));
    this->connect(this->cookieJar(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));
    this->connect(this->playbackQueue(), SIGNAL(alert(QString)), this, SLOT(onAlert(QString)));

    this->connect(this->youtube(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    this->connect(this->youtube(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
    this->connect(this->youtube(), SIGNAL(alert(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->youtube(), SIGNAL(error(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->youtube(), SIGNAL(info(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->dailymotion(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    this->connect(this->dailymotion(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
    this->connect(this->dailymotion(), SIGNAL(alert(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->dailymotion(), SIGNAL(error(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->dailymotion(), SIGNAL(info(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->vimeo(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    this->connect(this->vimeo(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
    this->connect(this->vimeo(), SIGNAL(alert(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->vimeo(), SIGNAL(error(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->vimeo(), SIGNAL(info(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->sharing(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    this->connect(this->sharing(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
    this->connect(this->sharing(), SIGNAL(alert(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->sharing(), SIGNAL(error(QString)), this, SLOT(cancelProgressDialog()));
    this->connect(this->sharing(), SIGNAL(info(QString)), this, SLOT(cancelProgressDialog()));

    this->connect(QApplication::clipboard(), SIGNAL(dataChanged()), this, SLOT(onClipboardTextChanged()));

    this->settings()->restoreSettings();
    this->database()->restoreAccounts();
#ifndef QML_USER_INTERFACE
    this->setNetworkAccessManager(new QNetworkAccessManager(this));
    this->restoreStoredDownloads();
    this->connect(this->thumbnailCache(), SIGNAL(downloadThumbnail(QUrl)), this->thumbnailDownloader(), SLOT(downloadThumbnail(QUrl)));
    this->connect(this->thumbnailDownloader(), SIGNAL(gotThumbnail(QNetworkReply*)), this->thumbnailCache(), SLOT(onThumbnailDownloaded(QNetworkReply*)));
#endif
}

Session::~Session() {
    if (m_thread) {
        m_thread->quit();
    }

    this->transferList()->clear();
    delete this->transferList();

    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void Session::setNetworkAccessManager(QNetworkAccessManager *manager) {
    m_nam = manager;

    this->networkAccessManager()->setCookieJar(this->cookieJar());
    this->youtube()->setNetworkAccessManager(this->networkAccessManager());
    this->dailymotion()->setNetworkAccessManager(this->networkAccessManager());
    this->vimeo()->setNetworkAccessManager(this->networkAccessManager());
    this->sharing()->setNetworkAccessManager(this->networkAccessManager());
    this->urlGrabber()->setNetworkAccessManager(this->networkAccessManager());
}

void Session::restoreStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers = this->database()->getStoredDownloads();

    for (int i = 0; i < transfers.size(); i++) {
        QSharedPointer<TransferItem> transfer = transfers.at(i);
        transfer.data()->setNetworkAccessManager(this->networkAccessManager());
        transfer.data()->setSession(this);
        transfer.data()->setStatus(Transfers::Paused);
#ifdef MEEGO_EDITION_HARMATTAN
        transfer.data()->setHarmattanTransfer(this->harmattanTransferUI()->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_DOWNLOAD));
#endif
        this->transferList()->append(transfer);
        this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
        this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferPriorityChanged(Transfers::TransferPriority)));
        this->connect(transfer.data(), SIGNAL(saveAsAudioChanged(bool)), this, SLOT(onTransferSaveAsAudioChanged(bool)));
        this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged(QString)));
        this->connect(transfer.data(), SIGNAL(rowChanged(uint)), this, SLOT(onTransferRowChanged(uint)));
    }
}

QString Session::generateTransferId(QString seedFileName) const {
    return QString("%1_%2").arg(QString::number(QDateTime::currentMSecsSinceEpoch())).arg(seedFileName);
}

void Session::addDownloadTransfer(VideoItem *video, bool saveAsAudio, bool notify) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(this->generateTransferId(video->videoId()), video, this->settings()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused, saveAsAudio));
    transfer.data()->setDownloadPath(this->settings()->downloadPath() + "incomplete/");
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    transfer.data()->setSession(this);
    transfer.data()->setRow(this->transferList()->size());
#ifdef MEEGO_EDITION_HARMATTAN
    transfer.data()->setHarmattanTransfer(this->harmattanTransferUI()->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_DOWNLOAD));
#endif
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
    this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferPriorityChanged(Transfers::TransferPriority)));
    this->connect(transfer.data(), SIGNAL(saveAsAudioChanged(bool)), this, SLOT(onTransferSaveAsAudioChanged(bool)));
    this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged(QString)));
    this->connect(transfer.data(), SIGNAL(rowChanged(uint)), this, SLOT(onTransferRowChanged(uint)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < this->settings()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    this->database()->storeDownload(transfer);

    if (notify) {
        this->onAlert(tr("Video download added to transfer queue"));
    }
}

void Session::addDownloadTransfer(QSharedPointer<VideoItem> video, bool saveAsAudio, bool notify) {
    this->addDownloadTransfer(video.data(), saveAsAudio, notify);
}

void Session::addDownloadTransfers(QList<VideoItem *> videos, bool saveAsAudio) {
    while (!videos.isEmpty()) {
        this->addDownloadTransfer(videos.takeFirst(), saveAsAudio, false);
    }

    this->onAlert(tr("Video download(s) added to transfer queue"));
}

void Session::addDownloadTransfers(QList<QSharedPointer<VideoItem> > videos, bool saveAsAudio) {
    while (!videos.isEmpty()) {
        this->addDownloadTransfer(videos.takeFirst().data(), saveAsAudio, false);
    }

    this->onAlert(tr("Video download(s) added to transfer queue"));
}

void Session::addUploadTransfer(const QVariantMap &metadata) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(metadata));
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    transfer.data()->setSession(this);
    transfer.data()->setRow(this->transferList()->size());
#ifdef MEEGO_EDITION_HARMATTAN
    transfer.data()->setHarmattanTransfer(this->harmattanTransferUI()->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_UPLOAD));
#endif
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < this->settings()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    this->onAlert(tr("Video upload added to transfer queue"));
}

QSharedPointer<TransferItem> Session::getTransfer(TransferItem *transfer) const {
    for (int i = 0; i < this->transferList()->size(); i++) {
        if (this->transferList()->at(i).data() == transfer) {
            return this->transferList()->at(i);
        }
    }

    return QSharedPointer<TransferItem>();
}

QSharedPointer<TransferItem> Session::getNextTransfer() const {
    int i = 0;
    int priority = Transfers::HighPriority;

    while (priority <= Transfers::LowPriority) {
        while (i < this->transferList()->size()) {
            QSharedPointer<TransferItem> transfer = this->transferList()->at(i);

            if ((!transfer.isNull()) && (transfer.data()->priority() == priority) && (transfer.data()->status() == Transfers::Queued)) {
                return transfer;
            }

            i++;
        }

        priority++;
        i = 0;
    }

    return QSharedPointer<TransferItem>();
}

bool Session::removeTransfer(QSharedPointer<TransferItem> transfer) {
#ifdef MEEGO_EDITION_HARMATTAN
    if (transfer.data()->harmattanTransfer()) {
        this->harmattanTransferUI()->removeTransfer(transfer.data()->harmattanTransfer()->transferId());
    }
#endif
    int row = this->transferList()->indexOf(transfer);

    if (row >= 0) {
        this->transferList()->removeAt(row);
        emit transferRemoved(row);
        return true;
    }

    return false;
}

void Session::onTransferStatusChanged(Transfers::TransferStatus status) {
    if ((status <= Transfers::Queued) && (this->concurrentTransfers() < this->settings()->maximumConcurrentTransfers())) {
        if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
            transfer.data()->startTransfer();
        }
    }

    if ((status == Transfers::Completed) || (status == Transfers::Cancelled)) {
        if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
            if (status == Transfers::Completed) {
                this->onTransferCompleted(this->getTransfer(transfer));
            }
            else  {
                this->onTransferCancelled(this->getTransfer(transfer));
            }
        }
    }
}

void Session::onTransferRowChanged(uint row) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "rowNumber"), Q_ARG(QVariant, row));
    }
}

void Session::onTransferPriorityChanged(Transfers::TransferPriority priority) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "priority"), Q_ARG(QVariant, priority));
    }
}

void Session::onTransferSaveAsAudioChanged(bool saveAsAudio) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "saveAsAudio"), Q_ARG(QVariant, saveAsAudio));
    }
}

void Session::onTransferFileNameChanged(const QString &fileName) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        QMetaObject::invokeMethod(this->database(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "fileName"), Q_ARG(QVariant, fileName));
    }
}

void Session::onTransferCompleted(QSharedPointer<TransferItem> transfer) {
    QMetaObject::invokeMethod(this->database(), "removeStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer.data()->id()));
    this->onAlert(QString("%1 '%3' %4").arg(transfer.data()->transferType() == Transfers::Upload ? tr("Upload of") : tr("Download of")).arg(transfer.data()->title()).arg(tr("completed")));
    this->removeTransfer(transfer);
}

void Session::onTransferCancelled(QSharedPointer<TransferItem> transfer) {
    QMetaObject::invokeMethod(this->database(), "removeStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer.data()->id()));
    this->removeTransfer(transfer);
}

void Session::onMaximumConcurrentTransfersChanged(int oldMaximum, int newMaximum) {
    if ((newMaximum > oldMaximum) && (newMaximum > this->concurrentTransfers())) {
        if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
            transfer.data()->startTransfer();
        }
    }
}

int Session::concurrentTransfers() const {
    int concurrent = 0;

    for (int i = 0; i < this->transferList()->size(); i++) {
        if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
            if (transfer.data()->status() > Transfers::Queued) {
                concurrent++;
            }
        }
    }

    return concurrent;
}

void Session::onClipboardTextChanged() {
    if (!QApplication::clipboard()) {
        return;
    }
#ifdef MEEGO_EDITION_HARMATTAN
    if (m_clipboardTimer->isActive()) {
        return;
    }
    else {
        m_clipboardTimer->start();
    }
#endif
    QString text = QApplication::clipboard()->text();

    if (text.contains(QRegExp("((http://(www.|m.|)youtube.com/(v/|.+)(v=|list=|)|http://youtu.be/))|((http(s|)://(www.|)dailymotion.com/(video|playlist)/|http://dai.ly/)\\w{6})|(http://vimeo.com/\\d+)"))) {
        this->setClipboardUrl(text);
    }
}

void Session::setClipboardUrl(const QString &url) {
    qDebug() << "Clipboard url: " + url;
    m_clipboardUrl = url;

    if (this->settings()->monitorClipboard()) {
        emit clipboardUrlChanged(url);
    }
}

void Session::onMediaPlayerChanged(const QString &player) {
    if (player == "cutetube") {
        this->disconnect(this->urlGrabber(), SIGNAL(gotVideoUrl(QUrl)), this->videoLauncher(), SLOT(launchVideo(QUrl)));
        this->disconnect(this->urlGrabber(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
        this->disconnect(this->urlGrabber(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
        this->disconnect(this->urlGrabber(), SIGNAL(error(QString)), this, SLOT(cancelProgressDialog()));
    }
    else {
        this->connect(this->urlGrabber(), SIGNAL(gotVideoUrl(QUrl)), this->videoLauncher(), SLOT(launchVideo(QUrl)));
        this->connect(this->urlGrabber(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
        this->connect(this->urlGrabber(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
        this->connect(this->urlGrabber(), SIGNAL(error(QString)), this, SLOT(cancelProgressDialog()));
    }
}

ThumbnailCache* Session::newThumbnailCache() {
#ifndef QML_USER_INTERFACE
    ThumbnailCache *cache = new ThumbnailCache(m_cache);
    cache->moveToThread(m_thread);
    ThumbnailDownloader *downloader = new ThumbnailDownloader;
    downloader->setNetworkAccessManager(networkAccessManager());

    this->connect(cache, SIGNAL(downloadThumbnail(QUrl)), downloader, SLOT(downloadThumbnail(QUrl)));
    this->connect(downloader, SIGNAL(gotThumbnail(QNetworkReply*)), cache, SLOT(onThumbnailDownloaded(QNetworkReply*)));
    this->connect(cache, SIGNAL(destroyed()), downloader, SLOT(deleteLater()));

    return cache;
#endif
    qWarning() << "ThumbnailCache should not be used with QML.";
    return 0;
}

void Session::onAlert(const QString &message) {
#if (!defined (QML_USER_INTERFACE)) && (defined (Q_WS_MAEMO_5))
    QMaemo5InformationBox::information(0, message);
#else
    emit alert(message);
#endif
}

void Session::onInfo(const QString &message) {
#if (!defined (QML_USER_INTERFACE)) && (defined (Q_WS_MAEMO_5))
    QMaemo5InformationBox::information(0, message, QMaemo5InformationBox::NoTimeout);
#else
    emit info(message);
#endif
}

void Session::onError(const QString &message) {
#if (!defined (QML_USER_INTERFACE)) && (defined (Q_WS_MAEMO_5))
    QMaemo5InformationBox::information(0, message, QMaemo5InformationBox::NoTimeout);
#else
    emit error(message);
#endif
}

void Session::showProgressDialog(const QString &message, int numberOfOperations) {
#ifndef QML_USER_INTERFACE
    if (!m_progressDialog) {
        m_progressDialog = new QProgressDialog;
        m_progressDialog->setWindowTitle(tr("Please wait"));
        m_progressDialog->setCancelButtonText(QString());
        m_progressDialog->setMinimumDuration(0);
        m_progressDialog->setMinimum(0);
    }

#ifdef Q_WS_MAEMO_5
    switch (settings()->screenOrientation()) {
    case ScreenOrientation::LockLandscape:
        m_progressDialog->setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        break;
    case ScreenOrientation::LockPortrait:
        m_progressDialog->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        break;
    default:
        m_progressDialog->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    }
#endif
    m_progressDialog->setLabelText(message);
    m_progressDialog->setMaximum(numberOfOperations);
    m_progressDialog->setValue(0);
    m_progressDialog->show();
#else
    emit busy(message, numberOfOperations);
#endif
}

void Session::updateProgressDialog(int progress) {
#ifndef QML_USER_INTERFACE
    if (m_progressDialog) {
        m_progressDialog->setValue(progress);
    }
#else
    emit busyProgressChanged(progress);
#endif
}

void Session::cancelProgressDialog() {
#ifndef QML_USER_INTERFACE
    if ((m_progressDialog) && (m_progressDialog->isVisible())) {
        m_progressDialog->cancel();
    }
#endif
}

void Session::toggleBusy(bool busy) const {
#if (!defined (QML_USER_INTERFACE)) && (defined (Q_WS_MAEMO_5))
    if (QWidget *window = currentWindow()) {
        window->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
    }
#else
    Q_UNUSED(busy)
#endif
}

QWidget* Session::currentWindow() const {
#ifndef QML_USER_INTERFACE
    if (!m_windowStack.isEmpty()) {
        return m_windowStack.last();
    }
#endif
    return 0;
}

void Session::addWindowToStack(QWidget *window) {
#ifndef QML_USER_INTERFACE
    m_windowStack.append(window);
    this->connect(window, SIGNAL(destroyed(QObject*)), this, SLOT(removeWindowFromStack(QObject*)));
#else
    Q_UNUSED(window)
#endif
}

void Session::removeWindowFromStack(QObject *window) {
#ifndef QML_USER_INTERFACE
    m_windowStack.removeOne(qobject_cast<QWidget*>(window));
#else
    Q_UNUSED(window)
#endif
}

void Session::unwindWindowStack() {
#ifndef QML_USER_INTERFACE
    for (int i = m_windowStack.size() - 1; i > 0; i--) {
        m_windowStack.at(i)->close();
    }
#endif
}

void Session::preventDisturb(bool prevent) {
#ifdef Q_WS_MAEMO_5
    if (m_windowStack.isEmpty()) {
        return;
    }

    Atom atom = XInternAtom(QX11Info::display() , "_HILDON_DO_NOT_DISTURB", False);

    if (prevent) {
        long state = 1;
        XChangeProperty(
                    QX11Info::display(),
                    m_windowStack.last()->winId(),
                    atom,
                    XA_INTEGER,
                    32,
                    PropModeReplace,
                    (unsigned char *) &state,
                    1);
    }
    else {
        XDeleteProperty(QX11Info::display(), m_windowStack.last()->winId(), atom);
    }
#else
    Q_UNUSED(prevent)
#endif
}
