#include "cookiejar.h"
#include <QRegExp>

CookieJar::CookieJar(CookieJar::Mode mode, QObject *parent) :
    QNetworkCookieJar(parent),
    m_mode(mode)
{
}

QList<QNetworkCookie> CookieJar::cookiesForUrl(const QUrl &url) const {
    QList<QNetworkCookie> cookies;

    if (url.toString().contains(QRegExp("(dailymotion.com/video|dailymotion.com/cdn)"))) {
        if (m_mode == CookieJar::DownloadMode) {
            cookies = m_dailymotionCookies;
        }

        cookies << QNetworkCookie("family_filter", "false");
    }
    else {
        cookies = QNetworkCookieJar::cookiesForUrl(url);
    }

    return cookies;
}

bool CookieJar::youtubeCookiesSet() const {
    return !m_youtubeCookies.isEmpty();
}

bool CookieJar::dailymotionCookiesSet() const {
    return !m_dailymotionCookies.isEmpty();
}

bool CookieJar::vimeoCookiesSet() const {
    return !m_vimeoCookies.isEmpty();
}

void CookieJar::setYouTubeCookies(const QByteArray &cookieString) {
    this->clearYouTubeCookies();
    m_youtubeCookies = QNetworkCookie::parseCookies(cookieString);
    this->setCookiesFromUrl(m_youtubeCookies, QUrl("http://www.youtube.com"));
}

void CookieJar::setYouTubeCookies() {
    QList<QNetworkCookie> cookies = this->allCookies();
    QByteArray cookieString;

    for (int i = 0; i < cookies.size(); i++) {
        QNetworkCookie cookie = cookies.at(i);

        if (cookie.domain().contains("google")) {
            cookieString.append(cookie.toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
            cookies.removeAt(i);
        }
    }

    this->setAllCookies(cookies);
    this->setYouTubeCookies(cookieString);
    emit gotYouTubeCookies(cookieString);
}

void CookieJar::setDailymotionCookies(const QByteArray &cookieString) {
    this->clearDailymotionCookies();
    m_dailymotionCookies = QNetworkCookie::parseCookies(cookieString);
    this->setCookiesFromUrl(m_dailymotionCookies, QUrl("http://www.dailymotion.com"));
}

void CookieJar::setDailymotionCookies() {
    QList<QNetworkCookie> cookies = this->allCookies();
    QByteArray cookieString;

    for (int i = 0; i < cookies.size(); i++) {
        QNetworkCookie cookie = cookies.at(i);

        if (cookie.domain().contains("dailymotion")) {
            cookieString.append(cookie.toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
            cookies.removeAt(i);
        }
    }

    this->setAllCookies(cookies);
    this->setDailymotionCookies(cookieString);
    emit gotDailymotionCookies(cookieString);
}

void CookieJar::setVimeoCookies(const QByteArray &cookieString) {
    this->clearVimeoCookies();
    m_vimeoCookies = QNetworkCookie::parseCookies(cookieString);
    this->setCookiesFromUrl(m_vimeoCookies, QUrl("http://vimeo.com"));
}

void CookieJar::setVimeoCookies() {
    QList<QNetworkCookie> cookies = this->allCookies();
    QByteArray cookieString;

    for (int i = 0; i < cookies.size(); i++) {
        QNetworkCookie cookie = cookies.at(i);

        if (cookie.domain().contains("vimeo")) {
            cookieString.append(cookie.toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
            cookies.removeAt(i);
        }
    }

    this->setAllCookies(cookies);
    this->setVimeoCookies(cookieString);
    emit gotVimeoCookies(cookieString);
}

void CookieJar::storeYouTubeCookies() {
    QList<QNetworkCookie> cookies = this->cookiesForUrl(QUrl("http://www.youtube.com"));
    m_youtubeCookies = cookies;
    QByteArray cookieString;

    while (!cookies.isEmpty()) {
        cookieString.append(cookies.takeFirst().toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
    }

    emit gotYouTubeCookies(cookieString);
    emit alert(tr("YouTube cookies stored"));
}

void CookieJar::storeDailymotionCookies() {
    QList<QNetworkCookie> cookies = this->cookiesForUrl(QUrl("http://www.dailymotion.com"));
    m_dailymotionCookies = cookies;
    QByteArray cookieString;

    while (!cookies.isEmpty()) {
        cookieString.append(cookies.takeFirst().toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
    }

    emit gotDailymotionCookies(cookieString);
    emit alert(tr("Dailymotion cookies stored"));
}

void CookieJar::storeVimeoCookies() {
    QList<QNetworkCookie> cookies = this->cookiesForUrl(QUrl("http://vimeo.com"));
    m_vimeoCookies = cookies;
    QByteArray cookieString;

    while (!cookies.isEmpty()) {
        cookieString.append(cookies.takeFirst().toRawForm(QNetworkCookie::NameAndValueOnly) + "\n");
    }

    emit gotVimeoCookies(cookieString);
    emit alert(tr("Vimeo cookies stored"));
}

void CookieJar::clearYouTubeCookies() {
    m_youtubeCookies.clear();
    QList<QNetworkCookie> oldCookies = this->allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains(QRegExp("(google|youtube)"))) {
            newCookies.append(cookie);
        }
    }

    this->setAllCookies(newCookies);
}

void CookieJar::clearYouTubeCookiesFromDB(bool displayAlert) {
    this->clearYouTubeCookies();
    emit gotYouTubeCookies(QByteArray());

    if (displayAlert) {
        emit alert(tr("YouTube cookies deleted"));
    }
}

void CookieJar::clearDailymotionCookies() {
    m_dailymotionCookies.clear();
    QList<QNetworkCookie> oldCookies = this->allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains("dailymotion")) {
            newCookies.append(cookie);
        }
    }

    this->setAllCookies(newCookies);
}

void CookieJar::clearDailymotionCookiesFromDB(bool displayAlert) {
    this->clearDailymotionCookies();
    emit gotDailymotionCookies(QByteArray());

    if (displayAlert) {
        emit alert(tr("Dailymotion cookies deleted"));
    }
}

void CookieJar::clearVimeoCookies() {
    m_vimeoCookies.clear();
    QList<QNetworkCookie> oldCookies = this->allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains("vimeo")) {
            newCookies.append(cookie);
        }
    }

    this->setAllCookies(newCookies);
}

void CookieJar::clearVimeoCookiesFromDB(bool displayAlert) {
    this->clearVimeoCookies();
    emit gotVimeoCookies(QByteArray());

    if (displayAlert) {
        emit alert(tr("Vimeo cookies deleted"));
    }
}

void CookieJar::clearTwitterCookies() {
    QList<QNetworkCookie> oldCookies = allCookies();
    QList<QNetworkCookie> newCookies;

    while (!oldCookies.isEmpty()) {
        QNetworkCookie cookie = oldCookies.takeFirst();

        if (!cookie.domain().contains("twitter")) {
            newCookies.append(cookie);
        }
    }

    this->setAllCookies(newCookies);
}
