#include "abstractvideolistmodel.h"
#include "thumbnailcache.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

AbstractVideoListModel::AbstractVideoListModel(ThumbnailCache *cache, QObject *parent) :
    QAbstractListModel(parent),
    m_cache(cache),
    m_thumbnailPressedRow(-1)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[VideoIdRole] = "videoId";
    roles[FavouriteIdRole] = "favouriteId";
    roles[PlaylistVideoIdRole] = "playlistVideoId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[DurationRole] = "duration";
    roles[ViewCountRole] = "viewCount";
    roles[LikesRole] = "likes";
    roles[DislikesRole] = "dislikes";
    roles[RatingCountRole] = "ratingCount";
    roles[RatingRole] = "rating";
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[ArchiveRole] = "archive";
    roles[ServiceRole] = "service";
    roles[FavouriteRole] = "favourite";
    roles[SelectedRole] = "selected";
    roles[ThumbnailPressedRole] = "thumbnailPressed";
    this->setRoleNames(roles);

#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

AbstractVideoListModel::~AbstractVideoListModel() {
    m_list.clear();

    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void AbstractVideoListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
}

void AbstractVideoListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

int AbstractVideoListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant AbstractVideoListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case VideoIdRole:
        return m_list.at(index.row()).data()->videoId();
    case FavouriteIdRole:
        return m_list.at(index.row()).data()->favouriteId();
    case PlaylistVideoIdRole:
        return m_list.at(index.row()).data()->playlistVideoId();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case DurationRole:
        return m_list.at(index.row()).data()->duration();
    case ViewCountRole:
        return m_list.at(index.row()).data()->viewCount();
    case LikesRole:
        return m_list.at(index.row()).data()->likes();
    case DislikesRole:
        return m_list.at(index.row()).data()->dislikes();
    case RatingCountRole:
        return m_list.at(index.row()).data()->ratingCount();
    case RatingRole:
        return m_list.at(index.row()).data()->rating();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(160, 120));
#endif
    case ThumbnailUrlRole:
        return m_list.at(index.row()).data()->thumbnailUrl();
    case ServiceRole:
        return m_list.at(index.row()).data()->service();
    case FavouriteRole:
        return m_list.at(index.row()).data()->favourite();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    case ThumbnailPressedRole:
        return QVariant(m_thumbnailPressedRow == index.row());
    default:
        return QVariant();
    }
}

QVariant AbstractVideoListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool AbstractVideoListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    switch (role) {
    case ThumbnailPressedRole:
        m_thumbnailPressedRow = value.toInt();
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);
    return true;
}

QSharedPointer<VideoItem> AbstractVideoListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<VideoItem>(m_list.at(row));
    }

    return QSharedPointer<VideoItem>();
}

VideoItem* AbstractVideoListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    if ((row >= 0) && (row < m_list.size())) {
        VideoItem *video = new VideoItem(m_list.at(row).data());
        QDeclarativeEngine::setObjectOwnership(video, QDeclarativeEngine::JavaScriptOwnership);
        return video;
    }
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void AbstractVideoListModel::insertVideo(int row, QSharedPointer<VideoItem> video) {
    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, video);
    this->endInsertRows();
}

void AbstractVideoListModel::appendVideo(QSharedPointer<VideoItem> video) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(video);
    this->endInsertRows();
}

bool AbstractVideoListModel::removeVideo(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool AbstractVideoListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

void AbstractVideoListModel::toggleSelected(int row) {
    if (!m_selectedRows.contains(row)) {
        m_selectedRows.append(row);
    }
    else {
        m_selectedRows.removeOne(row);
    }

    emit dataChanged(this->index(row), this->index(row));
}

void AbstractVideoListModel::selectAll() {
    m_selectedRows.clear();

    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}

void AbstractVideoListModel::selectNone() {
    m_selectedRows.clear();

    if (this->rowCount() > 0) {
        emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    }
}

QList< QSharedPointer<VideoItem> > AbstractVideoListModel::selectedItems() const {
    QList< QSharedPointer<VideoItem> > items;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->get(m_selectedRows.at(i)));
    }

    return items;
}

QList<VideoItem*> AbstractVideoListModel::selectedItemsFromQML() const {
    QList<VideoItem*> items;
#ifdef QML_USER_INTERFACE
    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->getFromQML(m_selectedRows.at(i)));
    }
#endif
    return items;
}

void AbstractVideoListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
