#include "youtubevideolistmodel.h"
#include "feedurls.h"
#include <QtNetwork/QNetworkReply>
#include <QtNetwork/QNetworkRequest>

YouTubeVideoListModel::YouTubeVideoListModel(Session *session, QObject *parent) :
    AbstractVideoListModel(session->newThumbnailCache(), parent),
    m_session(session),
    m_favourites(false),
    m_uploads(false),
    m_watchLater(false),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void YouTubeVideoListModel::reset() {
    if (!loading()) {
        m_list.clear();
        setLoading(false);
        setOffset(1);
        setMoreResults(true);
        setError(false);
        getMoreVideos();
    }
}

void YouTubeVideoListModel::setFeed(const QString &afeed) {
    m_feed = afeed + "&fields=" + YOUTUBE_VIDEO_FIELDS;

    if (afeed == YOUTUBE_UPLOADS_FEED) {
        setUploads(true);
        connect(m_session->youtube(), SIGNAL(deletedFromUploads(QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromFeed(QSharedPointer<VideoItem>)));
    }
    else if (afeed == YOUTUBE_FAVOURITES_FEED) {
        setFavourites(true);
        connect(m_session->youtube(), SIGNAL(addedToFavourites(QSharedPointer<VideoItem>)), this, SLOT(onAddedToFeed(QSharedPointer<VideoItem>)));
        connect(m_session->youtube(), SIGNAL(deletedFromFavourites(QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromFeed(QSharedPointer<VideoItem>)));
    }
    else if (afeed == YOUTUBE_WATCH_LATER_FEED) {
        setWatchLater(true);
        connect(m_session->youtube(), SIGNAL(addedToWatchLaterPlaylist(QSharedPointer<VideoItem>)), this, SLOT(onAddedToFeed(QSharedPointer<VideoItem>)));
        connect(m_session->youtube(), SIGNAL(deletedFromWatchLaterPlaylist(QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromFeed(QSharedPointer<VideoItem>)));
    }
    else if (afeed.startsWith(YOUTUBE_PLAYLISTS_BASE_URL)) {
        setPlaylistId(afeed.section('/', -1).section('?', 0, 0));
        connect(m_session->youtube(), SIGNAL(addedToPlaylist(QString,QSharedPointer<VideoItem>)), this, SLOT(onAddedToPlaylist(QString,QSharedPointer<VideoItem>)));
        connect(m_session->youtube(), SIGNAL(deletedFromPlaylist(QString,QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromPlaylist(QString,QSharedPointer<VideoItem>)));
    }
}

void YouTubeVideoListModel::getVideos(const QString &afeed) {
    if (!afeed.isEmpty()) {
        setFeed(afeed);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->youtube()->createReply(feed(), offset());
    connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void YouTubeVideoListModel::getMoreVideos() {
    if ((moreResults()) && (!loading())) {
        if (!query().isEmpty()) {
            search();
        }
        else {
            getVideos();
        }
    }
}

void YouTubeVideoListModel::getRelatedVideos(const QString &id) {
    setFeed(YOUTUBE_VIDEOS_BASE_URL + QString("/%1/related?v=2.1&max-results=30").arg(id));
    getVideos();
}

void YouTubeVideoListModel::getPlaylistVideos(const QString &id) {
    setFeed(YOUTUBE_PLAYLISTS_BASE_URL + QString("/%1?v=2.1&max-results=50").arg(id));
    getVideos();
}

void YouTubeVideoListModel::search(const QString &aquery, Queries::QueryOrder anorder, Queries::TimeFilter time, Queries::DurationFilter duration, const QString &language) {
    if (!aquery.isEmpty()) {
        setQuery(aquery);
        setOrder(anorder);
        setTimeFilter(time);
        setDurationFilter(duration);
        setLanguageFilter(language);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->youtube()->createSearchReply(Queries::Videos, query(), offset(), order(), timeFilter(), durationFilter(), languageFilter());
    connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void YouTubeVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        setLoading(false);
        setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(m_session->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideos()));
        connect(m_session->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
        m_session->youtube()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.count(); i++) {
            appendVideo(QSharedPointer<VideoItem>(new VideoItem(entries.at(i), favourites())));
        }

        setLoading(false);
        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();
        setMoreResults((totalResults > rowCount()) && (!entries.isEmpty()));

        if ((!playlistId().isEmpty()) && (moreResults())) {
            setOffset(offset() + 50);
            getVideos();
        }
        else {
            setOffset(offset() + 30);
        }

        disconnect(m_session->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideos()));
        disconnect(m_session->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    reply->deleteLater();
}

void YouTubeVideoListModel::onAddedToFeed(QSharedPointer<VideoItem> video) {
    insertVideo(0, QSharedPointer<VideoItem>(video));
}

void YouTubeVideoListModel::onDeletedFromFeed(QSharedPointer<VideoItem> video) {
    removeVideo(video.data()->id());
}

void YouTubeVideoListModel::onAddedToPlaylist(const QString &id, QSharedPointer<VideoItem> video) {
    if ((id == playlistId()) && (!video.isNull())) {
        appendVideo(video);
    }
}

void YouTubeVideoListModel::onDeletedFromPlaylist(const QString &id, QSharedPointer<VideoItem> video) {
    if ((id == playlistId()) && (!video.isNull())) {
        removeVideo(video.data()->id());
    }
}
