#ifndef YOUTUBE_H
#define YOUTUBE_H

#include "enums.h"
#include "feedurls.h"
#include "videoitem.h"
#include "playlistitem.h"
#include "useritem.h"
#include "commentitem.h"
#include "newplaylist.h"
#include "newcomment.h"
#include "videometadata.h"
#include <QObject>
#include <QByteArray>
#include <QString>
#include <QtNetwork/QNetworkAccessManager>
#include <QUrl>
#include <QVariantMap>

class QNetworkAccessManager;
class QNetworkReply;

class YouTube : public QObject {
    Q_OBJECT

public:
    explicit YouTube(QObject *parent = 0);
    ~YouTube();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString username() const { return m_user; }
    inline bool userSignedIn() const { return !accessToken().isEmpty(); }
    inline QString uploadsFeed() const { return YOUTUBE_UPLOADS_FEED; }
    inline QString favouritesFeed() const { return YOUTUBE_FAVOURITES_FEED; }
    inline QString recommendedFeed() const { return YOUTUBE_RECOMMENDED_FEED; }
    inline QString playlistsFeed() const { return YOUTUBE_PLAYLISTS_FEED; }
    inline QString subscriptionsFeed() const { return YOUTUBE_SUBSCRIPTIONS_FEED; }
    inline QString newSubscriptionVideosFeed() const { return YOUTUBE_NEW_SUBSCRIPTION_VIDEOS_FEED; }
    inline QString watchLaterFeed() const { return YOUTUBE_WATCH_LATER_FEED; }
    inline QString watchHistoryFeed() const { return YOUTUBE_WATCH_HISTORY_FEED; }
    QNetworkReply* createReply(QString feed, int offset = 0);
    QNetworkReply* createSearchReply(
            Queries::QueryType queryType,
            const QString &query,
            int offset = 1,
            Queries::QueryOrder order = Queries::Relevance,
            Queries::TimeFilter time = Queries::AllTime,
            Queries::DurationFilter duration = Queries::Any,
            const QString &language = QString());
    QUrl authUrl() const;
    Q_INVOKABLE void signIn(const QString &displayName, const QString &code);
    inline QString accessToken() const { return m_token; }
    inline QString developerKey() const { return m_key; }
    inline bool safeSearch() const { return m_safeSearch; }
    inline QList< QSharedPointer<PlaylistItem> > * playlists() const { return m_playlistCache; }
    inline QList< QSharedPointer<UserItem> > * subscriptions() const { return m_subscriptionCache; }
    inline bool playlistsLoaded() const { return m_playlistCacheLoaded; }
    inline bool subscriptionsLoaded() const { return m_subscriptionCacheLoaded; }
    inline bool busy() const { return m_busy; }

public slots:
    void getPlaylists(int offset = 1);
    void getSubscriptions(int offset = 1);
    inline void setSafeSearch(bool safe) { m_safeSearch = safe; }
    void setAccount(const QString &user = QString(), const QString &token = QString(), const QString &refresh = QString());
    inline void setUsername(const QString &user) { m_user = user; emit usernameChanged(user); }
    void deleteFromUploads(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deleteFromUploads(QSharedPointer<VideoItem> video, bool appendToList = true);
    void addToFavourites(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void addToFavourites(QSharedPointer<VideoItem> video, bool appendToList = true);
    void deleteFromFavourites(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deleteFromFavourites(QSharedPointer<VideoItem> video, bool appendToList = true);
    void addToPlaylist(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >(), const QString &playlistId = QString());
    void addToPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId = QString(), bool appendToList = true);
    void deleteFromPlaylist(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >(), const QString &playlistId = QString());
    void deleteFromPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId = QString(), bool appendToList = true);
    void addToWatchLaterPlaylist(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void addToWatchLaterPlaylist(QSharedPointer<VideoItem> video, bool appendToList = true);
    void deleteFromWatchLaterPlaylist(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deleteFromWatchLaterPlaylist(QSharedPointer<VideoItem> video, bool appendToList = true);
    void createPlaylist(const NewPlaylist &playlist = NewPlaylist(), QSharedPointer<VideoItem> video = QSharedPointer<VideoItem>(), QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deletePlaylist(const QString &id = QString());
    void subscribe(QSharedPointer<UserItem> user = QSharedPointer<UserItem>());
    void unsubscribe(QSharedPointer<UserItem> user = QSharedPointer<UserItem>());
    void rateVideo(QSharedPointer<VideoItem> video = QSharedPointer<VideoItem>(), const QString &likeOrDislike = QString());
    void addComment(const NewComment &comment = NewComment());
    void replyToComment(const NewComment &comment = NewComment());
    void updateVideoMetadata(QSharedPointer<VideoItem> video = QSharedPointer<VideoItem>(), const VideoMetadata &metadata = VideoMetadata());
    void refreshAccessToken();
    void getFullVideo(QString id);
    void getCurrentUserProfile();
    void getUserProfile(const QString &id = QString());
    void linkGoogleAccount(QString username);
    void checkUsernameAvailability(QString username);
    void cancelLinkGoogleAccount();
    void subscribedToChannel(QSharedPointer<UserItem> user = QSharedPointer<UserItem>());
    void getVideosFromIds(QStringList ids);
    void getVideoMetadata(const QString &id = QString());
    void getFullComment(const QString &videoId, const QString &commentId);

private slots:
    void postRequest(const QUrl &url, const QByteArray &xml);
    void putRequest(const QUrl &url, const QByteArray &xml);
    void deleteRequest(const QUrl &url);
    void postFinished();
    void checkIfSignedIn();
    void checkTokenRefresh();
    void onVideoDeleted();
    void onVideoMetadataUpdated();
    void onAddedToFavourites(const QString &response);
    void onDeletedFromFavourites();
    void onAddedToPlaylist();
    void onDeletedFromPlaylist();
    void onAddedToWatchLaterPlaylist(const QString &response);
    void onDeletedFromWatchLaterPlaylist();
    void onPlaylistCreated(const QString &response);
    void onPlaylistDeleted();
    void onSubscribed(const QString &response);
    void onUnsubscribed();
    void onVideoRated();
    void onCommentAdded(const QString &response);
    void onVideoActionError(const QString &error);
    void onPlaylistActionError(const QString &error);
    void onUserActionError(const QString &error);
    void onCommentActionError(const QString &error);
    void checkFullVideo();
    void checkUserProfile();
    void checkCurrentUserProfile();
    void checkSuggestedUsernames();
    void onGoogleAccountLinked(const QString &response);
    void addPlaylists();
    void addSubscriptions();
    void checkFullVideos();
    void checkVideoMetadata();
    void checkFullComment();
    void checkCachePlaylist();
    void checkAddedComment();

private:
    inline void setAccessToken(const QString &token) { m_token = token; emit userSignedInChanged(); }
    inline QString refreshToken() const { return m_refreshToken; }
    inline void setRefreshToken(const QString &token) { m_refreshToken = token; }
    void clearCache();
    void setPlaylistsLoaded(bool loaded) { m_playlistCacheLoaded = loaded; emit playlistsLoadedChanged(); }
    void setSubscriptionsLoaded(bool loaded) { m_subscriptionCacheLoaded = loaded; emit subscriptionsLoadedChanged(); }
    void addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist);
    bool removePlaylistFromCache(const QString &id);
    void addNewSubscriptionToCache(QSharedPointer<UserItem> user);
    bool removeSubscriptionFromCache(const QString &id);
    void updatePlaylistVideoCount(const QString &id, int change);
    void updatePlaylistThumbnail(const QString &id, const QString &thumbnailUrl);
    void getPlaylistForCache(const QString &id);
    void getAddedComment(const QString &videoId, const QString &commentId);
    inline void setBusy(bool busy) { m_busy = busy; emit busyChanged(); }

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged();
    void info(const QString &message);
    void warning(const QString &message);
    void usernameChanged(const QString &username);
    void postSuccessful(const QString &response);
    void postFailed(const QString &error);
    void deletedFromUploads(QSharedPointer<VideoItem> video);
    void addedToFavourites(QSharedPointer<VideoItem> video);
    void deletedFromFavourites(QSharedPointer<VideoItem> video);
    void addedToPlaylist(const QString &id, QSharedPointer<VideoItem> video);
    void deletedFromPlaylist(const QString &id, QSharedPointer<VideoItem> video);
    void addedToWatchLaterPlaylist(QSharedPointer<VideoItem> video);
    void deletedFromWatchLaterPlaylist(QSharedPointer<VideoItem> video);
    void commentAdded(QSharedPointer<CommentItem> comment);
    void videoRated(QSharedPointer<VideoItem> video);
    void userSignedInChanged();
    void signedIn(const QString &displayName, const QString &token, const QString &refresh);
    void newAccountSet();
    void accessTokenRefreshed(const QString &token);
    void refreshError();
    void requestToLinkGoogleAccount();
    void googleAccountLinked();
    void gotSuggestedUsernames(const QStringList &usernames);
    void usernameUnavailable();
    void usernameAvailable();
    void gotVideo(QSharedPointer<VideoItem> video);
    void gotUser(QSharedPointer<UserItem> user);
    void playlistsLoadedChanged();
    void subscriptionsLoadedChanged();
    void allPlaylistsLoaded();
    void allSubscriptionsLoaded();
    void playlistAddedToCache(int row);
    void playlistRemovedFromCache(int row);
    void playlistUpdated(int row);
    void subscriptionAddedToCache(int row);
    void subscriptionRemovedFromCache(int row);
    void gotVideosFromIds(QList< QSharedPointer<VideoItem> > videos);
    void gotVideoMetadata(const VideoMetadata &metadata);
    void gotComment(QSharedPointer<CommentItem> comment);

private:
    QNetworkAccessManager *m_nam;
    QString m_clientId;
    QString m_clientSecret;
    QString m_key;
    QString m_user;
    QString m_token;
    QString m_refreshToken;
    bool m_safeSearch;
    QList< QSharedPointer<VideoItem> > m_videoActionList;
    QSharedPointer<VideoItem> m_videoAction;
    NewPlaylist m_playlistAction;
    QSharedPointer<UserItem> m_userAction;
    NewComment m_commentAction;
    QString m_playlistActionId;
    QString m_ratingAction;
    VideoMetadata m_metadataAction;
    int m_actionsProcessed;
    QString m_linkUsername;
    QList< QSharedPointer<PlaylistItem> > *m_playlistCache;
    QList< QSharedPointer<UserItem> > *m_subscriptionCache;
    bool m_playlistCacheLoaded;
    bool m_subscriptionCacheLoaded;
    bool m_busy;
    QHash<int, QString> m_queryOrders;
    QHash<int, QString> m_timeFilters;
    QHash<int, QString> m_durationFilters;
};

#endif // YOUTUBE_H
