#include "dailymotionmetadatadialog.h"
#include <QLabel>

DailymotionMetadataDialog::DailymotionMetadataDialog(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_video(video),
    m_titleEdit(new QLineEdit(this)),
    m_descriptionEdit(new QTextEdit(this)),
    m_tagsEdit(new QLineEdit(this)),
    m_categorySelector(new DailymotionUploadCategorySelector(tr("Channel"), this)),
    m_commentsCheckbox(new QCheckBox(tr("Allow comments?"), this)),
    m_privateCheckbox(new QCheckBox(tr("Private?"), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Done"), QDialogButtonBox::AcceptRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Edit metadata"));

    QLabel *titleLabel = new QLabel(tr("Title"), this);
    QLabel *descriptionLabel = new QLabel(tr("Description"), this);
    QLabel *tagsLabel = new QLabel(tr("Tags"), this);
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(titleLabel);
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(descriptionLabel);
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(tagsLabel);
    vbox->addWidget(m_tagsEdit);
    vbox->addWidget(m_categorySelector);
    vbox->addWidget(m_commentsCheckbox);
    vbox->addWidget(m_privateCheckbox);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_doneButton->setEnabled(false);
    m_descriptionEdit->setAcceptRichText(false);

    connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleTextChanged(QString)));
    connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(submitMetadata()));
    connect(m_session->dailymotion(), SIGNAL(gotVideoMetadata(VideoMetadata)), this, SLOT(setMetadata(VideoMetadata)));

    m_session->dailymotion()->getVideoMetadata(m_video.data()->videoId());

    toggleBusy(true);
    onOrientationChanged();
}

void DailymotionMetadataDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void DailymotionMetadataDialog::setPortraitLayout() {
    setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void DailymotionMetadataDialog::onTitleTextChanged(const QString &text) {
    m_doneButton->setEnabled(!text.isEmpty());
}

void DailymotionMetadataDialog::setMetadata(const VideoMetadata &metadata) {
    toggleBusy(false);

    m_titleEdit->setText(metadata.title());
    m_descriptionEdit->setText(metadata.description());
    m_tagsEdit->setText(metadata.tags());
    m_categorySelector->setValue(metadata.category());
    m_commentsCheckbox->setChecked(metadata.commentVotingPermission() == "true");
    m_privateCheckbox->setChecked(metadata.isPrivate());

    disconnect(m_session->dailymotion(), SIGNAL(gotVideoMetadata(VideoMetadata)), this, SLOT(setMetadata(VideoMetadata)));
}

void DailymotionMetadataDialog::submitMetadata() {
    VideoMetadata metadata;
    metadata.setTitle(m_titleEdit->text());
    metadata.setDescription(m_descriptionEdit->toPlainText());
    metadata.setTags(m_tagsEdit->text());
    metadata.setCategory(m_categorySelector->currentValue());
    metadata.setCommentsPermission(m_commentsCheckbox->isChecked() ? "true" : "false");
    metadata.setPrivate(m_privateCheckbox->isChecked());

    m_session->dailymotion()->updateVideoMetadata(m_video, metadata);
    accept();
}
