#include "accountsdialog.h"
#include "accountdelegate.h"
#include "youtubeauthdialog.h"
#include "dailymotionauthdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"

AccountsDialog::AccountsDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_dailymotionModel(new DailymotionAccountsListModel(m_session->dailymotion(), m_session->cookieJar(), this)),
    m_youtubeModel(new YouTubeAccountsListModel(m_session->youtube(), m_session->cookieJar(), this)),
    m_dailymotionView(new ListView(this)),
    m_youtubeView(new ListView(this)),
    m_dailymotionEdit(new QLineEdit(this)),
    m_youtubeEdit(new QLineEdit(this)),
    m_dailymotionLabel(new QLabel(m_session->dailymotion()->userSignedIn() ? tr("You are signed in to a Dailymotion account") : tr("Sign in to a Dailymotion account"), this)),
    m_youtubeLabel(new QLabel(m_session->youtube()->userSignedIn() ? tr("You are signed in to a YouTube account") : tr("Sign in to a YouTube account"), this)),
    m_facebookLabel(new QLabel(m_session->sharing()->userSignedInToFacebook() ? tr("You are signed in to a facebook account") : tr("Sign in to a facebook account"), this)),
    m_twitterLabel(new QLabel(m_session->sharing()->userSignedInToTwitter() ? tr("You are signed in to a twitter account") : tr("Sign in to a twitter account"), this)),
    m_youtubeButton(new QPushButton(QIcon::fromTheme("general_add"), QString(), this)),
    m_dailymotionButton(new QPushButton(QIcon::fromTheme("general_add"), QString(), this)),
    m_facebookButton(new QPushButton(m_session->sharing()->userSignedInToFacebook() ? tr("Sign out") : tr("Sign in"), this)),
    m_twitterButton(new QPushButton(m_session->sharing()->userSignedInToTwitter() ? tr("Sign out") : tr("Sign in"), this)),
    m_scrollArea(new QScrollArea(this)),
    m_contextMenu(new QMenu(this)),
    m_dailymotionDeleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deleteDailymotionAccount()))),
    m_youtubeDeleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deleteYouTubeAccount()))),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Accounts"));

    m_dailymotionView->setModel(m_dailymotionModel);
    m_dailymotionView->setItemDelegate(new AccountDelegate(m_dailymotionView));
    m_youtubeView->setModel(m_youtubeModel);
    m_youtubeView->setItemDelegate(new AccountDelegate(m_youtubeView));
    m_dailymotionEdit->setPlaceholderText(tr("Display name"));
    m_youtubeEdit->setPlaceholderText(tr("Display name"));
    m_dailymotionButton->setFixedSize(70, 70);
    m_youtubeButton->setFixedSize(70, 70);
    m_dailymotionLabel->setWordWrap(true);
    m_dailymotionLabel->setAlignment(Qt::AlignCenter);
    m_youtubeLabel->setWordWrap(true);
    m_youtubeLabel->setAlignment(Qt::AlignCenter);
    m_facebookLabel->setWordWrap(true);
    m_facebookLabel->setAlignment(Qt::AlignCenter);
    m_twitterLabel->setWordWrap(true);
    m_twitterLabel->setAlignment(Qt::AlignCenter);

    SeparatorLabel *dailymotionSeparator = new SeparatorLabel("Dailymotion", this);
    SeparatorLabel *youtubeSeparator = new SeparatorLabel("YouTube", this);
    SeparatorLabel *facebookSeparator = new SeparatorLabel("facebook", this);
    SeparatorLabel *twitterSeparator = new SeparatorLabel("twitter", this);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(youtubeSeparator, 0, 0, 1, 2);
    grid->addWidget(m_youtubeLabel, 1, 0, 1, 2);
    grid->addWidget(m_youtubeEdit, 2, 0);
    grid->addWidget(m_youtubeButton, 2, 1);
    grid->addWidget(m_youtubeView, 3, 0, 1, 2);
    grid->addWidget(dailymotionSeparator, 4, 0, 1, 2);
    grid->addWidget(m_dailymotionLabel, 5, 0, 1, 2);
    grid->addWidget(m_dailymotionEdit, 6, 0);
    grid->addWidget(m_dailymotionButton, 6, 1);
    grid->addWidget(m_dailymotionView, 7, 0, 1, 2);
    grid->addWidget(facebookSeparator, 8, 0, 1, 2);
    grid->addWidget(m_facebookLabel, 9, 0, 1, 2);
    grid->addWidget(m_facebookButton, 10, 0, 1, 2, Qt::AlignHCenter);
    grid->addWidget(twitterSeparator, 11, 0, 1, 2);
    grid->addWidget(m_twitterLabel, 12, 0, 1, 2);
    grid->addWidget(m_twitterButton, 13, 0, 1, 2, Qt::AlignHCenter);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);

    m_grid->addWidget(m_scrollArea, 0, 0);

    connect(m_youtubeEdit, SIGNAL(returnPressed()), this, SLOT(showYouTubeAuthDialog()));
    connect(m_youtubeButton, SIGNAL(clicked()), this, SLOT(showYouTubeAuthDialog()));
    connect(m_youtubeView, SIGNAL(clicked(QModelIndex)), this, SLOT(switchYouTubeAccount(QModelIndex)));
    connect(m_youtubeView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showYouTubeContextMenu(QPoint)));
    connect(m_youtubeModel, SIGNAL(alert(QString)), m_session, SLOT(showInfoBanner(QString)));
    connect(m_youtubeModel, SIGNAL(warning(QString)), m_session, SLOT(showNoteBanner(QString)));
    connect(m_youtubeModel, SIGNAL(info(QString)), m_session, SLOT(showNoteBanner(QString)));
    connect(m_dailymotionEdit, SIGNAL(returnPressed()), this, SLOT(showDailymotionAuthDialog()));
    connect(m_dailymotionButton, SIGNAL(clicked()), this, SLOT(showDailymotionAuthDialog()));
    connect(m_dailymotionView, SIGNAL(clicked(QModelIndex)), this, SLOT(switchDailymotionAccount(QModelIndex)));
    connect(m_dailymotionView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showDailymotionContextMenu(QPoint)));
    connect(m_dailymotionModel, SIGNAL(alert(QString)), m_session, SLOT(showInfoBanner(QString)));
    connect(m_dailymotionModel, SIGNAL(warning(QString)), m_session, SLOT(showNoteBanner(QString)));
    connect(m_dailymotionModel, SIGNAL(info(QString)), m_session, SLOT(showNoteBanner(QString)));
    connect(m_facebookButton, SIGNAL(clicked()), this, SLOT(onFacebookButtonClicked()));
    connect(m_twitterButton, SIGNAL(clicked()), this, SLOT(onTwitterButtonClicked()));
    connect(m_session->sharing(), SIGNAL(userSignedInToFacebookChanged()), this, SLOT(onSignedInOrOutOfFacebook()));
    connect(m_session->sharing(), SIGNAL(userSignedInToTwitterChanged()), this, SLOT(onSignedInOrOutOfTwitter()));

    onOrientationChanged();
}

void AccountsDialog::showYouTubeAuthDialog() {
    YouTubeAuthDialog *dialog = new YouTubeAuthDialog(m_session, m_youtubeEdit->text(), this);
    dialog->open();

    m_youtubeEdit->clear();
}

void AccountsDialog::showDailymotionAuthDialog() {
    DailymotionAuthDialog *dialog = new DailymotionAuthDialog(m_session, m_dailymotionEdit->text(), this);
    dialog->open();

    m_dailymotionEdit->clear();
}

void AccountsDialog::showFacebookAuthDialog() {
    FacebookAuthDialog *dialog = new FacebookAuthDialog(m_session, this);
    dialog->open();
}

void AccountsDialog::showTwitterAuthDialog() {
    TwitterAuthDialog *dialog = new TwitterAuthDialog(m_session, this);
    dialog->open();
}

void AccountsDialog::onFacebookButtonClicked() {
    if (m_session->sharing()->userSignedInToFacebook()) {
        m_session->sharing()->signOutOfFacebook();
    }
    else {
        showFacebookAuthDialog();
    }
}

void AccountsDialog::onTwitterButtonClicked() {
    if (m_session->sharing()->userSignedInToTwitter()) {
        m_session->sharing()->signOutOfTwitter();
    }
    else {
        showTwitterAuthDialog();
    }
}

void AccountsDialog::onSignedInOrOutOfYouTube() {
    m_youtubeLabel->setText(m_session->youtube()->userSignedIn() ? tr("You are signed in to a YouTube account") : tr("Sign in to a YouTube account"));
}

void AccountsDialog::onSignedInOrOutOfDailymotion() {
    m_dailymotionLabel->setText(m_session->dailymotion()->userSignedIn() ? tr("You are signed in to a Dailymotion account") : tr("Sign in to a Dailymotion account"));
}

void AccountsDialog::onSignedInOrOutOfFacebook() {
    bool signedIn = m_session->sharing()->userSignedInToFacebook();
    m_facebookButton->setText(signedIn ? tr("Sign out") : tr("Sign in"));
    m_facebookLabel->setText(signedIn ? tr("You are signed in to a facebook account") : tr("Sign in to a facebook account"));
}

void AccountsDialog::onSignedInOrOutOfTwitter() {
    bool signedIn = m_session->sharing()->userSignedInToTwitter();
    m_twitterButton->setText(signedIn ? tr("Sign out") : tr("Sign in"));
    m_twitterLabel->setText(signedIn ? tr("You are signed in to a twitter account") : tr("Sign in to a twitter account"));
}

void AccountsDialog::switchYouTubeAccount(const QModelIndex &index) {
    if (index.data(YouTubeAccountsListModel::UsernameRole).toString() != m_session->youtube()->username()) {
        m_youtubeModel->switchAccount(index.row());
    }
}

void AccountsDialog::switchDailymotionAccount(const QModelIndex &index) {
    if (index.data(DailymotionAccountsListModel::UsernameRole).toString() != m_session->dailymotion()->username()) {
        m_dailymotionModel->switchAccount(index.row());
    }
}

void AccountsDialog::showYouTubeContextMenu(const QPoint &pos) {
    m_youtubeDeleteAction->setVisible(true);
    m_dailymotionDeleteAction->setVisible(false);
    m_contextMenu->popup(pos, m_youtubeDeleteAction);
}

void AccountsDialog::showDailymotionContextMenu(const QPoint &pos) {
    m_dailymotionDeleteAction->setVisible(true);
    m_youtubeDeleteAction->setVisible(false);
    m_contextMenu->popup(pos, m_dailymotionDeleteAction);
}

void AccountsDialog::deleteYouTubeAccount() {
    m_youtubeModel->deleteAccount(m_youtubeView->currentIndex().row());
}

void AccountsDialog::deleteDailymotionAccount() {
    m_dailymotionModel->deleteAccount(m_dailymotionView->currentIndex().row());
}
