#include "youtubevideolistwindow.h"
#include "youtubevideoinfowindow.h"
#include "videoselectionwindow.h"
#include "videolistdelegate.h"
#include <QMenuBar>

YouTubeVideoListWindow::YouTubeVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_deleteFromUploadsAction(menuBar()->addAction(tr("Delete from uploads"), this, SLOT(deleteVideosFromUploads()))),
    m_addToFavouritesAction(menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_deleteFromFavouritesAction(menuBar()->addAction(tr("Delete from favourites"), this, SLOT(deleteVideosFromFavourites()))),
    m_addToPlaylistAction(menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_watchLaterAction(menuBar()->addAction(tr("Watch later"), this, SLOT(addVideosToWatchLaterPlaylist()))),
    m_addToQueueAction(menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));

    onOrientationChanged();
}

void YouTubeVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeVideoListWindow::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    setMenuActions();
}

void YouTubeVideoListWindow::getRelatedVideos(const QString &id) {
    m_model->getRelatedVideos(id);
    setMenuActions();
}

void YouTubeVideoListWindow::search(const QString &query, Queries::QueryOrder order, Queries::TimeFilter time, Queries::DurationFilter duration, const QString &language) {
    setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order, time, duration, language);
    setMenuActions();
}

void YouTubeVideoListWindow::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->uploads()));
    m_addToFavouritesAction->setEnabled((signedIn) && (!m_model->favourites()));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->favourites()));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled((signedIn) && (!m_model->watchLater()));
}

void YouTubeVideoListWindow::onLoadingChanged() {
    toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void YouTubeVideoListWindow::deleteVideosFromUploads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), this);
    window->show();
}

void YouTubeVideoListWindow::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::deleteVideosFromFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideosToWatchLaterPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, this);
    window->show();
}

void YouTubeVideoListWindow::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
}
