#include "youtube.h"
#include "json.h"
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QRegExp>
#include <QUrl>
#include <QMap>
#include <QStringList>

using namespace QtJson;

YouTube::YouTube(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_clientId("446197852855.apps.googleusercontent.com"),
    m_clientSecret("YzuBGnTyRdz9cK5J3Pg5Xlte"),
    m_key("AI39si6x9O1gQ1Z_BJqo9j2n_SdVsHu1pk2uqvoI3tVq8d6alyc1og785IPCkbVY3Q5MFuyt-IFYerMYun0MnLdQX5mo2BueSw"),
    m_actionsProcessed(0),
    m_playlistCache(new QList< QSharedPointer<PlaylistItem> >),
    m_subscriptionCache(new QList< QSharedPointer<UserItem> >),
    m_playlistCacheLoaded(false),
    m_subscriptionCacheLoaded(false),
    m_busy(false)
{
    m_queryOrders[Queries::Relevance] = "relevance";
    m_queryOrders[Queries::Date] = "published";
    m_queryOrders[Queries::Views] = "viewCount";
    m_queryOrders[Queries::Rating] = "rating";

    m_timeFilters[Queries::AllTime] = "all_time";
    m_timeFilters[Queries::ThisWeek] = "this_week";
    m_timeFilters[Queries::ThisMonth] = "this_month";

    m_durationFilters[Queries::Any] = "";
    m_durationFilters[Queries::Short] = "short";
    m_durationFilters[Queries::Medium] = "medium";
    m_durationFilters[Queries::Long] = "long";
}

YouTube::~YouTube() {
    clearCache();
}

QNetworkReply* YouTube::createReply(QString feed, int offset) {
    QUrl url(feed);

    if (userSignedIn()) {
        url.addQueryItem("access_token", accessToken());
    }

    if (offset) {
        url.addQueryItem("start-index", QString::number(offset));
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey().toUtf8());

    return networkAccessManager()->get(request);
}

QNetworkReply* YouTube::createSearchReply(Queries::QueryType queryType, const QString &query, int offset, Queries::QueryOrder order, Queries::TimeFilter time, Queries::DurationFilter duration, const QString &language) {
    QString qs = query.simplified().replace(' ', '+');

    QUrl url;

    if (queryType == Queries::Videos) {
        url.setUrl(YOUTUBE_VIDEOS_BASE_URL);
        url.addQueryItem("fields", YOUTUBE_VIDEO_FIELDS);
        url.addQueryItem("orderby", m_queryOrders[order]);
        url.addQueryItem("time", m_timeFilters[time]);

        if (duration != Queries::Any) {
            url.addQueryItem("duration", m_durationFilters[duration]);
        }
        if (!language.isEmpty()) {
            url.addQueryItem("lr", language);
        }
    }
    else if (queryType == Queries::Playlists) {
        url.setUrl(YOUTUBE_PLAYLISTS_SEARCH_BASE_URL);
        url.addQueryItem("fields", YOUTUBE_PLAYLIST_FIELDS);
    }
    else if (queryType == Queries::Users) {
        url.setUrl(YOUTUBE_CHANNELS_BASE_URL);
        url.addQueryItem("fields", YOUTUBE_USER_FIELDS);
    }

    url.addQueryItem("q", "\"" + qs + "\"|" + qs);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("safeSearch", safeSearch() ? "strict" : "none");
    url.addQueryItem("max-results", "30");

    if (offset) {
        url.addQueryItem("start-index", QString::number(offset));
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");

    return networkAccessManager()->get(request);
}

QUrl YouTube::authUrl() const {
    QUrl url("https://accounts.google.com/o/oauth2/auth");
    url.addQueryItem("client_id", m_clientId);
    url.addQueryItem("redirect_uri", "urn:ietf:wg:oauth:2.0:oob");
    url.addQueryItem("response_type", "code");
    url.addQueryItem("scope", "https://gdata.youtube.com");
    url.addQueryItem("access_type", "offline");
    url.addQueryItem("display", "popup");

    return url;
}

void YouTube::signIn(const QString &displayName, const QString &code) {
    emit busy(tr("Signing in"));
    m_user = displayName;
    QUrl url("https://accounts.google.com/o/oauth2/token");

    QByteArray body("code=" + code.toUtf8() +
                    "&client_id=" + m_clientId.toUtf8() +
                    "&client_secret=" + m_clientSecret.toUtf8() +
                    "&redirect_uri=urn:ietf:wg:oauth:2.0:oob&grant_type=authorization_code");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = networkAccessManager()->post(request, body);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
}

void YouTube::checkIfSignedIn() {
    emit busyProgressChanged(1);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit warning(tr("Cannot parse server response"));
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if ((token.isEmpty()) || (refresh.isEmpty())) {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit warning(statusText);
        }
        else {
            emit alert(tr("You are signed in to your YouTube account"));
            emit signedIn(username(), token, refresh);
        }
    }
}

void YouTube::refreshAccessToken() {
    QUrl url("https://accounts.google.com/o/oauth2/token");

    QByteArray body("client_id=" + m_clientId.toUtf8() +
                    "&client_secret=" + m_clientSecret.toUtf8() +
                    "&refresh_token=" + refreshToken().toUtf8() +
                    "&grant_type=refresh_token");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = networkAccessManager()->post(request, body);
    connect(reply, SIGNAL(finished()), this, SLOT(checkTokenRefresh()));
}

void YouTube::checkTokenRefresh() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit postFailed(tr("Cannot parse server response"));
        emit refreshError();
    }
    else {
        QString token = result.value("access_token").toString();

        if (token.isEmpty()) {
            QString error = result.value("error").toString();

            if (error == "invalid_request") {
                emit postFailed(tr("Error refreshing access token. Token is no longer valid"));
                emit refreshError();
                setAccount();
            }
            else {
                QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                emit postFailed(statusText);
                emit refreshError();
            }

            disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, 0);
        }
        else {
            setAccessToken(token);
            emit accessTokenRefreshed(token);
        }
    }

    reply->deleteLater();
}

void YouTube::linkGoogleAccount(QString username) {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default?v=2.1");
    QByteArray xml("<entry xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<yt:username>" + username.toUtf8() + "</yt:username>\n" \
                   "</entry>");

    putRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onGoogleAccountLinked(QString)));
}

void YouTube::onGoogleAccountLinked(const QString &response) {
    m_linkUsername = "";

    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");
    QString username = entry.firstChildElement("yt:username").attribute("display");
    QString userId = entry.firstChildElement("yt:userId").text();
    setUsername(userId);

    emit googleAccountLinked();
    emit info(tr("Your Google account is now linked to your YouTube account '%1'").arg(username));
}

void YouTube::checkUsernameAvailability(QString username) {
    m_linkUsername = username;
    QUrl url("https://gdata.youtube.com/feeds/api/suggest/username");
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", "entry/title");
    url.addQueryItem("hint", m_linkUsername);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey().toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkSuggestedUsernames()));
}

void YouTube::checkSuggestedUsernames() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNodeList entries = doc.elementsByTagName("entry");
    QStringList usernames;

    for (int i = 0; i < entries.size(); i++) {
        usernames.append(entries.at(i).firstChildElement("title").text());
    }

    if (usernames.isEmpty()) {
        emit usernameUnavailable();
    }
    else if (usernames.contains(m_linkUsername)) {
        emit usernameAvailable();
        linkGoogleAccount(m_linkUsername);
    }
    else {
        emit gotSuggestedUsernames(usernames);
    }

    reply->deleteLater();
}

void YouTube::cancelLinkGoogleAccount() {
    disconnect(this, SIGNAL(googleAccountLinked()), this, 0);
}

void YouTube::setAccount(const QString &user, const QString &token, const QString &refresh) {
    if (user != username()) {
        setUsername(user);
    }

    setAccessToken(token);
    setRefreshToken(refresh);
    clearCache();

    if ((user.isEmpty()) && (!token.isEmpty())) {
        emit newAccountSet();
    }
}

void YouTube::clearCache() {
    m_playlistCache->clear();
    m_subscriptionCache->clear();

    setPlaylistsLoaded(false);
    setSubscriptionsLoaded(false);
}

void YouTube::postRequest(const QUrl &url, const QByteArray &xml) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2.1");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey().toUtf8());
    QNetworkReply* reply = networkAccessManager()->post(request, xml);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void YouTube::putRequest(const QUrl &url, const QByteArray &xml) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2.1");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey().toUtf8());
    QNetworkReply* reply = networkAccessManager()->put(request, xml);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void YouTube::deleteRequest(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setRawHeader("Authorization", "Bearer " + accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2.1");
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey().toUtf8());
    QNetworkReply* reply = networkAccessManager()->deleteResource(request);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void YouTube::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (userSignedIn())) {
        refreshAccessToken();
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            QString response(reply->readAll());
            emit postSuccessful(response);
        }
        else {
            QDomDocument doc;
            doc.setContent(reply->readAll());
            QDomNode errorNode = doc.namedItem("errors").namedItem("error");

            if (!errorNode.isNull()) {
                QString errorString = errorNode.firstChildElement("internalReason").text();
                QString errorCode = errorNode.firstChildElement("code").text();

                if ((errorString.contains(QRegExp("(already in favorite|already in playlist)"))) && (m_videoActionList.size() > 1)) {
                    emit postSuccessful(QString());
                }
                else if (errorCode == "youtube_signup_required") {
                    emit requestToLinkGoogleAccount();
                }
                else if (!errorString.isEmpty()) {
                    emit postFailed(errorString);
                }
            }
            else {
                QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                emit postFailed(statusText);
            }
        }
    }

    reply->deleteLater();
}

void YouTube::getPlaylists(int offset) {
    setPlaylistsLoaded(false);
    QUrl url(YOUTUBE_PLAYLISTS_FEED);
    url.addQueryItem("start-index", QString::number(offset));
    url.addQueryItem("fields", YOUTUBE_PLAYLIST_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void YouTube::addPlaylists() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getPlaylists()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.count(); i++) {
            m_playlistCache->append(QSharedPointer<PlaylistItem>(new PlaylistItem(entries.at(i))));
        }

        setPlaylistsLoaded(true);

        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();

        if ((totalResults > m_playlistCache->size()) && (!entries.isEmpty())) {
            getPlaylists(m_playlistCache->size() + 1);
        }
        else {
            emit allPlaylistsLoaded();
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getPlaylists()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void YouTube::addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist) {
    m_playlistCache->insert(0, playlist);
    emit playlistAddedToCache(0);
}

bool YouTube::removePlaylistFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    while ((!removed) && (i < m_playlistCache->size())) {
        removed = (m_playlistCache->at(i).data()->playlistId() == id);
        i++;
    }

    if (removed) {
        m_playlistCache->takeAt(i - 1).clear();
        emit playlistRemovedFromCache(i - 1);
    }

    return removed;
}

void YouTube::updatePlaylistVideoCount(const QString &id, int change) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_playlistCache->size())) {
        updated = (m_playlistCache->at(i)->playlistId() == id);
        i++;
    }

    if (updated) {
        QSharedPointer<PlaylistItem> playlist =  m_playlistCache->at(i - 1);
        playlist.data()->setVideoCount(playlist.data()->videoCount() + change);
        emit playlistUpdated(i - 1);
    }
}

void YouTube::updatePlaylistThumbnail(const QString &id, const QString &thumbnailUrl) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_playlistCache->size())) {
        updated = (m_playlistCache->at(i).data()->playlistId() == id);
        i++;
    }

    if (updated) {
        m_playlistCache->at(i - 1).data()->setThumbnailUrl(thumbnailUrl);
        emit playlistUpdated(i - 1);
    }
}

void YouTube::getSubscriptions(int offset) {
    setSubscriptionsLoaded(false);
    QUrl url(YOUTUBE_SUBSCRIPTIONS_FEED);
    url.addQueryItem("start-index", QString::number(offset));
    url.addQueryItem("fields", YOUTUBE_SUBSCRIPTION_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(addSubscriptions()));
}

void YouTube::addSubscriptions() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getSubscriptions()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.count(); i++) {
            m_subscriptionCache->append(QSharedPointer<UserItem>(new UserItem(entries.at(i), true, true)));
        }

        setSubscriptionsLoaded(true);

        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();

        if ((totalResults > m_subscriptionCache->size()) && (!entries.isEmpty())) {
            getSubscriptions(m_subscriptionCache->size() + 1);
        }
        else {
            emit allSubscriptionsLoaded();
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getSubscriptions()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void YouTube::addNewSubscriptionToCache(QSharedPointer<UserItem> user) {
    int i = 0;
    bool found = false;

    while ((!found) && (i < m_subscriptionCache->size())) {
        found = (m_subscriptionCache->at(i).data()->username() > user.data()->username());
        i++;
    }

    m_subscriptionCache->insert(i - 1, user);
    emit subscriptionAddedToCache(i - 1);
}

bool YouTube::removeSubscriptionFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    while ((!removed) && (i < m_subscriptionCache->size())) {
        removed = (m_subscriptionCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        m_subscriptionCache->takeAt(i - 1).clear();
        emit subscriptionRemovedFromCache(i - 1);
    }

    return removed;
}

void YouTube::subscribedToChannel(QSharedPointer<UserItem> user) {
    if (!user.isNull()) {
        m_userAction = user;
    }

    int i = 0;
    bool subscribed = false;

    while ((i < m_subscriptionCache->size()) && (!subscribed)) {
        subscribed = (m_subscriptionCache->at(i).data()->id() == m_userAction.data()->id());
        i++;
    }

    if (subscribed) {
        m_userAction.data()->setSubscribed(true);
        m_userAction.data()->setSubscriptionId(m_subscriptionCache->at(i - 1).data()->subscriptionId());
    }

    disconnect(this, 0, this, SLOT(subscribedToChannel()));
}

void YouTube::getFullVideo(QString id) {
    emit busy(tr("Retrieving video details"));

    if (id.size() > 11) {
        id = id.section(QRegExp("(v=|/)"), -1).left(11);
    }

    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/") + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_VIDEO_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideo()));
}

void YouTube::checkFullVideo() {
    emit busyProgressChanged(1);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        emit gotVideo(QSharedPointer<VideoItem>(new VideoItem(entry)));
    }
    else {
        emit warning(tr("Video could not be retrieved"));
        disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
    }

    reply->deleteLater();
}

void YouTube::getVideoMetadata(const QString &id) {
    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/") + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkVideoMetadata()));
}

void YouTube::checkVideoMetadata() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideoMetadata()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        QDomElement mediaElement = entry.firstChildElement("media:group");
        VideoMetadata metadata;
        metadata.setTitle(mediaElement.firstChildElement("media:title").text());
        metadata.setDescription(mediaElement.firstChildElement("media:description").text());
        metadata.setTags(mediaElement.firstChildElement("media:keywords").text());
        metadata.setCategory(mediaElement.firstChildElement("media:category").text());

        QDomElement permissionsElement = entry.firstChildElement("yt:accessControl");
        metadata.setCommentsPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setCommentVotingPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setResponsesPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setRatingsPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setEmbedPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setListingsPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setAutoPlayPermission(permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.setSyndicationPermission(permissionsElement.attribute("permission"));

        metadata.setPrivate(!mediaElement.firstChildElement("yt:private").isNull());

        emit gotVideoMetadata(metadata);

        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideoMetadata()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void YouTube::getFullComment(const QString &videoId, const QString &commentId) {
    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/") + videoId + QString("/comments/") + commentId);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_COMMENT_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFullComment()));
}

void YouTube::checkFullComment() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        emit gotComment(QSharedPointer<CommentItem>(new CommentItem(entry)));
    }
    else {
        emit warning(tr("Comment could not be retrieved"));
    }

    reply->deleteLater();
}

void YouTube::getCurrentUserProfile() {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default");
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_USER_PROFILE_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkCurrentUserProfile()));
}

void YouTube::checkCurrentUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getCurrentUserProfile()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        QSharedPointer<UserItem> user(new UserItem);
        user.data()->setId(entry.firstChildElement("yt:userId").text());
        user.data()->setUsername(entry.firstChildElement("yt:username").attribute("display"));
        user.data()->setChannelName(entry.firstChildElement("title").text());
        user.data()->setAge(entry.firstChildElement("yt:age").text().toInt());
        user.data()->setGender(entry.firstChildElement("yt:gender").text() == "m" ? tr("Male") : tr("Female"));
        user.data()->setLocation(entry.firstChildElement("yt:location").text());
        user.data()->setDescription(entry.firstChildElement("summary").text());
        user.data()->setAvatarUrl(entry.firstChildElement("media:thumbnail").attribute("url"));
        user.data()->setWebsiteUrl(entry.firstChildElement("link").attribute("href"));
        user.data()->setVideoCount(entry.firstChildElement("gd:feedLink").attribute("countHint").toInt());
        user.data()->setViewCount(entry.firstChildElement("yt:statistics").attribute("viewCount").toInt());
        user.data()->setSubscriberCount(entry.firstChildElement("yt:statistics").attribute("subscriberCount").toInt());
        user.data()->setSubscribed(false);
        user.data()->setAccountLinked(entry.firstChildElement("yt:incomplete").isNull());

        emit gotUser(user);

        if (username() != user.data()->id()) {
            setUsername(user.data()->id());
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getCurrentUserProfile()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void YouTube::getUserProfile(const QString &id) {
    QUrl url(YOUTUBE_USERS_BASE_URL + QString("/") + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_USER_PROFILE_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkUserProfile()));
}

void YouTube::checkUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        QSharedPointer<UserItem> user(new UserItem);
        user.data()->setId(entry.firstChildElement("yt:userId").text());
        user.data()->setUsername(entry.firstChildElement("yt:username").attribute("display"));
        user.data()->setChannelName(entry.firstChildElement("title").text());
        user.data()->setAge(entry.firstChildElement("yt:age").text().toInt());
        user.data()->setGender(entry.firstChildElement("yt:gender").text() == "m" ? "Male" : "Female");
        user.data()->setLocation(entry.firstChildElement("yt:location").text());
        user.data()->setDescription(entry.firstChildElement("summary").text());
        user.data()->setAvatarUrl(entry.firstChildElement("media:thumbnail").attribute("url"));
        user.data()->setWebsiteUrl(entry.firstChildElement("link").attribute("href"));
        user.data()->setVideoCount(entry.firstChildElement("gd:feedLink").attribute("countHint").toInt());
        user.data()->setViewCount(entry.firstChildElement("yt:statistics").attribute("viewCount").toInt());
        user.data()->setSubscriberCount(entry.firstChildElement("yt:statistics").attribute("subscriberCount").toInt());

        if (subscriptionsLoaded()) {
            subscribedToChannel(user);
        }
        else if (userSignedIn()) {
            m_userAction = user;
            getSubscriptions();
            connect(this, SIGNAL(allSubscriptionsLoaded()), this, SLOT(subscribedToChannel()));
        }

        emit gotUser(user);
    }
    else {
        emit warning(tr("Profile could not be retrieved"));
    }

    reply->deleteLater();
}

void YouTube::updateVideoMetadata(QSharedPointer<VideoItem> video, const VideoMetadata &metadata) {
    if (!video.isNull()) {
        m_videoAction = video;
    }
    if (!metadata.isEmpty()) {
        m_metadataAction = metadata;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/uploads/" + m_videoAction.data()->videoId() + "?v=2.1");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + m_metadataAction.title().toUtf8() + "</media:title>\n" \
                   "<media:description>\n" + m_metadataAction.description().toUtf8() + "\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">" + m_metadataAction.category().toUtf8() + "</media:category>\n" \
                   "<media:keywords>" + m_metadataAction.tags().toUtf8() + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "<yt:accessControl action=\"comment\" permission=\"" + m_metadataAction.commentsPermission().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"commentVote\" permission=\"" + m_metadataAction.commentVotingPermission().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"rate\" permission=\"" + m_metadataAction.ratingsPermission().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"syndicate\" permission=\"" + m_metadataAction.syndicationPermission().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"list\" permission=\"" + m_metadataAction.listingsPermission().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"embed\" permission=\"" + m_metadataAction.embedPermission().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"videoRespond\" permission=\"" + m_metadataAction.responsesPermission().toUtf8() + "\"/>\n" \
                   "</entry>");

    if (m_metadataAction.isPrivate()) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }

    putRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoMetadataUpdated()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(updateVideoMetadata()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onVideoMetadataUpdated() {
    if (!m_videoAction.isNull()) {
        m_videoAction.data()->setTitle(m_metadataAction.title());
        m_videoAction.data()->setDescription(m_metadataAction.description());
        m_videoAction.data()->setTags(m_metadataAction.tags().split(QRegExp("(, |,)"), QString::SkipEmptyParts));

        emit alert(tr("Video metadata updated"));
    }
    else {
        emit warning(tr("Cannot update video metadata"));
    }

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoMetadataUpdated()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(updateVideoMetadata()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::deleteFromUploads(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from uploads"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromUploads(m_videoActionList.first(), false);
    }
}

void YouTube::deleteFromUploads(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/uploads/" + video.data()->videoId());
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDeleted()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromUploads()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onVideoDeleted() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        emit deletedFromUploads(m_videoActionList.takeFirst());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromUploads();
    }
    else {
        emit alert(tr("Video(s) deleted from uploads"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDeleted()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromUploads()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::addToFavourites(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Adding video(s) to favourites"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        addToFavourites(m_videoActionList.first(), false);
    }
}

void YouTube::addToFavourites(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/favorites");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\">\n" \
                   "<id>" + video.data()->videoId().toUtf8() + "</id>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToFavourites(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToFavourites()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onAddedToFavourites(const QString &response) {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        if (!response.isEmpty()) {
            QSharedPointer<VideoItem> video = m_videoActionList.takeFirst();
            QDomDocument doc;
            doc.setContent(response);
            QDomElement entry = doc.firstChildElement("entry");
            video.data()->setFavourite(true);
            video.data()->setId(entry.firstChildElement("id").text());
            emit addedToFavourites(video);
        }
        else {
            m_videoActionList.removeFirst();
        }
    }
    if (!m_videoActionList.isEmpty()) {
        addToFavourites();
    }
    else {
        emit alert(tr("Video(s) added to favourites"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToFavourites(QString)));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToFavourites()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::deleteFromFavourites(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from favourites"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromFavourites(m_videoActionList.first(), false);
    }
}

void YouTube::deleteFromFavourites(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/favorites/" + video.data()->id().section(':', -1));
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromFavourites()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromFavourites()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onDeletedFromFavourites() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        QSharedPointer<VideoItem> video = m_videoActionList.takeFirst();
        video.data()->setFavourite(false);
        emit deletedFromFavourites(video);
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromFavourites();
    }
    else {
        emit alert(tr("Video(s) deleted from favourites"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromFavourites()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromFavourites()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::addToPlaylist(QList< QSharedPointer<VideoItem> > videos, const QString &playlistId) {
    if ((m_actionsProcessed == 0) || (!videos.isEmpty())) {

        if (!videos.isEmpty()) {
            m_actionsProcessed = 0;
            m_videoActionList = videos;
        }

        emit busy(tr("Adding video(s) to playlist"), m_videoActionList.size());
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }
    if (!m_videoActionList.isEmpty()) {
        addToPlaylist(m_videoActionList.first(), "", false);
    }
}

void YouTube::addToPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/playlists/" + m_playlistActionId);
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<id>" + video.data()->videoId().toUtf8() + "</id>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToPlaylist()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
}

void YouTube::onAddedToPlaylist() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (playlistsLoaded()) {
        updatePlaylistVideoCount(m_playlistActionId, 1);
    }
    if (!m_videoActionList.isEmpty()) {
        emit addedToPlaylist(m_playlistActionId, m_videoActionList.takeFirst());
    }
    if (!m_videoActionList.isEmpty()) {
        addToPlaylist();
    }
    else {
        emit alert(tr("Video(s) added to playlist"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToPlaylist()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToPlaylist()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::deleteFromPlaylist(QList< QSharedPointer<VideoItem> > videos, const QString &playlistId) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from playlist"), m_videoActionList.size());
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromPlaylist(m_videoActionList.first(), "", false);
    }
}

void YouTube::deleteFromPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/playlists/" + m_playlistActionId + "/" + video.data()->id().section(':', -1));
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromPlaylist()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onDeletedFromPlaylist() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (playlistsLoaded()) {
        updatePlaylistVideoCount(m_playlistActionId, -1);
    }
    if (!m_videoActionList.isEmpty()) {
        emit deletedFromPlaylist(m_playlistActionId, m_videoActionList.takeFirst());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromPlaylist();
    }
    else {
        emit alert(tr("Video(s) deleted from playlist"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromPlaylist()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromPlaylist()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::addToWatchLaterPlaylist(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Adding video(s) to 'Watch Later' playlist"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        addToWatchLaterPlaylist(m_videoActionList.first(), false);
    }
}

void YouTube::addToWatchLaterPlaylist(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/watch_later");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<id>" + video.data()->videoId().toUtf8() + "</id>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToWatchLaterPlaylist(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToWatchLaterPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onAddedToWatchLaterPlaylist(const QString &response) {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        if (!response.isEmpty()) {
            QSharedPointer<VideoItem> video = m_videoActionList.takeFirst();
            QDomDocument doc;
            doc.setContent(response);
            QDomElement entry = doc.firstChildElement("entry");
            video.data()->setId(entry.firstChildElement("id").text());
            emit addedToWatchLaterPlaylist(video);
        }
        else {
            m_videoActionList.removeFirst();
        }
    }
    if (!m_videoActionList.isEmpty()) {
        addToWatchLaterPlaylist();
    }
    else {
        emit alert(tr("Video(s) added to 'Watch Later' playlist"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToWatchLaterPlaylist(QString)));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToWatchLaterPlaylist()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::deleteFromWatchLaterPlaylist(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from 'Watch Later' playlist"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromWatchLaterPlaylist(m_videoActionList.first(), false);
    }
}

void YouTube::deleteFromWatchLaterPlaylist(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://gdata.youtube.com/feeds/api/playlists/watch_later/" + video.data()->id().section(':', -1));
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromWatchLaterPlaylist()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromWatchLaterPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onDeletedFromWatchLaterPlaylist() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        emit deletedFromWatchLaterPlaylist(m_videoActionList.takeFirst());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromWatchLaterPlaylist();
    }
    else {
        emit alert(tr("Video(s) deleted from 'Watch Later' playlist"));

        disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromWatchLaterPlaylist()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromWatchLaterPlaylist()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::createPlaylist(const NewPlaylist &playlist, QSharedPointer<VideoItem> video, QList< QSharedPointer<VideoItem> > videos) {
    if (!playlist.isEmpty()) {
        m_playlistAction = playlist;
    }
    if (!video.isNull()) {
        m_videoActionList.append(video);
    }
    else if (!videos.isEmpty()) {
        m_videoActionList = videos;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<title>" + m_playlistAction.title().toUtf8() + "</title>\n" \
                   "<summary>" + m_playlistAction.description().toUtf8() + "</summary>\n" \
                   "</entry>");

    if (m_playlistAction.isPrivate()) {
        int index = xml.lastIndexOf("<");
        xml.insert(index, "<yt:private/>\n");
    }

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistCreated(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(createPlaylist()));
    connect(this, SIGNAL(googleAccountLinked()), this, SLOT(createPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::onPlaylistCreated(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");
    m_playlistActionId = entry.firstChildElement("yt:playlistId").text();

    if (!m_videoActionList.isEmpty()) {
        m_actionsProcessed = 0;

        if (playlistsLoaded()) {
            getPlaylistForCache(m_playlistActionId);
            connect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
        }
        else {
            addToPlaylist();
        }
    }
    else {
        if (playlistsLoaded()) {
            getPlaylistForCache(m_playlistActionId);
        }

        emit alert(tr("Playlist created"));
    }

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistCreated(QString)));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(createPlaylist()));
    disconnect(this, SIGNAL(googleAccountLinked()), this, SLOT(createPlaylist()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::getPlaylistForCache(const QString &id) {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists/" + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_PLAYLIST_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("X-Gdata-Key", "key=" + developerKey().toUtf8());
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkCachePlaylist()));
}

void YouTube::checkCachePlaylist() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        addNewPlaylistToCache(QSharedPointer<PlaylistItem>(new PlaylistItem(entry)));
    }
    else {
        emit warning(tr("Unable to retrieve playlist details"));
        disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
    }

    reply->deleteLater();
}

void YouTube::deletePlaylist(const QString &id) {
    if (!id.isEmpty()) {
        m_playlistActionId = id;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists/" + m_playlistActionId);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistDeleted()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deletePlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::onPlaylistDeleted() {
    emit alert(tr("Playlist deleted"));

    if (playlistsLoaded()) {
        removePlaylistFromCache(m_playlistActionId);
    }

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistDeleted()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deletePlaylist()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::subscribe(QSharedPointer<UserItem> user) {
    if (!user.isNull()) {
        m_userAction = user;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/subscriptions");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<category scheme=\"http://gdata.youtube.com/schemas/2007/subscriptiontypes.cat\"\n" \
                   "term=\"channel\"/>\n" \
                   "<yt:channelId>UC" + m_userAction.data()->id().toUtf8() + "</yt:channelId>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onSubscribed(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(subscribe()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::onSubscribed(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.namedItem("entry");
    m_userAction.data()->setSubscribed(true);
    m_userAction.data()->setSubscriptionId(entry.firstChildElement("id").text().section(':', -1));

    if (subscriptionsLoaded()) {
        addNewSubscriptionToCache(QSharedPointer<UserItem>(m_userAction));
    }

    emit alert(tr("You have subscribed to '%1'").arg(m_userAction.data()->username()));

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onSubscribed(QString)));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(subscribe()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::unsubscribe(QSharedPointer<UserItem> user) {
    if (!user.isNull()) {
        m_userAction = user;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/users/default/subscriptions/" + m_userAction.data()->subscriptionId());
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUnsubscribed()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(unsubscribe()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::onUnsubscribed() {
    m_userAction.data()->setSubscribed(false);

    emit alert(tr("You have unsubscribed to '%1'").arg(m_userAction.data()->username()));

    if (subscriptionsLoaded()) {
        removeSubscriptionFromCache(m_userAction.data()->id());
    }

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUnsubscribed()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(unsubscribe()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::rateVideo(QSharedPointer<VideoItem> video, const QString &likeOrDislike) {
    if (!video.isNull()) {
        m_videoAction = video;
    }
    if (!likeOrDislike.isEmpty()) {
        m_ratingAction = likeOrDislike;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/videos/" + m_videoAction.data()->videoId() + "/ratings");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<yt:rating value=\"" + m_ratingAction.toUtf8() + "\"/>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoRated()));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(rateVideo()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onVideoRated() {
    if (m_ratingAction == "like") {
        m_videoAction.data()->like();
        emit alert(tr("You liked this video"));
    }
    else {
        m_videoAction.data()->dislike();
        emit alert(tr("You disliked this video"));
    }

    emit videoRated(m_videoAction);

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoRated()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(rateVideo()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::addComment(const NewComment &comment) {
    if (!comment.isEmpty()) {
        m_commentAction = comment;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/videos/" + m_commentAction.videoId() + "/comments");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<content>" + m_commentAction.body().toUtf8() + "</content>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addComment()));
    connect(this, SIGNAL(googleAccountLinked()), this, SLOT(addComment()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::replyToComment(const NewComment &comment) {
    if (!comment.isEmpty()) {
        m_commentAction = comment;
    }

    QUrl url("https://gdata.youtube.com/feeds/api/videos/" + m_commentAction.videoId() + "/comments");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<link rel=\"http://gdata.youtube.com/schemas/2007#in-reply-to\"\n" \
                   "type=\"application/atom+xml\"\n" \
                   "href=\"https://gdata.youtube.com/feeds/api/videos/" + m_commentAction.videoId().toUtf8() + "/comments/" + m_commentAction.replyId().toUtf8() + "\"/>\n" \
                   "<content>" + m_commentAction.body().toUtf8() + "</content>\n" \
                   "</entry>");

    postRequest(url, xml);
    connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(replyToComment()));
    connect(this, SIGNAL(googleAccountLinked()), this, SLOT(addComment()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::onCommentAdded(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");
    QString videoId = entry.firstChildElement("yt:videoid").text();
    QString commentId = entry.firstChildElement("id").text().section(':', -1);
    getAddedComment(videoId, commentId);

    disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addComment()));
    disconnect(this, SIGNAL(googleAccountLinked()), this, SLOT(addComment()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::getAddedComment(const QString &videoId, const QString &commentId) {
    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/") + videoId + QString("/comments/") + commentId);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_COMMENT_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkAddedComment()));
}

void YouTube::checkAddedComment() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        emit commentAdded(QSharedPointer<CommentItem>(new CommentItem(entry)));
        emit alert(tr("Your comment has been added"));
    }
    else {
        emit warning(tr("New comment could not be retrieved"));
    }

    reply->deleteLater();
}

void YouTube::onVideoActionError(const QString &error) {
    m_videoActionList.clear();
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onPlaylistActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::onUserActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::onCommentActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::getVideosFromIds(QStringList ids) {
    setBusy(true);
    QUrl url("https://gdata.youtube.com/feeds/api/videos/batch");
    url.addQueryItem("v", "2.1");

    QByteArray xml("<feed xmlns=\"http://www.w3.org/2005/Atom\" xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:batch=\"http://schemas.google.com/gdata/batch\" xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n"
                   "<batch:operation type=\"query\" />\n");

    int i = 0;

    while ((!ids.isEmpty()) && (i < 50)) {
        xml.append("<entry>\n" \
                   "<id>https://gdata.youtube.com/feeds/api/videos/" + ids.takeFirst().toUtf8() + "</id>\n" \
                   "</entry>\n");
        i++;
    }

    xml.append("</feed>");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.0 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    QNetworkReply *reply = networkAccessManager()->post(request, xml);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideos()));
}

void YouTube::checkFullVideos() {
    setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNodeList entries = doc.elementsByTagName("entry");

    if (!entries.isEmpty()) {
        QList< QSharedPointer<VideoItem> > videos;

        for (int i = 0; i < entries.count(); i++) {
            videos.append(QSharedPointer<VideoItem>(new VideoItem(entries.at(i))));
        }

        emit gotVideosFromIds(videos);
    }

    reply->deleteLater();
}
