#include "useritem.h"
#include "youtube.h"
#include "dailymotion.h"
#include "vimeo.h"

UserItem::UserItem(QObject *parent) :
    QObject(parent),
    m_videoCount(0),
    m_viewCount(0),
    m_subscriberCount(0),
    m_subscription(false),
    m_subscribed(false),
    m_accountLinked(true),
    m_service(Services::NoService)
{
}

UserItem::UserItem(UserItem *user, QObject *parent) :
    QObject(parent),
    m_id(user->id()),
    m_username(user->username()),
    m_channelName(user->channelName()),
    m_description(user->description()),
    m_avatarUrl(user->avatarUrl()),
    m_videoCount(user->videoCount()),
    m_viewCount(user->viewCount()),
    m_subscriberCount(user->subscriberCount()),
    m_subscription(user->subscription()),
    m_subscribed(user->subscribed()),
    m_subscriptionId(user->subscriptionId()),
    m_accountLinked(user->accountLinked()),
    m_service(user->service())
{
    switch (this->service()) {
    case Services::YouTube:
        this->connect(YouTube::instance(), SIGNAL(subscriptionChanged(QString,bool,QString)), this, SLOT(onSubscriptionChanged(QString,bool,QString)));
        break;
    case Services::Dailymotion:
        this->connect(Dailymotion::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
        break;
    case Services::Vimeo:
        this->connect(Vimeo::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
        break;
    default:
        break;
    }
}

void UserItem::loadUser(UserItem *user) {
    this->setId(user->id());
    this->setUsername(user->username());
    this->setChannelName(user->channelName());
    this->setDescription(user->description());
    this->setAvatarUrl(user->avatarUrl());
    this->setVideoCount(user->videoCount());
    this->setViewCount(user->viewCount());
    this->setSubscriberCount(user->subscriberCount());
    this->setSubscription(user->subscription());
    this->setSubscribed(user->subscribed());
    this->setSubscriptionId(user->subscriptionId());
    this->setAccountLinked(user->accountLinked());
    this->setService(user->service());

    switch (this->service()) {
    case Services::YouTube:
        this->connect(YouTube::instance(), SIGNAL(subscriptionChanged(QString,bool,QString)), this, SLOT(onSubscriptionChanged(QString,bool,QString)));
        break;
    case Services::Dailymotion:
        this->connect(Dailymotion::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
        break;
    case Services::Vimeo:
        this->connect(Vimeo::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
        break;
    default:
        break;
    }
}

void UserItem::loadDailymotionUser(const QVariantMap &user, bool subscribed) {
    this->setId(user.value("id").toString());
    this->setUsername(user.value("username").toString());
    this->setChannelName(user.value("screenname").toString());
    this->setDescription(user.value("description").toString());
    this->setAvatarUrl(user.value("avatar_medium_url").toUrl());
    this->setVideoCount(user.value("videos_total").toInt());
    this->setViewCount(user.value("views_total").toInt());
    this->setSubscriberCount(0);
    this->setSubscribed(subscribed);
    this->setSubscriptionId(this->id());
    this->setService(Services::Dailymotion);
    this->connect(Dailymotion::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
}

void UserItem::loadYouTubeUser(const QDomNode &user, bool subscription, bool subscribed) {
    this->setId(subscription ? user.firstChildElement("yt:channelId").text().section("UC", 1) : user.firstChildElement("author").firstChildElement("yt:userId").text());
    this->setUsername(subscription ? user.firstChildElement("yt:username").attribute("display") : user.firstChildElement("author").firstChildElement("name").text());
    this->setChannelName(user.firstChildElement("title").text());
    this->setDescription(user.firstChildElement("summary").text());
    this->setAvatarUrl(subscription ? QUrl(user.firstChildElement("media:thumbnail").attribute("url")) : QUrl(QString("http://i2.ytimg.com/i/%1/1.jpg").arg(this->id())));
    this->setVideoCount(subscription ? user.firstChildElement("yt:countHint").text().toInt() : user.firstChildElement("gd:feedLink").attribute("countHint").toInt());
    this->setViewCount(user.firstChildElement("yt:statistics").attribute("viewCount").toInt());
    this->setSubscriberCount(user.firstChildElement("yt:statistics").attribute("subscriberCount").toInt());
    this->setLocation(user.firstChildElement("yt:location").text());
    this->setSubscription(subscription);
    this->setSubscribed(subscribed);
    this->setSubscriptionId(user.firstChildElement("id").text().section(':', -1));
    this->setAccountLinked(user.firstChildElement("yt:incomplete").isNull());
    this->setService(Services::YouTube);
    this->connect(YouTube::instance(), SIGNAL(subscriptionChanged(QString,bool,QString)), this, SLOT(onSubscriptionChanged(QString,bool,QString)));
}

void UserItem::loadVimeoContact(const QDomElement &contact, bool subscribed) {
    this->setSubscribed((subscribed) || (contact.attribute("mutual").toInt()));
    this->setId(contact.attribute("id"));
    this->setSubscriptionId(this->id());
    this->setUsername(contact.attribute("username"));
    this->setChannelName(contact.attribute("display_name"));
    this->setService(Services::Vimeo);

    QDomNodeList avatars = contact.firstChildElement("portraits").childNodes();

    if (avatars.size() > 1) {
        this->setAvatarUrl(QUrl(avatars.at(1).toElement().text()));
    }

    this->connect(Vimeo::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
}

void UserItem::loadVimeoProfile(const QDomElement &profile) {
    this->setSubscribed(profile.attribute("is_contact").toInt());
    this->setId(profile.attribute("id"));
    this->setSubscriptionId(this->id());
    this->setUsername(profile.firstChildElement("username").text());
    this->setChannelName(profile.firstChildElement("display_name").text());
    this->setDescription(profile.firstChildElement("bio").text());
    this->setWebsiteUrl(QUrl(profile.firstChildElement("url").text()));
    this->setVideoCount(profile.firstChildElement("number_of_uploads").text().toInt());
    this->setLocation(profile.firstChildElement("location").text());
    this->setService(Services::Vimeo);

    QDomNodeList avatars = profile.firstChildElement("portraits").childNodes();

    if (avatars.size() > 1) {
        this->setAvatarUrl(QUrl(avatars.at(1).toElement().text()));
    }

    this->connect(Vimeo::instance(), SIGNAL(subscriptionChanged(QString,bool)), this, SLOT(onSubscriptionChanged(QString,bool)));
}

void UserItem::onSubscriptionChanged(const QString &userId, bool subscribed, const QString &subscriptionId) {
    if ((userId == this->id()) || (userId == this->subscriptionId())) {
        this->setSubscribed(subscribed);
        this->setSubscriptionId(subscriptionId);
    }
}
