#include "resourceloader.h"
#include "dailymotionchannelwindow.h"
#include "dailymotionplaylistvideoswindow.h"
#include "dailymotionvideoinfowindow.h"
#include "vimeochannelwindow.h"
#include "vimeoalbumvideoswindow.h"
#include "vimeovideoinfowindow.h"
#include "youtubechannelwindow.h"
#include "youtubeplaylistvideoswindow.h"
#include "youtubevideoinfowindow.h"
#include "videoplaybackwindow.h"
#include "windowstack.h"
#include "../base/youtube.h"
#include "../base/dailymotion.h"
#include "../base/vimeo.h"
#include "../base/utils.h"
#include "../base/urlgrabber.h"
#include "../base/settings.h"

ResourceLoader* ResourceLoader::self = 0;

ResourceLoader::ResourceLoader(QObject *parent) :
    QObject(parent)
{
    if (!self) {
        self = this;
    }
}

ResourceLoader::~ResourceLoader() {}

ResourceLoader* ResourceLoader::instance() {
    return !self ? new ResourceLoader : self;
}

bool ResourceLoader::display(const QString &url, bool unwindStack) {
    QVariantMap map = Utils::parseIdFromUrl(url);

    if (map.isEmpty()) {
        return false;
    }

    if (unwindStack) {
        WindowStack::instance()->unwindWindowStack();
    }

    switch (map.value("type").toInt()) {
    case Queries::Users:
        this->displayUser(static_cast<Services::VideoService>(map.value("service").toInt()), map.value("id").toString());
        break;
    case Queries::Playlists:
        this->displayUser(static_cast<Services::VideoService>(map.value("service").toInt()), map.value("id").toString());
        break;
    default:
        this->displayVideo(static_cast<Services::VideoService>(map.value("service").toInt()), map.value("id").toString());
        break;
    }

    return true;
}

bool ResourceLoader::display(const QStringList &url, bool unwindStack) {
    if (url.isEmpty()) {
        return false;
    }

    return this->display(url.first(), unwindStack);
}

bool ResourceLoader::play(const QString &url, bool unwindStack) {
    QVariantMap map = Utils::parseIdFromUrl(url);

    if (map.isEmpty()) {
        return false;
    }

    switch (map.value("type").toInt()) {
    case Queries::Videos:
        if (unwindStack) {
            WindowStack::instance()->unwindWindowStack();
        }

        this->playVideo(static_cast<Services::VideoService>(map.value("service").toInt()), map.value("id").toString());
        return true;
    default:
        return false;
    }
}

bool ResourceLoader::play(const QStringList &url, bool unwindStack) {
    if (url.isEmpty()) {
        return false;
    }

    return this->play(url.first(), unwindStack);
}

void ResourceLoader::displayUser(Services::VideoService service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->displayYouTubeUser(id);
        return;
    case Services::Dailymotion:
        this->displayDailymotionUser(id);
        return;
    case Services::Vimeo:
        this->displayVimeoUser(id);
        return;
    default:
        return;
    }
}

void ResourceLoader::displayYouTubeUser(const QString &id) {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(id, WindowStack::instance()->currentWindow());
    window->show();
}

void ResourceLoader::displayDailymotionUser(const QString &id) {
    DailymotionChannelWindow *window = new DailymotionChannelWindow(id, WindowStack::instance()->currentWindow());
    window->show();
}

void ResourceLoader::displayVimeoUser(const QString &id) {
    VimeoChannelWindow *window = new VimeoChannelWindow(id, WindowStack::instance()->currentWindow());
    window->show();
}

void ResourceLoader::displayVideo(Services::VideoService service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        YouTube::instance()->getFullVideo(id);
        this->connect(YouTube::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(displayYouTubeVideo(QSharedPointer<VideoItem>)));
        return;
    case Services::Dailymotion:
        Dailymotion::instance()->getFullVideo(id);
        this->connect(Dailymotion::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(displayDailymotionVideo(QSharedPointer<VideoItem>)));
        return;
    case Services::Vimeo:
        Vimeo::instance()->getFullVideo(id);
        this->connect(Vimeo::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(displayVimeoVideo(QSharedPointer<VideoItem>)));
        return;
    default:
        return;
    }
}

void ResourceLoader::displayYouTubeVideo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(video, WindowStack::instance()->currentWindow());
    window->show();

    this->disconnect(YouTube::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(displayYouTubeVideo(QSharedPointer<VideoItem>)));
}

void ResourceLoader::displayDailymotionVideo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(video, WindowStack::instance()->currentWindow());
    window->show();

    this->disconnect(Dailymotion::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(displayDailymotionVideo(QSharedPointer<VideoItem>)));
}

void ResourceLoader::displayVimeoVideo(QSharedPointer<VideoItem> video) {
    VimeoVideoInfoWindow *window = new VimeoVideoInfoWindow(video, WindowStack::instance()->currentWindow());
    window->show();

    this->disconnect(Vimeo::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(displayDailymotionVideo(QSharedPointer<VideoItem>)));
}

void ResourceLoader::playVideo(Services::VideoService service, const QString &id) {
    if (Settings::instance()->mediaPlayer() != "cutetube") {
        UrlGrabber::instance()->getVideoUrl(service, id);
        return;
    }

    switch (service) {
    case Services::YouTube:
        YouTube::instance()->getFullVideo(id);
        this->connect(YouTube::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(playVideo(QSharedPointer<VideoItem>)));
        return;
    case Services::Dailymotion:
        Dailymotion::instance()->getFullVideo(id);
        this->connect(Dailymotion::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(playVideo(QSharedPointer<VideoItem>)));
        return;
    case Services::Vimeo:
        Vimeo::instance()->getFullVideo(id);
        this->connect(Vimeo::instance(), SIGNAL(gotVideo(QSharedPointer<VideoItem>)), this, SLOT(playVideo(QSharedPointer<VideoItem>)));
        return;
    default:
        return;
    }
}

void ResourceLoader::playVideo(QSharedPointer<VideoItem> video) {
    VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << video, WindowStack::instance()->currentWindow());
    window->show();
}
