#include "dailymotionvideolistmodel.h"
#include "dailymotion.h"
#include "feedurls.h"
#include "json.h"
#include "playbacklistmodel.h"
#include "transfermanager.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif

using namespace QtJson;

DailymotionVideoListModel::DailymotionVideoListModel(QObject * parent) :
    AbstractVideoListModel(parent),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1),
    m_refresh(false)
{
}

void DailymotionVideoListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        this->disconnect(Dailymotion::instance());

        if (feed == DAILYMOTION_UPLOADS_FEED) {
            this->connect(Dailymotion::instance(), SIGNAL(deletedFromUploads(QString)), this, SLOT(onDeletedFromUploads(QString)));
        }
        else if (feed == DAILYMOTION_FAVOURITES_FEED) {
            this->connect(Dailymotion::instance(), SIGNAL(favouriteChanged(QString,bool)), this, SLOT(onFavouriteChanged(QString,bool)));
        }
        else if (feed.toString().startsWith("https://api.dailymotion.com/playlist")) {
            this->setPlaylistId(feed.toString().section('/', -2, -2));
            this->connect(Dailymotion::instance(), SIGNAL(addedToPlaylist(QString,QString)), this, SLOT(onAddedToPlaylist(QString,QString)));
            this->connect(Dailymotion::instance(), SIGNAL(deletedFromPlaylist(QString,QString)), this, SLOT(onDeletedFromPlaylist(QString,QString)));
        }
        else if (feed.toString().startsWith("https://api.dailymotion.com/group")) {
            this->setGroupId(feed.toString().section('/', -2, -2));
        }

        emit feedChanged(feed);
    }
}

void DailymotionVideoListModel::clear() {
    AbstractVideoListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void DailymotionVideoListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreVideos();
    }
}

void DailymotionVideoListModel::getMoreVideos() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getVideos();
        }
    }
}

void DailymotionVideoListModel::getVideos(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }
#if QT_VERSION >= 0x050000
    QUrlQuery query(feed);
    query.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    feed.setQuery(query);
#else
    feed.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
#endif
    this->setLoading(true);
    QNetworkReply *reply = Dailymotion::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void DailymotionVideoListModel::getRelatedVideos(const QString &id) {
    this->setFeed(QUrl(QString("https://api.dailymotion.com/video/%1/related?limit=30").arg(id)));
    this->getVideos();
}

void DailymotionVideoListModel::getPlaylistVideos(const QString &id) {
    this->setFeed(QUrl(QString("https://api.dailymotion.com/playlist/%1/videos?limit=100").arg(id)));
    this->getVideos();
}

void DailymotionVideoListModel::getGroupVideos(const QString &id) {
    this->setFeed(QUrl(QString("https://api.dailymotion.com/group/%1/videos?limit=30").arg(id)));
    this->getVideos();
}

void DailymotionVideoListModel::search(const QString &query, int order, const QString &language) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
        this->setLanguageFilter(language);
    }

    this->setLoading(true);
    QNetworkReply *reply = Dailymotion::instance()->createSearchReply(Queries::Videos, this->query(), this->offset(), this->order(), this->languageFilter());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void DailymotionVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideos()));
        this->connect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        Dailymotion::instance()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                VideoItem *video = new VideoItem;
                video->loadDailymotionVideo(entries.at(i).toMap(), this->feed() == DAILYMOTION_FAVOURITES_FEED);
                QMetaObject::invokeMethod(this, "appendVideo", Qt::QueuedConnection, Q_ARG(QSharedPointer<VideoItem>, QSharedPointer<VideoItem>(video)));
            }

            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);

            if ((!this->playlistId().isEmpty()) && (this->moreResults())) {
                this->getVideos();
            }
            else {
                QTimer::singleShot(1000, this, SLOT(stopLoading()));
            }
        }

        this->disconnect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideos()));
        this->disconnect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionVideoListModel::onDeletedFromUploads(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void DailymotionVideoListModel::onFavouriteChanged(const QString &videoId, bool favourite) {
    if (favourite) {
        this->setRefreshRequired(true);
    }
    else if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void DailymotionVideoListModel::onAddedToPlaylist(const QString &videoId, const QString &playlistId) {
    Q_UNUSED(videoId)

    if (playlistId == this->playlistId()) {
        this->setRefreshRequired(true);
    }
}

void DailymotionVideoListModel::onDeletedFromPlaylist(const QString &videoId, const QString &playlistId) {
    if ((playlistId == this->playlistId()) && (this->removeVideo(videoId))) {
        emit countChanged(this->rowCount());
    }
}

void DailymotionVideoListModel::deleteSelectedVideosFromUploads() {
    if (this->feed() == DAILYMOTION_UPLOADS_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        Dailymotion::instance()->deleteFromUploads(videoIds);
    }

    this->selectNone();
}

void DailymotionVideoListModel::addSelectedVideosToFavourites() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Dailymotion::instance()->addToFavourites(videoIds);
    this->selectNone();
}

void DailymotionVideoListModel::deleteSelectedVideosFromFavourites() {
    if (this->feed() == DAILYMOTION_FAVOURITES_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        Dailymotion::instance()->deleteFromFavourites(videoIds);
    }

    this->selectNone();
}

void DailymotionVideoListModel::addSelectedVideosToPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Dailymotion::instance()->addToPlaylist(videoIds, playlistId);
    this->selectNone();
}

void DailymotionVideoListModel::addSelectedVideosToNewPlaylist(const QVariantMap &playlist) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Dailymotion::instance()->createPlaylist(playlist, videoIds);
    this->selectNone();
}

void DailymotionVideoListModel::deleteSelectedVideosFromPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Dailymotion::instance()->deleteFromPlaylist(videoIds, playlistId);
    this->selectNone();
}

void DailymotionVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    TransferManager::instance()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void DailymotionVideoListModel::queueSelectedVideos() {
    PlaybackListModel::playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}
