#include "transferdetailsdialog.h"
#include "transferpriorityselector.h"
#include "connectionsselector.h"
#include "subtitlesselector.h"
#include "../base/utils.h"
#include <QLabel>
#include <QGridLayout>
#include <QScrollArea>
#include <QLineEdit>
#include <QCheckBox>
#include <QMaemo5Style>

TransferDetailsDialog::TransferDetailsDialog(QSharedPointer<TransferItem> transfer, QWidget *parent) :
    AbstractDialog(parent),
    m_transfer(transfer),
    m_titleHeader(new QLabel(tr("Details") + ":", this)),
    m_statusHeader(new QLabel(tr("Status") + ":", this)),
    m_sizeHeader(new QLabel(tr("Size") + ":", this)),
    m_progressHeader(new QLabel(tr("Progress") + ":", this)),
    m_statusLabel(new QLabel(this)),
    m_statusIcon(new QLabel(this)),
    m_sizeLabel(new QLabel(this)),
    m_progressLabel(new QLabel(this)),
    m_speedLabel(new QLabel(this)),
    m_connectionsSelector(0),
    m_prioritySelector(new TransferPrioritySelector(tr("Priority"), this)),
    m_subtitlesSelector(0),
    m_subtitlesCheckbox(0),
    m_audioCheckbox(0),
    m_nameEdit(new QLineEdit(m_transfer.data()->fileName(), this)),
    m_nameButton(new QPushButton(tr("Change"), this)),
    m_pauseResumeButton(new QPushButton(this)),
    m_cancelButton(new QPushButton(tr("Cancel"), this)),
    m_grid(0)
{
    this->setWindowTitle(tr("Transfer status"));
    this->setAttribute(Qt::WA_DeleteOnClose, true);

    switch (m_transfer.data()->transferType()) {
    case Transfers::Upload:
        m_nameEdit->setEnabled(false);
        break;
    default:
        m_connectionsSelector = new ConnectionsSelector(tr("Connections"), this);
        m_subtitlesSelector = new SubtitlesSelector(tr("Subtitles language"), this);
        m_subtitlesCheckbox = new QCheckBox(tr("Download subtitles"), this);
        m_audioCheckbox = new QCheckBox(tr("Save as audio"), this);
        m_connectionsSelector->setValue(m_transfer.data()->preferredConnections());
        m_subtitlesSelector->setValue(m_transfer.data()->subtitlesLanguage());
        m_subtitlesCheckbox->setChecked(m_transfer.data()->downloadSubtitles());
        m_audioCheckbox->setChecked(m_transfer.data()->convertToAudio());
        this->connect(m_connectionsSelector, SIGNAL(valueChanged(int)), m_transfer.data(), SLOT(setPreferredConnections(int)));
        this->connect(m_subtitlesSelector, SIGNAL(valueChanged(QString)), m_transfer.data(), SLOT(setSubtitlesLanguage(QString)));
        this->connect(m_nameEdit, SIGNAL(textEdited(QString)), this, SLOT(onFileNameTextChanged(QString)));
        this->connect(m_nameButton, SIGNAL(clicked()), this, SLOT(setTransferFileName()));
        this->connect(m_subtitlesCheckbox, SIGNAL(clicked(bool)), m_transfer.data(), SLOT(setDownloadSubtitles(bool)));
        this->connect(m_audioCheckbox, SIGNAL(clicked(bool)), m_transfer.data(), SLOT(setConvertToAudio(bool)));
        break;
    }

    m_prioritySelector->setValue(m_transfer.data()->priority());
    m_titleHeader->setFixedWidth(100);
    m_statusHeader->setFixedWidth(100);
    m_sizeHeader->setFixedWidth(100);
    m_progressHeader->setFixedWidth(100);
    m_statusLabel->setWordWrap(true);
    m_sizeLabel->setFixedWidth(100);
    m_sizeLabel->setAlignment(Qt::AlignCenter);
    m_progressLabel->setFixedWidth(100);
    m_progressLabel->setAlignment(Qt::AlignCenter);
    m_statusIcon->setFixedSize(48, 48);
    m_statusIcon->setScaledContents(true);
    m_nameEdit->setCursorPosition(0);
    m_nameButton->setEnabled(false);
    m_nameButton->setFixedWidth(150);
    m_pauseResumeButton->setFixedWidth(150);
    m_cancelButton->setFixedWidth(150);

    QScrollArea *scrollArea = new QScrollArea(this);
    QWidget *scrollWidget = new QWidget(scrollArea);

    m_grid = new QGridLayout(scrollWidget);
    m_grid->setContentsMargins(0, 0, 0, 0);

    scrollArea->setWidgetResizable(true);
    scrollArea->setWidget(scrollWidget);
    scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    QGridLayout *grid2 = new QGridLayout(this);
    grid2->addWidget(scrollArea);

    this->onTransferStatusChanged(m_transfer.data()->status());
    this->onTransferSizeChanged(m_transfer.data()->size());
    this->onTransferProgressChanged(m_transfer.data()->progress());
    this->onTransferSpeedChanged(m_transfer.data()->speed());

    this->connect(m_transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));
    this->connect(m_transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged(qint64)));
    this->connect(m_transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged(int)));
    this->connect(m_transfer.data(), SIGNAL(speedChanged(int)), this, SLOT(onTransferSpeedChanged(int)));
    this->connect(m_prioritySelector, SIGNAL(valueChanged(Transfers::Priority)), m_transfer.data(), SLOT(setPriority(Transfers::Priority)));
    this->connect(m_pauseResumeButton, SIGNAL(clicked()), this, SLOT(onPauseResumeClicked()));
    this->connect(m_cancelButton, SIGNAL(clicked()), m_transfer.data(), SLOT(cancel()));

    this->onOrientationChanged();
}

void TransferDetailsDialog::setLandscapeLayout() {
    this->setMinimumHeight(340);

    switch (m_transfer.data()->transferType()) {
    case Transfers::Upload:
        if (m_grid->isEmpty()) {
            m_grid->addWidget(m_titleHeader, 0, 0);
            m_grid->addWidget(m_nameEdit, 0, 1, 1, 2);
            m_grid->addWidget(m_nameButton, 0, 3);
            m_grid->addWidget(m_statusHeader, 1, 0);
            m_grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
            m_grid->addWidget(m_statusLabel, 1, 2, 1, 2);
            m_grid->addWidget(m_sizeHeader, 2, 0);
            m_grid->addWidget(m_sizeLabel, 2, 1);
            m_grid->addWidget(m_progressHeader, 3, 0);
            m_grid->addWidget(m_progressLabel, 3, 1);
            m_grid->addWidget(m_speedLabel, 3, 2);
            m_grid->addWidget(m_pauseResumeButton, 3, 3);
            m_grid->addWidget(m_prioritySelector, 4, 0, 1, 3);
            m_grid->addWidget(m_cancelButton, 4, 3);
        }

        return;
    default:
        if (m_grid->isEmpty()) {
            m_grid->addWidget(m_titleHeader, 0, 0);
            m_grid->addWidget(m_nameEdit, 0, 1, 1, 2);
            m_grid->addWidget(m_nameButton, 0, 3);
            m_grid->addWidget(m_statusHeader, 1, 0);
            m_grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
            m_grid->addWidget(m_statusLabel, 1, 2, 1, 2);
            m_grid->addWidget(m_sizeHeader, 2, 0);
            m_grid->addWidget(m_sizeLabel, 2, 1);
            m_grid->addWidget(m_progressHeader, 3, 0);
            m_grid->addWidget(m_progressLabel, 3, 1);
            m_grid->addWidget(m_speedLabel, 3, 2);
            m_grid->addWidget(m_audioCheckbox, 4, 0, 1, 3);
            m_grid->addWidget(m_subtitlesCheckbox, 5, 0, 1, 3);
            m_grid->addWidget(m_subtitlesSelector, 6, 0, 1, 3);
            m_grid->addWidget(m_connectionsSelector, 7, 0, 1, 3);
            m_grid->addWidget(m_pauseResumeButton, 7, 3);
            m_grid->addWidget(m_prioritySelector, 8, 0, 1, 3);
            m_grid->addWidget(m_cancelButton, 8, 3);
        }

        return;
    }
}

void TransferDetailsDialog::setPortraitLayout() {
    this->setMinimumHeight(420);

    switch (m_transfer.data()->transferType()) {
    case Transfers::Upload:
        if (m_grid->isEmpty()) {
            m_grid->addWidget(m_titleHeader, 0, 0);
            m_grid->addWidget(m_nameEdit, 0, 1, 1, 2);
            m_grid->addWidget(m_nameButton, 0, 3);
            m_grid->addWidget(m_statusHeader, 1, 0);
            m_grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
            m_grid->addWidget(m_statusLabel, 1, 2, 1, 2);
            m_grid->addWidget(m_sizeHeader, 2, 0);
            m_grid->addWidget(m_sizeLabel, 2, 1);
            m_grid->addWidget(m_progressHeader, 3, 0);
            m_grid->addWidget(m_progressLabel, 3, 1);
            m_grid->addWidget(m_speedLabel, 3, 2);
            m_grid->addWidget(m_pauseResumeButton, 3, 3);
            m_grid->addWidget(m_prioritySelector, 4, 0, 1, 3);
            m_grid->addWidget(m_cancelButton, 4, 3);
        }

        return;
    default:
        if (m_grid->isEmpty()) {
            m_grid->addWidget(m_titleHeader, 0, 0);
            m_grid->addWidget(m_nameEdit, 0, 1, 1, 2);
            m_grid->addWidget(m_nameButton, 0, 3);
            m_grid->addWidget(m_statusHeader, 1, 0);
            m_grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
            m_grid->addWidget(m_statusLabel, 1, 2, 1, 2);
            m_grid->addWidget(m_sizeHeader, 2, 0);
            m_grid->addWidget(m_sizeLabel, 2, 1);
            m_grid->addWidget(m_progressHeader, 3, 0);
            m_grid->addWidget(m_progressLabel, 3, 1);
            m_grid->addWidget(m_speedLabel, 3, 2);
            m_grid->addWidget(m_audioCheckbox, 4, 0, 1, 3);
            m_grid->addWidget(m_subtitlesCheckbox, 5, 0, 1, 3);
            m_grid->addWidget(m_subtitlesSelector, 6, 0, 1, 3);
            m_grid->addWidget(m_connectionsSelector, 7, 0, 1, 3);
            m_grid->addWidget(m_pauseResumeButton, 7, 3);
            m_grid->addWidget(m_prioritySelector, 8, 0, 1, 3);
            m_grid->addWidget(m_cancelButton, 8, 3);
        }

        return;
    }
}

void TransferDetailsDialog::onTransferStatusChanged(Transfers::Status status) {
    m_statusLabel->setText(m_transfer.data()->statusString());

    switch (status) {
    case Transfers::Downloading:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/as_proxy_attachment_download.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        break;
    case Transfers::Connecting:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_packetdata.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        break;
    case Transfers::Paused:
        m_statusIcon->setPixmap(QPixmap("/etc/hildon/theme/mediaplayer/Pause.png"));
        m_pauseResumeButton->setText(tr("Start"));
        break;
    case Transfers::Converting:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_audio_file.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        break;
    case Transfers::Failed:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_stop.png"));
        m_pauseResumeButton->setText(tr("Start"));
        break;
    case Transfers::Completed:
        this->accept();
        break;
    case Transfers::Cancelled:
        this->accept();
        break;
    default:
        m_statusIcon->setPixmap(QPixmap("/etc/hildon/theme/mediaplayer/Stop.png"));
        m_pauseResumeButton->setText(tr("Pause"));
    }
}

void TransferDetailsDialog::onTransferSizeChanged(qint64 size) {
    m_sizeLabel->setText(size > 0 ? Utils::fileSizeFromBytes(size) : "-");
}

void TransferDetailsDialog::onTransferProgressChanged(int progress) {
    m_progressLabel->setText(QString("%1%").arg(progress));
}

void TransferDetailsDialog::onTransferSpeedChanged(int speed) {
    m_speedLabel->setText(QString::number(speed) + " kB/s");
}

void TransferDetailsDialog::onFileNameTextChanged(const QString &text) {
    m_nameButton->setEnabled(!text.isEmpty());
}

void TransferDetailsDialog::setTransferFileName() {
    m_transfer.data()->setFileName(m_nameEdit->text());
    m_nameButton->setEnabled(false);
}

void TransferDetailsDialog::onPauseResumeClicked() {
    switch (m_transfer.data()->status()) {
    case Transfers::Paused:
    case Transfers::Failed:
        m_transfer.data()->queue();
        return;
    default:
        m_transfer.data()->pause();
        return;
    }
}
