import socket
from select import select
import random

from square import SquareGame, SquareObserver
import timer

# Server <=> Client
# ==> game description
# <== local usernames
# ==> player numbers
# ==> all usernames
# <=> setting points

PORT = 9557

class SocketException(Exception):
    pass

def read_int(str):
    num = 0

    for i in reversed(str):
        num = num << 8
        num += ord(i)

    return num

def pack_int(num):
    l = []

    while num:
        l.append(chr(num & 0xff))
        num = num >> 8

    return ''.join(l)

# TODO: overflow-detection ...
class BufferedSocket():
    '''Buffering socket wrapper working with byte-iterators'''

    def __init__(self, socket):
        self.socket = socket
        self.buffer = ""

    def fileno(self):
        return self.socket.fileno()

    def close(self):
        self.socket.close()

    def package_size(self):
        size = ord(self.buffer[0])
        return read_int(self.buffer[1:size+1]) + 1 + ord(self.buffer[0])

    def package_ready(self):
        # anything received
        if len(self.buffer) > 0:
            # package size received
            if len(self.buffer) > ord(self.buffer[0]):
                # package ready
                if len(self.buffer) >= self.package_size():
                    return True

        return False

    def read(self):
        ret = None
        while ret == None:
            ret = self.try_read()

        return ret

    def try_read(self):
        if not self.package_ready():
            recv = self.socket.recv(1024)
        
            if len(recv) == 0:
                raise SocketException()

            self.buffer += recv

        return self.unpack_list()

    def unpack_list(self):
        if self.package_ready():
            size = self.package_size()
            
            data = self.buffer[:size]
            self.buffer = self.buffer[size:]

            # killing the size
            data = data[ord(data[0])+1:]

            list = []

            # enter the tuple loop
            while data:
                size_size = ord(data[0])
                size = read_int(data[1:size_size+1])
                data = data[size_size+1:]

                sub_data = data[:size]
                data = data[size:]

                type = sub_data[0]
                payload = sub_data[1:]

                if type == 'i':
                    res = read_int(payload)
                elif type == 's':
                    res = payload
                else:
                    print "unknown type: %s" % type
                    raise NotImplementedError

                list.append(res)

            return list
        else:
            return None

    def write(self, seq):
        size = 0
        out = []
        for e in seq:
            if isinstance(e, str):
                data = e
                type = 's'
            elif isinstance(e, int) or isinstance(e, long):
                data = pack_int(e)
                type = 'i'
            else:
                raise NotImplementedError

            size = pack_int(len(data) + 1)

            out.append(chr(len(size)))
            out.append(size)
            out.append(type)
            out.append(data)

        payload = ''.join(out)
        size = pack_int(len(payload))
        data = ''.join([chr(len(size)), size, payload])

        self.socket.sendall(data)

class Client(SquareObserver):
    '''The actual game, setting points'''

    def __init__(self, address, request_players):
        plain_socket = socket.socket()
        plain_socket.connect((address, PORT))

        self.socket = BufferedSocket(plain_socket)
        
        self.game = self.create_game(request_players)
        self.game.add_square_observer(self)

    def create_game(self, request_players):
        s = self.socket

        # getting game informations
        (size, player_amount, win_target, win_offset, timeout) = s.read()

        print "board with size %i for %i players" % (size, player_amount)
        print "win with %i points and an offset of %i" % (win_target, win_offset)

        # requesting players
        s.write(request_players)

        # who are we?
        self.local_player = s.read()
        print "our positions " + str(self.local_player)
        print "waiting for other players to connect ..."
        
        players = s.read()

        print "players ready " + str(players)
        
        game = SquareGame(size, players, win_target, win_offset)
        
        if timeout:
            print "setting timeout of %i" % timeout
            timer.BlitzSquare(game, timeout, managed = self.local_player)

        return game

    def point_set(self, player, point, squares):
        if player in self.local_player:
            self.socket.write((player, point[0], point[1]))

    def run(self):
        # TODO: Game ends?
        try:
            while True:
                (player, x, y) = self.socket.read()
                self.game.set_point((x, y), player)
        except SocketException:
            print 'communication error'

        self.socket.close()


class Server():
    '''Dumb server'''

    def __init__(self, size, player, win_target, win_offset, timeout):
        self.size = size
        self.player = player
        self.win_target = win_target
        self.win_offset = win_offset
        self.timeout = timeout
        self.sockets = []

    def run(self):
        print('==> Initializing')
        self.initialize()
        print('==> Starting the game')
        self.repeat()
        print('==> Cleaning up')
        self.close()

    def close(self):
        for s in self.sockets:
            s.close()

    def initialize(self):
        server = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        server.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        server.bind(('', PORT))

        server.listen(4)

        slots = list(range(self.player))
        random.shuffle(slots)

        players = [None]*self.player

        while(slots):
            print 'Waiting for connection ...'
            plain_socket, address = server.accept()
            print 'Accepted ' + str(address)

            s = BufferedSocket(plain_socket)

            self.sockets.append(s)

            # game info
            s.write((self.size, self.player, self.win_target, self.win_offset, self.timeout))

            # request
            socket_players = []
            requested_players = s.read()
            for name in requested_players:
                if slots:
                    print "Registered " + name
                    players[slots[0]] = name
                    socket_players.append(slots[0])
                    del slots[0]

            # player positions
            s.write(socket_players)

        server.close()

        print "Sending Player Names ..."
        for s in self.sockets:
            s.write(players)

    def repeat(self):
        try:
            # TODO: sterben ...
            while True:
                (rlist, _, _) = select(self.sockets, [], [])
                
                for recv in rlist:
                    msg = recv.try_read()

                    if msg != None:
                        print "sending " + str(msg)

                        for send in self.sockets:
                            if send != recv:
                                send.write(msg)
        except SocketException:
            print "error in communication"

