/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef TUNEINSTATIONLISTMODEL_H
#define TUNEINSTATIONLISTMODEL_H

#include <QAbstractListModel>
#include "../base/station.h"

class TuneInStationList;
#ifndef QML_USER_INTERFACE
class ThumbnailCache;
#endif

class TuneInStationListModel : public QAbstractListModel
{
    Q_OBJECT
    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool moreResults
               READ moreResults
               NOTIFY loadingChanged)
    Q_PROPERTY(QUrl nextUrl
               READ nextUrl
               NOTIFY loadingChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(QString searchQuery
               READ searchQuery
               NOTIFY searchQueryChanged)
    Q_PROPERTY(Queries::QueryType queryType
               READ queryType
               NOTIFY queryTypeChanged)

public:
    enum Roles {
        ServiceRole = Qt::UserRole + 1,
        IdRole,
        TitleRole,
        DescriptionRole,
        LogoRole,
        GenreRole,
        CountryRole,
        LanguageRole,
        SourceRole,
        FavouriteRole,
        LastPlayedRole,
        SectionRole
    };

    explicit TuneInStationListModel(QObject *parent = 0);
    ~TuneInStationListModel();

#if QT_VERSION >= 0x050000
    QHash<int, QByteArray> roleNames() const;
#endif

    int rowCount(const QModelIndex &parent = QModelIndex()) const;

    QVariant data(const QModelIndex &index, int role) const;
    Q_INVOKABLE QVariant data(int row, const QByteArray &role) const;
    bool setData(const QModelIndex &index, const QVariant &value, int role);
    Q_INVOKABLE bool setData(int row, const QVariant &value, const QByteArray &role);

    bool moreResults() const;

    QUrl nextUrl() const;

    bool loading() const;

    QString searchQuery() const;

    Queries::QueryType queryType() const;

    Station* get(const QModelIndex &index) const;
    Q_INVOKABLE Station* get(int row) const;

public slots:
    void searchStations(const QString &query);
    void showStationsByGenre(const QString &id);
    void showStationsByCountry(const QString &id);
    void showStations(const QString &url);

    void getMoreResults();

    void clear();
    void reload();

protected:
    void setNextUrl(const QUrl &url);

    void setLoading(bool loading);

    void setSearchQuery(const QString &query);

    void setQueryType(Queries::QueryType type);

protected slots:
    void onStationListFinished(TuneInStationList *list);
    void onStationListCanceled(TuneInStationList *list);

    void addStations(QList<Station*> stations);
    void addStation(Station *station);
    void insertStation(int row, Station *station);
    void removeStation(int row);
    void removeStation(int role, const QVariant &value);
    void onStationUpdated(const QString &id, const QVariantMap &properties);
    void onStationUpdated(const QString &id, const QString &property, const QVariant &value);
    void onStationDeleted(const QString &id);
#ifndef QML_USER_INTERFACE
    void onThumbnailReady();
#endif

signals:
    void countChanged(int count);
    void loadingChanged(bool loading);
    void searchQueryChanged(const QString &query);
    void queryTypeChanged(Queries::QueryType type);
    void error(const QString &errorString);

private:
#ifndef QML_USER_INTERFACE
    ThumbnailCache *m_cache;
#endif
    QList<Station*> m_list;

    QUrl m_nextUrl;

    bool m_loading;

    QString m_searchQuery;

    Queries::QueryType m_queryType;

    QHash<int, QByteArray> m_roleNames;
};

#endif // TUNEINSTATIONLISTMODEL_H
