/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QDomDocument>
#include <QDomElement>
#include "tuneinstation.h"
#include "tuneinurls.h"
#include "../base/networkaccessmanager.h"

TuneInStation::TuneInStation(QObject *parent) :
    QObject(parent),
    m_reply(0),
    m_result(0)
{
}

TuneInStation::TuneInStation(QNetworkReply *reply, QObject *parent) :
    QObject(parent),
    m_reply(reply),
    m_result(0)
{
    this->connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

TuneInStation::~TuneInStation() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

Station* TuneInStation::result() const {
    return m_result;
}

void TuneInStation::setResult(Station *result) {
    m_result = result;
}

QNetworkReply::NetworkError TuneInStation::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString TuneInStation::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

void TuneInStation::load(const QString &id) {
    if (m_reply) {
        m_reply->deleteLater();
        m_reply = 0;
    }

    QUrl url(BASE_URL + DESCRIBE_METHOD);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("id", id);
    query.addQueryItem("partnerId", PARTNER_ID);
#else
    url.addQueryItem("id", id);
    url.addQueryItem("partnerId", PARTNER_ID);
#endif
    m_reply = NetworkAccessManager::instance()->get(QNetworkRequest(url));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void TuneInStation::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void TuneInStation::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    QDomDocument doc;
    doc.setContent(m_reply->readAll());

    QDomElement result = doc.documentElement().firstChildElement("body").firstChildElement("outline").firstChildElement("station");

    if (!result.isNull()) {
        Station *station = new Station;
        station->setService(Services::TuneIn);
        station->setId(result.firstChildElement("guide_id").text());
        station->setTitle(result.firstChildElement("name").text());
        station->setDescription(result.firstChildElement("description").text());
        station->setLogo(QString("http://d1i6vahw24eb07.cloudfront.net/%1q.png").arg(station->id()));
        station->setGenre(result.firstChildElement("genre_name").text());
        station->setCountry(result.firstChildElement("location").text().section(',', -1).trimmed());
        station->setLanguage(result.firstChildElement("language").text());
        station->setSource("http://opml.radiotime.com/Tune.ashx?id=" + station->id());
        this->setResult(station);
    }

    emit finished(this);
}
