/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QDomDocument>
#include <QDomElement>
#include "tuneinpodcastlist.h"

TuneInPodcastList::TuneInPodcastList(QNetworkReply *reply, QObject *parent) :
    QObject(parent),
    m_reply(reply)
{
    this->connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

TuneInPodcastList::~TuneInPodcastList() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

int TuneInPodcastList::count() const {
    return m_results.size();
}

QList<Station*> TuneInPodcastList::results() const {
    return m_results;
}

bool TuneInPodcastList::moreResults() const {
    return !m_nextUrl.isEmpty();
}

QUrl TuneInPodcastList::nextUrl() const {
    return m_nextUrl;
}

void TuneInPodcastList::setNextUrl(const QUrl &url) {
    m_nextUrl = url;
}

QNetworkReply::NetworkError TuneInPodcastList::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString TuneInPodcastList::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

void TuneInPodcastList::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void TuneInPodcastList::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    QDomDocument doc;
    doc.setContent(m_reply->readAll());

    QDomNodeList results = doc.documentElement().firstChildElement("body").elementsByTagName("outline");

    for (int i = 0; i < results.size(); i++) {
        QDomElement result = results.at(i).toElement();

        if (result.attribute("item") == "show") {
            Station *station = new Station;
            station->setService(Services::TuneIn);
            station->setId(result.attribute("guide_id"));
            station->setTitle(result.attribute("text"));
            station->setDescription(result.attribute("subtext"));
            station->setLogo(QString("http://d1i6vahw24eb07.cloudfront.net/%1q.png").arg(station->id()));
            station->setSource(result.attribute("URL"));
            m_results.append(station);
        }
        else if (result.attribute("key") == "nextShows") {
            m_nextUrl.setUrl(result.attribute("URL"));
        }
    }

    emit finished(this);
}
