/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QMessageBox>
#include <QVBoxLayout>
#include <QMaemo5Style>
#include "podcastlistwindow.h"
#include "podcastlistdelegate.h"
#include "podcastepisodelistwindow.h"
#include "tuneinpodcastepisodelistwindow.h"
#include "listview.h"
#include "filterbox.h"
#include "nowplayingbutton.h"
#include "editpodcastdialog.h"
#include "launcher.h"
#include "../base/podcastfilterlistmodel.h"
#include "../base/database.h"
#include "../base/settings.h"
#include "../base/utils.h"
#include "../base/streamextractor.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

PodcastListWindow::PodcastListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new PodcastFilterListModel(this)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_editAction(m_contextMenu->addAction(tr("Edit"), this, SLOT(showEditDialog()))),
    m_deleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deletePodcast()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No podcasts found")), this))
{
    this->setWindowTitle(tr("Podcasts"));
    this->setCentralWidget(m_noResultsLabel);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new PodcastListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_filterBox->hide();

    m_vbox = new QVBoxLayout(m_noResultsLabel);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), this, SLOT(onScrollingStopped()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(cleared()), NowPlayingButton::instance(), SLOT(poke()));
}

void PodcastListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void PodcastListWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (m_model->rowCount() > 0) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        m_filterBox->show();
        m_filterBox->setFocus(Qt::OtherFocusReason);
        m_filterBox->setFilterText(event->text());
    }
}

void PodcastListWindow::searchPodcasts(const QString &query) {
    this->setWindowTitle(tr("Search ('%1')").arg(query));
    m_model->searchPodcasts(query);
}

void PodcastListWindow::showPodcasts() {
    this->setWindowTitle(tr("Podcasts"));
    m_model->showPodcasts();
}

void PodcastListWindow::clear() {
    m_model->clear();
}

void PodcastListWindow::reload() {
    m_model->reload();
}

void PodcastListWindow::onScrollingStopped() {
    if (m_filterBox->isHidden()) {
        NowPlayingButton::instance()->poke();
    }
}

void PodcastListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_view->setHidden((!loading) && (m_model->rowCount() == 0));
}

void PodcastListWindow::onCountChanged(int count) {
    m_view->setHidden((!m_model->loading()) && (count == 0));
}

void PodcastListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_editAction);
}

void PodcastListWindow::showEditDialog() {
    if (m_view->currentIndex().isValid()) {
        EditPodcastDialog *dialog = new EditPodcastDialog(m_model->get(m_view->currentIndex()), this);
        dialog->open();
    }
}

void PodcastListWindow::deletePodcast() {
    if ((m_view->currentIndex().isValid()) && (QMessageBox::question(this, tr("Delete podcast?"), tr("Do you wish to delete the podcast '%1' from the database?")
                                                                     .arg(m_view->currentIndex().data(PodcastListModel::TitleRole).toString()), QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)) {

        Database::asyncDeletePodcast(m_view->currentIndex().data(PodcastListModel::IdRole).toString());
    }
}

void PodcastListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        switch (index.data(PodcastListModel::ServiceRole).toInt()) {
        case Services::TuneIn:
        {
            TuneInPodcastEpisodeListWindow *window = new TuneInPodcastEpisodeListWindow(this);
            window->showPodcastEpisodes(index.data(PodcastListModel::TitleRole).toString(), index.data(PodcastListModel::SourceRole).toString());
            window->show();
            return;
        }
        default:
        {
            PodcastEpisodeListWindow *window = new PodcastEpisodeListWindow(this);
            window->showPodcastEpisodes(index.data(PodcastListModel::TitleRole).toString(), index.data(PodcastListModel::SourceRole).toString());
            window->show();
            return;
        }
        }
    }
}
