/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef STATION_H
#define STATION_H

#include <QObject>
#include <QUrl>
#include <QVariantMap>
#include "enums.h"

class Station : public QObject
{
    Q_OBJECT
    Q_PROPERTY(Stations::StationType stationType
               READ stationType
               WRITE setStationType
               NOTIFY stationTypeChanged)
    Q_PROPERTY(Services::RadioService service
               READ service
               WRITE setService
               NOTIFY serviceChanged)
    Q_PROPERTY(QString id
               READ id
               WRITE setId
               NOTIFY idChanged)
    Q_PROPERTY(QString title
               READ title
               WRITE setTitle
               NOTIFY titleChanged)
    Q_PROPERTY(QString description
               READ description
               WRITE setDescription
               NOTIFY descriptionChanged)
    Q_PROPERTY(QUrl logo
               READ logo
               WRITE setLogo
               NOTIFY logoChanged)
    Q_PROPERTY(QString genre
               READ genre
               WRITE setGenre
               NOTIFY genreChanged)
    Q_PROPERTY(QString country
               READ country
               WRITE setCountry
               NOTIFY countryChanged)
    Q_PROPERTY(QString language
               READ language
               WRITE setLanguage
               NOTIFY languageChanged)
    Q_PROPERTY(QUrl source
               READ source
               WRITE setSource
               NOTIFY sourceChanged)
    Q_PROPERTY(bool favourite
               READ favourite
               WRITE setFavourite
               NOTIFY favouriteChanged)
    Q_PROPERTY(qint64 lastPlayed
               READ lastPlayed
               WRITE setLastPlayed
               NOTIFY lastPlayedChanged)
    Q_PROPERTY(QString lastPlayedString
               READ lastPlayedString
               NOTIFY lastPlayedChanged)

public:
    explicit Station(QObject *parent = 0);
    explicit Station(Stations::StationType type,
                     Services::RadioService service,
                     const QString &id,
                     const QString &title,
                     const QString &description,
                     const QUrl &logo,
                     const QString &genre,
                     const QString &country,
                     const QString &language,
                     const QUrl &source,
                     bool favourite,
                     qint64 lastPlayed,
                     QObject *parent = 0);
    explicit Station(Station *station, QObject *parent = 0);
    ~Station();

    Q_INVOKABLE void loadStation(Station *station);

    Stations::StationType stationType() const;
    void setStationType(Stations::StationType type);

    Services::RadioService service() const;
    void setService(Services::RadioService service);

    QString id() const;
    void setId(const QString &id);

    QString title() const;
    void setTitle(const QString &title);

    QString description() const;
    void setDescription(const QString &description);

    QUrl logo() const;
    void setLogo(const QUrl &url);

    QString genre() const;
    void setGenre(const QString &genre);

    QString country() const;
    void setCountry(const QString &country);

    QString language() const;
    void setLanguage(const QString &language);

    QUrl source() const;
    void setSource(const QUrl &source);

    bool favourite() const;
    void setFavourite(bool favourite);

    qint64 lastPlayed() const;
    void setLastPlayed(qint64 lastPlayed);

    QString lastPlayedString() const;

public slots:
    void clear();

private:
    void setLastPlayedString();

private slots:
    void onStationUpdated(const QString &id, const QVariantMap &properties);
    void onStationUpdated(const QString &id, const QString &property, const QVariant &value);
    
signals:
    void stationTypeChanged(Stations::StationType type);
    void serviceChanged(Services::RadioService service);
    void idChanged(const QString &id);
    void titleChanged(const QString &title);
    void descriptionChanged(const QString &description);
    void logoChanged(const QUrl &logo);
    void genreChanged(const QString &genre);
    void countryChanged(const QString &country);
    void languageChanged(const QString &language);
    void sourceChanged(const QUrl &source);
    void favouriteChanged(bool favourite);
    void lastPlayedChanged(qint64 lastPlayed);
    void stationUpdated(const QString &id, const QVariantMap &properties);
    void stationUpdated(const QString &id, const QString &property, const QVariant &value);

private:
    Stations::StationType m_stationType;
    Services::RadioService m_service;
    QString m_id;
    QString m_title;
    QString m_description;
    QUrl m_logo;
    QString m_genre;
    QString m_country;
    QString m_language;
    QUrl m_source;
    bool m_favourite;
    qint64 m_lastPlayed;
    QString m_lastPlayedString;
};

#endif // STATION_H
