/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QDateTime>
#include "station.h"
#include "database.h"

Station::Station(QObject *parent) :
    QObject(parent),
    m_stationType(Stations::Radio),
    m_service(Services::NoService),
    m_title(tr("Unknown title")),
    m_description(tr("No description")),
    m_genre(tr("Unknown genre")),
    m_country(tr("Unknown country")),
    m_language(tr("Unknown language")),
    m_favourite(false),
    m_lastPlayed(0)
{
    this->setLastPlayedString();
    this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
    this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
}

Station::Station(Stations::StationType type,
                 Services::RadioService service,
                 const QString &id,
                 const QString &title,
                 const QString &description,
                 const QUrl &logo,
                 const QString &genre,
                 const QString &country,
                 const QString &language,
                 const QUrl &source,
                 bool favourite,
                 qint64 lastPlayed,
                 QObject *parent) :
    QObject(parent),
    m_stationType(type),
    m_service(service),
    m_id(id),
    m_title(title),
    m_description(description),
    m_logo(logo),
    m_genre(genre),
    m_country(country),
    m_language(language),
    m_source(source),
    m_favourite(favourite),
    m_lastPlayed(lastPlayed)
{
    this->setLastPlayedString();

    switch (type) {
    case Stations::Podcast:
        this->connect(Database::instance(), SIGNAL(podcastUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
        this->connect(Database::instance(), SIGNAL(podcastUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
        return;
    default:
        this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
        this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
        return;
    }
}

Station::Station(Station *station, QObject *parent) :
    QObject(parent),
    m_stationType(station->stationType()),
    m_service(station->service()),
    m_id(station->id()),
    m_title(station->title()),
    m_description(station->description()),
    m_logo(station->logo()),
    m_genre(station->genre()),
    m_country(station->country()),
    m_language(station->language()),
    m_source(station->source()),
    m_favourite(station->favourite()),
    m_lastPlayed(station->lastPlayed()),
    m_lastPlayedString(station->lastPlayedString())
{
    switch (this->stationType()) {
    case Stations::Podcast:
        this->connect(Database::instance(), SIGNAL(podcastUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
        this->connect(Database::instance(), SIGNAL(podcastUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
        return;
    default:
        this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
        this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
        return;
    }
}

Station::~Station() {}

void Station::loadStation(Station *station) {
    this->setStationType(station->stationType());
    this->setService(station->service());
    this->setId(station->id());
    this->setTitle(station->title());
    this->setDescription(station->description());
    this->setLogo(station->logo());
    this->setGenre(station->genre());
    this->setCountry(station->country());
    this->setLanguage(station->language());
    this->setSource(station->source());
    this->setFavourite(station->favourite());
    this->setLastPlayed(station->lastPlayed());
}

Stations::StationType Station::stationType() const {
    return m_stationType;
}

void Station::setStationType(Stations::StationType type) {
    if (type != this->stationType()) {
        m_stationType = type;
        emit stationTypeChanged(type);

        switch (type) {
        case Stations::Podcast:
            this->connect(Database::instance(), SIGNAL(podcastUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
            this->connect(Database::instance(), SIGNAL(podcastUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
            this->disconnect(Database::instance(), SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
            this->disconnect(Database::instance(), SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
            return;
        default:
            this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)), Qt::UniqueConnection);
            this->connect(Database::instance(), SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)), Qt::UniqueConnection);
            this->disconnect(Database::instance(), SIGNAL(podcastUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
            this->disconnect(Database::instance(), SIGNAL(podcastUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
            return;
        }
    }
}

Services::RadioService Station::service() const {
    return m_service;
}

void Station::setService(Services::RadioService service) {
    if (service != this->service()) {
        m_service = service;
        emit serviceChanged(service);
    }
}

QString Station::id() const {
    return m_id;
}

void Station::setId(const QString &id) {
    if (id != this->id()) {
        m_id = id;
        emit idChanged(id);
    }
}

QString Station::title() const {
    return m_title;
}

void Station::setTitle(const QString &title) {
    if (title != this->title()) {
        m_title = title;
        emit titleChanged(title);
    }
}

QString Station::description() const {
    return m_description;
}

void Station::setDescription(const QString &description) {
    if (description != this->description()) {
        m_description = description;
        emit descriptionChanged(description);
    }
}

QUrl Station::logo() const {
    return m_logo;
}

void Station::setLogo(const QUrl &url) {
    if (url != this->logo()) {
        m_logo = url;
        emit logoChanged(url);
    }
}

QString Station::genre() const {
    return m_genre;
}

void Station::setGenre(const QString &genre) {
    if (genre != this->genre()) {
        m_genre = genre;
        emit genreChanged(genre);
    }
}

QString Station::country() const {
    return m_country;
}

void Station::setCountry(const QString &country) {
    if (country != this->country()) {
        m_country = country;
        emit countryChanged(country);
    }
}

QString Station::language() const {
    return m_language;
}

void Station::setLanguage(const QString &language) {
    if (language != this->language()) {
        m_language = language;
        emit languageChanged(language);
    }
}

QUrl Station::source() const {
    return m_source;
}

void Station::setSource(const QUrl &source) {
    if (source != this->source()) {
        m_source = source;
        emit sourceChanged(source);
    }
}

bool Station::favourite() const {
    return m_favourite;
}

void Station::setFavourite(bool favourite) {
    if (favourite != this->favourite()) {
        m_favourite = favourite;
        emit favouriteChanged(favourite);
    }
}

qint64 Station::lastPlayed() const {
    return m_lastPlayed;
}

void Station::setLastPlayed(qint64 lastPlayed) {
    if (lastPlayed != this->lastPlayed()) {
        m_lastPlayed = lastPlayed;
        this->setLastPlayedString();
        emit lastPlayedChanged(lastPlayed);
    }
}

QString Station::lastPlayedString() const {
    return m_lastPlayedString;
}

void Station::setLastPlayedString() {
    if (!this->lastPlayed()) {
        m_lastPlayedString = tr("Never");
    }
    else {
        QDate date = QDateTime::fromMSecsSinceEpoch(this->lastPlayed()).date();
        QDate currentDate = QDate::currentDate();

        if (date == currentDate) {
            m_lastPlayedString = tr("Today");
        }
        else if (date.daysTo(currentDate) == 1) {
            m_lastPlayedString = tr("Yesterday");
        }
        else {
            m_lastPlayedString = date.toString();
        }
    }
}

void Station::clear() {
    this->setService(Services::NoService);
    this->setId(QString());
    this->setTitle(QString());
    this->setDescription(QString());
    this->setLogo(QUrl());
    this->setGenre(QString());
    this->setCountry(QString());
    this->setLanguage(QString());
    this->setSource(QUrl());
    this->setFavourite(false);
    this->setLastPlayed(0);
    this->disconnect(Database::instance(), 0, this, 0);
}

void Station::onStationUpdated(const QString &id, const QVariantMap &properties) {
    if (id == this->id()) {
        foreach (QString key, properties.uniqueKeys()) {
            this->setProperty(key.toUtf8(), properties.value(key));
        }

        emit stationUpdated(id, properties);
    }
}

void Station::onStationUpdated(const QString &id, const QString &property, const QVariant &value) {
    if (id == this->id()) {
        this->setProperty(property.toUtf8(), value);
        emit stationUpdated(id, property, value);
    }
}
