/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.models 1.0
import com.cuteradio.items 1.0


MyPage {
    id: root

    function searchStations(query) {
        appWindow.pageStack.push(Qt.resolvedUrl("StationListPage.qml"), { title: qsTr("Search") + " ('" + query + "')" }).searchStations(query);
        SearchModel.addSearch(query);
    }

    tools: ToolBarLayout {

        BackToolButton {
            id: backButton
        }

        NowPlayingButton {}

        MyToolButton {
            iconSource: "toolbar-view-menu"
            toolTipText: qsTr("Menu")
            onClicked: menu.open()
        }
    }

    Menu {
        id: menu

        MenuLayout {

            MenuItem {
                text: qsTr("Add station")
                onClicked: {
                    loader.sourceComponent = addStationDialog;
                    loader.item.open();
                }
            }

            MenuItem {
                text: qsTr("Play station")
                onClicked: {
                    loader.sourceComponent = playStationDialog;
                    loader.item.open();
                }
            }

            MenuItem {
                text: qsTr("Settings")
                onClicked: appWindow.pageStack.push(Qt.resolvedUrl("SettingsPage.qml"))
            }

            MenuItem {
                text: qsTr("About")
                onClicked: {
                    loader.sourceComponent = aboutDialog;
                    loader.item.open();
                }
            }
        }

        onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
    }

    MySearchBox {
        id: searchBox

        z: Number.MAX_VALUE
        placeHolderText: qsTr("Search")
        anchors {
            top: parent.top
            left: parent.left
            right: parent.right
        }
        onSearchTextChanged: {
            SearchModel.query = searchText;

            if (!searchText) {
                view.forceActiveFocus();
            }
        }
        onAccepted: {
            if (searchText) {
                var query = searchText;
                searchStations(query);
                searchText = "";
            }
        }
    }

    SearchSuggestionBox {
        id: suggestionBox

        z: Number.MAX_VALUE
        width: parent.width - platformStyle.paddingLarge * 2
        anchors {
            left: parent.left
            top: searchBox.bottom
            margins: platformStyle.paddingLarge
        }
        opacity: (searchBox.searchText != "") && (count > 0) ? 1 : 0
        visible: opacity > 0
        onSuggestionChosen: {
            searchStations(suggestion);
            searchBox.searchText = "";
            view.focus = true;
        }
    }

    MyListView {
        id: view

        anchors {
            left: parent.left
            right: parent.right
            top: searchBox.bottom
            bottom: parent.bottom
        }
        enabled: (!inputContext.softwareInputPanelVisible) && (!inputContext.customSoftwareInputPanelVisible) && (!suggestionBox.visible)
        model: HomescreenModel {
            id: homescreenModel
        }
        delegate: DrillDownDelegate {
            onClicked: {
                switch (index) {
                case 0:
                    appWindow.pageStack.push(Qt.resolvedUrl("GenreListPage.qml"));
                    return;
                case 1:
                    appWindow.pageStack.push(Qt.resolvedUrl("CountryListPage.qml"));
                    return;
                case 2:
                    appWindow.pageStack.push(Qt.resolvedUrl("LanguageListPage.qml"));
                    return;
                case 3:
                    appWindow.pageStack.push(Qt.resolvedUrl("StationListPage.qml"), { title: name }).showRecentlyPlayedStations();
                    return;
                case 4:
                    appWindow.pageStack.push(Qt.resolvedUrl("StationListPage.qml"), { title: name }).showFavouriteStations();
                    return;
                case 5:
                    appWindow.pageStack.push(Qt.resolvedUrl("TuneInMainPage.qml"));
                    return;
                default:
                    return
                }
            }
        }

        Keys.forwardTo: searchBox
    }

    MyScrollBar {
        flickableItem: view
    }

    Loader {
        id: loader
    }

    Component {
        id: addStationDialog

        AddStationDialog {
            onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
        }
    }

    Component {
        id: playStationDialog

        PlayStationDialog {
            onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
        }
    }

    Component {
        id: aboutDialog

        AboutDialog {
            onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
        }
    }
}
