/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QCheckBox>
#include <QLineEdit>
#include <QGridLayout>
#include <QScrollArea>
#include <QLabel>
#include <QDialogButtonBox>
#include <QIntValidator>
#include "networkproxydialog.h"
#include "valueselector.h"
#include "../base/settings.h"
#include "../base/selectionmodels.h"

NetworkProxyDialog::NetworkProxyDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_proxyCheckBox(new QCheckBox(tr("Use network proxy"), this)),
    m_proxyWidget(new QWidget(this)),
    m_proxyTypeSelector(new ValueSelector(tr("Proxy type"), this)),
    m_hostEdit(new QLineEdit(this)),
    m_portEdit(new QLineEdit(this)),
    m_userEdit(new QLineEdit(this)),
    m_passEdit(new QLineEdit(this)),
    m_scrollArea(new QScrollArea(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Save, Qt::Vertical, this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Network proxy"));
    this->setAttribute(Qt::WA_DeleteOnClose, false);

    m_proxyTypeSelector->setModel(new NetworkProxyTypeModel(m_proxyTypeSelector));
    m_hostEdit->setMinimumWidth(380);
    m_portEdit->setValidator(new QIntValidator(0, 100000, this));
    m_passEdit->setEchoMode(QLineEdit::Password);

    QGridLayout *proxyGrid = new QGridLayout(m_proxyWidget);
    proxyGrid->setContentsMargins(0, 0, 0, 0);
    proxyGrid->addWidget(m_proxyTypeSelector, 0, 0, 1, 2);
    proxyGrid->addWidget(new QLabel(tr("Host"), this), 1, 0, 1, 1);
    proxyGrid->addWidget(new QLabel(tr("Port"), this), 1, 1, 1, 1);
    proxyGrid->addWidget(m_hostEdit, 2, 0, 1, 1);
    proxyGrid->addWidget(m_portEdit, 2, 1, 1, 1);
    proxyGrid->addWidget(new QLabel(tr("Username"), this), 3, 0, 1, 2);
    proxyGrid->addWidget(m_userEdit, 4, 0, 1, 2);
    proxyGrid->addWidget(new QLabel(tr("Password"), this), 5, 0, 1, 2);
    proxyGrid->addWidget(m_passEdit, 6, 0, 1, 2);

    QWidget *scrollWidget = new QWidget(this);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_proxyCheckBox);
    vbox->addWidget(m_proxyWidget);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setWidget(scrollWidget);

    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));
    this->connect(m_proxyCheckBox, SIGNAL(toggled(bool)), m_proxyWidget, SLOT(setEnabled(bool)));

    this->loadSettings();
    this->onOrientationChanged();
}

void NetworkProxyDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_hostEdit->setMinimumWidth(380);
    m_buttonBox->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1);
    }
}

void NetworkProxyDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_hostEdit->setMinimumWidth(280);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
}

void NetworkProxyDialog::loadSettings() {
    m_proxyCheckBox->setChecked(Settings::instance()->networkProxyEnabled());
    m_proxyTypeSelector->setValue(Settings::instance()->networkProxyType());
    m_hostEdit->setText(Settings::instance()->networkProxyHostName());
    m_portEdit->setText(QString::number(Settings::instance()->networkProxyPort()));
    m_userEdit->setText(Settings::instance()->networkProxyUser());
    m_passEdit->setText(Settings::instance()->networkProxyPassword());
    m_proxyWidget->setEnabled(m_proxyCheckBox->isChecked());
}

void NetworkProxyDialog::saveSettings() {
    Settings::instance()->setNetworkProxyEnabled(m_proxyCheckBox->isChecked());
    Settings::instance()->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(m_proxyTypeSelector->currentValue().toInt()));
    Settings::instance()->setNetworkProxyHostName(m_hostEdit->text());
    Settings::instance()->setNetworkProxyPort(m_portEdit->text().toUInt());
    Settings::instance()->setNetworkProxyUser(m_userEdit->text());
    Settings::instance()->setNetworkProxyPassword(m_passEdit->text());
    Settings::instance()->setNetworkProxy();

    this->accept();
}
