/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.models 1.0

MyPage {
    id: root

    title: qsTr("Settings")
    tools: ToolBarLayout {

        BackToolButton {
            onClicked: Settings.setNetworkProxy()
        }
    }

    KeyNavFlickable {
        id: flicker

        anchors.fill: parent
        contentHeight: column.height + platformStyle.paddingLarge

        Column {
            id: column

            anchors {
                left: parent.left
                right: parent.right
                top: parent.top
            }
            spacing: platformStyle.paddingLarge

            SelectionItem {
                width: parent.width
                title: qsTr("Screen orientation")
                model: ScreenOrientationModel {}
                initialValue: Settings.screenOrientation
                onValueChosen: Settings.screenOrientation = value
            }

            SelectionItem {
                width: parent.width
                title: qsTr("Language")
                model: LanguageModel {}
                initialValue: Settings.language
                onValueChosen: Settings.language = value
            }

            Row {
                x: platformStyle.paddingLarge
                spacing: platformStyle.paddingLarge

                Label {
                    width: column.width - 60 - platformStyle.paddingLarge * 3
                    height: sleepTimerField.height
                    verticalAlignment: Text.AlignVCenter
                    text: qsTr("Sleep timer duration (mins)")
                }

                MyTextField {
                    id: sleepTimerField

                    width: 60
                    inputMethodHints: Qt.ImhDigitsOnly
                    maximumLength: 3
                    validator: IntValidator {
                        bottom: 1
                        top: 999
                    }
                    text: Settings.sleepTimerDuration
                    onTextChanged: if (acceptableInput) Settings.sleepTimerDuration = parseInt(text);
                    onAccepted: {
                        closeSoftwareInputPanel();
                        flicker.forceActiveFocus();
                    }
                }
            }

            MySwitch {
                id: proxySwitch

                width: parent.width
                text: qsTr("Use network proxy")
                visible: !inputContext.visible
                Component.onCompleted: checked = Settings.networkProxyEnabled
                onCheckedChanged: Settings.networkProxyEnabled = checked
            }

            Flow {
                id: flow

                x: platformStyle.paddingLarge
                width: parent.width - platformStyle.paddingLarge * 2
                opacity: Settings.networkProxyEnabled ? 1 : 0
                visible: opacity > 0
                spacing: platformStyle.paddingLarge
                onVisibleChanged: if ((!visible) && (flicker.currentIndex > 2)) flicker.currentIndex = 2;

                Behavior on opacity { NumberAnimation { duration: 200 } }

                Label {
                    width: inputContext.visible ? parent.width : hostEdit.width
                    text: qsTr("Host")
                    visible: hostEdit.visible
                }

                Label {
                    width: inputContext.visible ? parent.width : portEdit.width
                    text: qsTr("Port")
                    visible: portEdit.visible
                }

                TextField {
                    id: hostEdit

                    width: parent.width - portEdit.width - platformStyle.paddingLarge
                    inputMethodHints: Qt.ImhNoPredictiveText | Qt.ImhNoAutoUppercase
                    visible: (!inputContext.visible) || (focus)
                    onTextChanged: Settings.networkProxyHostName = text
                    Keys.onReturnPressed: portEdit.focus = true
                    Component.onCompleted: text = Settings.networkProxyHostName
                }

                TextField {
                    id: portEdit

                    width: 200
                    inputMethodHints: Qt.ImhDigitsOnly
                    visible: (!inputContext.visible) || (focus)
                    onTextChanged: Settings.networkProxyPort = parseInt(text)
                    Component.onCompleted: text = Settings.networkProxyPort
                }

                Label {
                    text: qsTr("Username")
                    visible: usernameEdit.visible
                }

                TextField {
                    id: usernameEdit

                    x: platformStyle.paddingLarge
                    width: parent.width
                    inputMethodHints: Qt.ImhNoPredictiveText | Qt.ImhNoAutoUppercase
                    onTextChanged: Settings.networkProxyUser = text
                    visible: (!inputContext.visible) || (focus)
                    Keys.onReturnPressed: passwordEdit.focus = true
                    Component.onCompleted: text = Settings.networkProxyUser
                }

                Label {
                    width: parent.width
                    text: qsTr("Password")
                    visible: passwordEdit.visible
                }

                TextField {
                    id: passwordEdit

                    x: platformStyle.paddingLarge
                    width: parent.width
                    echoMode: TextInput.Password
                    inputMethodHints: Qt.ImhNoPredictiveText | Qt.ImhNoAutoUppercase
                    visible: (!inputContext.visible) || (focus)
                    onTextChanged: Settings.networkProxyPassword = text
                    Component.onCompleted: text = Settings.networkProxyPassword
                }
            }
        }
    }

    MyScrollBar {
        flickableItem: flicker
    }
}
