/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLabel>
#include <QVBoxLayout>
#include <QMaemo5Style>
#include "browselistwindow.h"
#include "stationlistwindow.h"
#include "browselistdelegate.h"
#include "listview.h"
#include "filterbox.h"
#include "nowplayingbutton.h"
#include "../base/pairlistmodel.h"

BrowseListWindow::BrowseListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new PairListModel(this)),
    m_filterBox(new FilterBox(this)),
    m_noResultsLabel(new QLabel(this))
{
    this->setCentralWidget(m_noResultsLabel);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new BrowseListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_filterBox->hide();

    m_vbox = new QVBoxLayout(m_noResultsLabel);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), this, SLOT(onScrollingStopped()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(cleared()), NowPlayingButton::instance(), SLOT(poke()));
}

void BrowseListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void BrowseListWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (m_model->rowCount() > 0) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        NowPlayingButton::instance()->hide();
        m_filterBox->show();
        m_filterBox->setFocus(Qt::OtherFocusReason);
        m_filterBox->setFilterText(event->text());
    }
}

void BrowseListWindow::showGenres() {
    this->setWindowTitle(tr("Stations by genre"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No genres found")));
    m_model->showGenres();
}

void BrowseListWindow::showCountries() {
    this->setWindowTitle(tr("Stations by country"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No countries found")));
    m_model->showCountries();
}

void BrowseListWindow::showLanguages() {
    this->setWindowTitle(tr("Stations by language"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No languages found")));
    m_model->showLanguages();
}

void BrowseListWindow::clear() {
    m_model->clear();
}

void BrowseListWindow::reload() {
    m_model->reload();
}

void BrowseListWindow::onScrollingStopped() {
    if (m_filterBox->isHidden()) {
        NowPlayingButton::instance()->poke();
    }
}

void BrowseListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_view->setHidden((!loading) && (m_model->rowCount() == 0));
}

void BrowseListWindow::onCountChanged(int count) {
    m_view->setHidden((count == 0) && (!m_model->loading()));
}

void BrowseListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        StationListWindow *window = new StationListWindow(this);

        switch (m_model->queryType()) {
        case Queries::Genres:
            window->showStationsByGenre(index.data().toString());
            break;
        case Queries::Countries:
            window->showStationsByCountry(index.data().toString());
            break;
        default:
            window->showStationsByLanguage(index.data().toString());
            break;
        }

        window->show();
    }
}
