/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.items 1.0

MySheet {
    id: root

    function loadStation(stationItem) {
        station.loadStation(stationItem);
    }

    function addStation() {
        Database.asyncAddStation(titleField.text,
                            !descriptionField.text ? qsTr("No description") : descriptionField.text,
                            !genreField.text ? qsTr("Unknown genre") : genreField.text,
                            !countryField.text ? qsTr("Unknown country") : countryField.text,
                            !languageField.text ? qsTr("Unknown language") : languageField.text,
                            sourceField.text,
                            favouriteCheckBox.checked ? 1 : 0);
        clear();
    }

    function updateStation() {
        var properties = {};

        if (titleField.text != station.title) {
            properties["title"] = titleField.text;
        }
        if (descriptionField.text != station.description) {
            properties["description"] = !descriptionField.text ? qsTr("No description") : descriptionField.text
        }
        if (genreField.text != station.genre) {
            properties["genre"] = !genreField.text ? qsTr("Unknown genre") : genreField.text;
        }
        if (countryField.text != station.country) {
            properties["country"] = !countryField.text ? qsTr("Unknown country") : countryField.text;
        }
        if (languageField.text != station.language) {
            properties["language"] = !languageField.text ? qsTr("Unknown language") : languageField.text
        }
        if (sourceField.text != station.source) {
            properties["source"] = sourceField.text;
        }
        if (favouriteCheckBox.checked != station.favourite) {
            properties["favourite"] = favouriteCheckBox.checked ? 1 : 0;
        }

        if (station.isRecording) {
            Database.asyncUpdateRecording(station.id, properties);
        }
        else {
            Database.asyncUpdateStation(station.id, properties);
        }

        clear();
    }

    function clear() {
        if (station.id) {
            station.clear();
        }
        else {
            titleField.text = "";
            descriptionField.text = "";
            genreField.text = "";
            countryField.text = "";
            languageField.text = "";
            sourceField.text = "";
            favouriteCheckBox.checked = false;
        }
    }

    Station {
        id: station
    }

    acceptButtonText: (titleField.text) && (sourceField.text) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent
        clip: true

        KeyNavFlickable {
            id: flicker

            anchors {
                fill: parent
                margins: platformStyle.paddingLarge
            }
            contentHeight: inputContext.visible ? height : column.height + platformStyle.paddingLarge

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: platformStyle.paddingLarge

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Title")
                    visible: titleField.visible
                }

                MyTextField {
                    id: titleField

                    width: parent.width
                    text: station.title
                    visible: (focus) || (!inputContext.visible)
                    onAccepted: descriptionField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Description")
                    visible: descriptionField.visible
                }

                TextArea {
                    id: descriptionField

                    width: parent.width
                    text: station.description
                    visible: (focus) || (!inputContext.visible)
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Genre")
                    visible: genreField.visible
                }

                MyTextField {
                    id: genreField

                    width: parent.width
                    text: station.genre
                    visible: (focus) || (!inputContext.visible)
                    onAccepted: countryField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Country")
                    visible: countryField.visible
                }

                MyTextField {
                    id: countryField

                    width: parent.width
                    text: station.country
                    visible: (focus) || (!inputContext.visible)
                    onAccepted: languageField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Language")
                    visible: languageField.visible
                }

                MyTextField {
                    id: languageField

                    width: parent.width
                    text: station.language
                    visible: (focus) || (!inputContext.visible)
                    onAccepted: sourceField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                    visible: sourceField.visible
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    text: station.source
                    visible: (!station.isRecording) && ((focus) || (!inputContext.visible))
                    onAccepted: closeSoftwareInputPanel()
                }

                MyCheckBox {
                    id: favouriteCheckBox

                    text: qsTr("Favourite?")
                    checked: station.favourite
                    visible: !inputContext.visible
                }
            }
        }

        MyScrollBar {
            flickableItem: flicker
        }
    }

    onAccepted: station.id ? updateStation() : addStation()
    onRejected: clear()
}
