/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QStringListModel>
#include <QSettings>
#include "stringlistmodel.h"

StringListModel::StringListModel(QObject *parent) :
    QSortFilterProxyModel(parent),
    m_model(new QStringListModel(this))
  #if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
  ,m_highlight(QSettings().value("Appearance/activeColor", "#0881cb").toString())
  #endif
{
    m_roleNames[NameRole] = "name";
    m_roleNames[QueryMatchRole] = "queryMatch";
    m_roleNames[SectionRole] = "section";
    this->setSourceModel(m_model);
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->setFilterRole(NameRole);
    this->setDynamicSortFilter(true);
}

StringListModel::~StringListModel() {}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> StringListModel::roleNames() const {
    return m_roleNames;
}
#endif

QVariant StringListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case QueryMatchRole:
        return this->queryMatch(QSortFilterProxyModel::data(index, this->filterRole()).toString());
    case SectionRole:
        return QSortFilterProxyModel::data(index, NameRole).toString().left(1).toUpper();
    default:
        return QSortFilterProxyModel::data(index, role);
    }
}

QVariant StringListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

QString StringListModel::query() const {
    return m_query;
}

void StringListModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit queryChanged(query);
    }
}

QString StringListModel::queryMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#else
        str.replace(re, "<b>" + re.cap() + "</b>");
#endif
    }

    return str;
}

#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
QString StringListModel::highlightColor() const {
    return m_highlight;
}

void StringListModel::setHighlightColor(const QString &color) {
    if (color != this->highlightColor()) {
        m_highlight = color;
        emit highlightColorChanged(color);
    }
}
#endif

void StringListModel::setStrings(const QStringList &strings) {
    m_model->setStringList(strings);
}

void StringListModel::addString(const QString &string) {
    m_model->insertRow(m_model->rowCount());
    m_model->setData(m_model->index(m_model->rowCount()), string, NameRole);
    emit countChanged(this->rowCount());
}

void StringListModel::removeString(const QModelIndex &index) {
    this->removeString(index.row());
}

void StringListModel::removeString(int row) {
    m_model->removeRow(row);
    emit countChanged(this->rowCount());
}

void StringListModel::clear() {
    this->beginResetModel();
    m_model->setStringList(QStringList());
    this->endResetModel();
    emit countChanged(this->rowCount());
}
