/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 *  Copyright (C) 2009 Artem Garmash. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Artem Garmash <artemgarmash@gmail.com>
 *
 */

#include "config.h"
#include "el-home-applet.h"
#include <libintl.h>
#include <hildon/hildon.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <sqlite3.h>
#include <string.h>
#include <libosso-abook/osso-abook-init.h>
#include <libosso-abook/osso-abook-aggregator.h>
#include <libosso-abook/osso-abook-contact.h>
#include <libosso-abook/osso-abook-waitable.h>
#include <libosso-abook/osso-abook-presence.h>
#include <libosso-abook/osso-abook-touch-contact-starter.h>
#include <libosso-abook/osso-abook-temporary-contact-dialog.h>
#include <libosso-abook/osso-abook-account-manager.h>

#include <telepathy-glib/interfaces.h>
#include <telepathy-glib/dbus.h>
#include <rtcom-telepathy-glib/extensions.h>

#define EL_HOME_APPLET_GET_PRIVATE(obj) ( \
        G_TYPE_INSTANCE_GET_PRIVATE (obj, \
                EL_TYPE_HOME_APPLET, ELHomeAppletPrivate))

#define BOX_WIDTH 352
#define BOX_HEIGHT 284

#define CONTENT_OFFSET_X HILDON_MARGIN_HALF
#define CONTENT_OFFSET_Y_TOP 4*HILDON_MARGIN_HALF
#define CONTENT_OFFSET_Y_BOTTOM HILDON_MARGIN_HALF
#define C_WIDTH (BOX_WIDTH - 2*CONTENT_OFFSET_X)
#define C_HEIGHT (BOX_HEIGHT - (CONTENT_OFFSET_Y_TOP + CONTENT_OFFSET_Y_BOTTOM))

#define HEADER_HEIGHT 48
#define FOOTER_HEIGHT 24
#define FOOTER_HEIGHT_PRESS FOOTER_HEIGHT*2 /* approx, used only for checking clicks */
#define FOOTER_WIDTH C_WIDTH/4
#define FOOTER_WIDTH_PRESS (FOOTER_WIDTH + FOOTER_WIDTH/2) /* approx, used only for checking clicks, bigger than controls */

#define MESSAGE_HEIGHT (C_HEIGHT - HEADER_HEIGHT - FOOTER_HEIGHT)
#define MESSAGE_WIDTH (C_WIDTH - 2*HILDON_MARGIN_DEFAULT)

#define AVATAR_SIZE HILDON_ICON_PIXEL_SIZE_THUMB

#define AVATAR_X (C_WIDTH - AVATAR_SIZE - HILDON_MARGIN_DEFAULT)
#define AVATAR_Y 3*HILDON_MARGIN_HALF

#define BOX_RADIOUS 20

#define SCROLL_PERIOD 100 /* ms */
#define SCROLL_STEP 1 /* pixel */
#define TEXT_Y_OFFSET (HEADER_HEIGHT + HILDON_MARGIN_HALF)

#define NOTIFICATION_UI_DBUS_NAME     "org.freedesktop.Telepathy.Client.NotificationUI"
#define NOTIFICATION_UI_DBUS_PATH     "/org/freedesktop/Telepathy/Client/NotificationUI"
#define NOTIFICATION_UI_DBUS_IFACE    "com.nokia.RtcomNotificationUi"

#define CONVERSATIONS_UI_DBUS_NAME     "com.nokia.MessagingUI"
#define CONVERSATIONS_UI_DBUS_PATH     "/com/nokia/MessagingUI"
#define CONVERSATIONS_UI_DBUS_IFACE    "com.nokia.MessagingUI"

static const gchar *conv_services[] = {"RTCOM_EL_SERVICE_SMS",
                                       "RTCOM_EL_SERVICE_CHAT",
                                       NULL};
static const gchar *conv_event_types[] = {"RTCOM_EL_EVENTTYPE_SMS_INBOUND",
                                          "RTCOM_EL_EVENTTYPE_CHAT_INBOUND",
                                          NULL};

typedef enum {
        SELECTED_NONE,
        SELECTED_HEADER,
        SELECTED_BODY,
        SELECTED_FOOTER
} WidgetActiveSelection;

struct _ELHomeAppletPrivate
{
        RTComEl *eventlogger;

        GtkWidget *sender;
        GtkWidget *icon;
        GtkWidget *unread;
        GtkWidget *received;
        GtkWidget *cut_message;

        /* empty view*/
        GtkWidget *empty;
        GtkWidget *sms_total;
        GtkWidget *chat_total;

        gchar *message;
        gint event_id;

        WidgetActiveSelection active;

        guint unread_count;

        struct {
                float red;
                float green;
                float blue;
        } active_color;
        PangoFontDescription *font_desc;

        GdkPixbuf *avatar_pixbuf;
        GdkPixbuf *presence_pixbuf;

        guint idle_id;

        cairo_surface_t *message_surface;

        gboolean scroll_on_click;
        gint scroll_offset;
        gint hidden_message_height;
        guint scroll_anim_id;

        OssoABookRoster *aggregator;
        OssoABookWaitableClosure *aggregator_ready_closure;
        gchar *contact_id;
        gchar *remote_id;
        gchar *local_id;
        gchar *group_uid;
        OssoABookContact *contact;
};

HD_DEFINE_PLUGIN_MODULE (ELHomeApplet, el_home_applet, HD_TYPE_HOME_PLUGIN_ITEM);

const gchar* g_module_check_init (GModule *module);
const gchar*
g_module_check_init (GModule *module)
{
	g_module_make_resident (module);
	return NULL;
}

static void
el_home_applet_class_finalize (ELHomeAppletClass *klass)
{
}

static void
el_home_applet_realize (GtkWidget *widget)
{
        GdkScreen *screen;

        screen = gtk_widget_get_screen (widget);
        gtk_widget_set_colormap (widget,
                                 gdk_screen_get_rgba_colormap (screen));

        gtk_widget_set_app_paintable (widget,
                                      TRUE);

        GTK_WIDGET_CLASS (el_home_applet_parent_class)->realize (widget);
}

enum {
        ROUND_CORNER_TL = 1,
        ROUND_CORNER_TR = 1<<1,
        ROUND_CORNER_BL = 1<<2,
        ROUND_CORNER_BR = 1<<3,
        ROUND_CORNER_ALL = ROUND_CORNER_TL | ROUND_CORNER_TR |
                           ROUND_CORNER_BL | ROUND_CORNER_BR
};

/**
 * Draw rectangle with optional round corners.
 *
 * @x
 * @y
 * @w width
 * @h height
 * @r round corner radious
 * @round_corners define which corners draw round, ROUND_CORNER_TL,
 *                ROUND_CORNER_TR, ROUND_CORNER_BL, ROUND_CORNER_BR
 */
static void
rounded_rectangle (cairo_t *cr,
                   double x,
                   double y,
                   double w,
                   double h,
                   double r,
                   guint round_corners)
{
        if (round_corners & ROUND_CORNER_TL)
                cairo_move_to (cr, x + r, y);
        else
                cairo_move_to (cr, x, y);

        if (round_corners & ROUND_CORNER_TR) {
                cairo_line_to (cr, x + w - r, y);
                cairo_rel_curve_to (cr,
                                    r, 0,
                                    r, 0,
                                    r, r);
        }
        else
                cairo_line_to (cr, x + w, y);

        if (round_corners & ROUND_CORNER_BR) {
                cairo_line_to (cr, x + w, y + h - r);
                cairo_rel_curve_to (cr,
                                    0, r,
                                    0, r,
                                    -r, r);
        }
        else
                cairo_line_to (cr, x + w, y + h);

        if (round_corners & ROUND_CORNER_BL) {
                cairo_line_to (cr, x + r, y + h);
                cairo_rel_curve_to (cr,
                                    -r, 0,
                                    -r, 0,
                                    -r, -r);
        }
        else
                cairo_line_to (cr, x, y + h);

        if (round_corners & ROUND_CORNER_TL) {
                cairo_line_to (cr, x, y + r);
                cairo_rel_curve_to (cr,
                                    0, -r,
                                    0, -r,
                                    r, -r);
        }
        else
                cairo_line_to (cr, x, y);
}

static cairo_surface_t*
draw_text (cairo_t              *cr,
           PangoFontDescription *desc,
           const gchar          *text,
           gint                  width,
           gint                 *height)
{
        PangoLayout *layout;
        PangoRectangle extent;

        cairo_surface_t *gdk_surface, *result_surface;
        cairo_t *msg_cr;

        /* Create a PangoLayout, set the font and text */
        layout = pango_cairo_create_layout (cr);
        pango_layout_set_text (layout,
                               text,
                               -1);
        pango_layout_set_font_description (layout, desc);

        pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);
        pango_layout_set_width (layout, PANGO_SCALE*width);

        pango_layout_get_pixel_extents (layout, NULL, &extent);
        *height = extent.height;

        gdk_surface = cairo_get_target (cr);
        result_surface = cairo_surface_create_similar
                (gdk_surface,
                 CAIRO_CONTENT_COLOR_ALPHA,
                 width,
                 extent.height);
        msg_cr = cairo_create (result_surface);

        pango_cairo_update_layout (msg_cr, layout);
        /* draw shadow */
        cairo_move_to (msg_cr, 1, 1);
        cairo_set_source_rgba (msg_cr, 0.2, 0.2, 0.2, 0.8);
        pango_cairo_show_layout (msg_cr, layout);

        /* draw fg */
        cairo_move_to (msg_cr, 0, 0);
        cairo_set_source_rgba (msg_cr, 1.0, 1.0, 1.0, 1.0);
        pango_cairo_show_layout (msg_cr, layout);

        cairo_destroy (msg_cr);
        g_object_unref (layout);

        return result_surface;
}

static gboolean
stop_scroll_anim (ELHomeAppletPrivate *priv)
{
        gboolean result = priv->scroll_anim_id > 0;

        if (result) {
                g_source_remove (priv->scroll_anim_id);
                priv->scroll_anim_id = 0;
                priv->scroll_on_click = FALSE;
                gtk_widget_hide (priv->cut_message);
        }

        return result;
}

static void
style_set_cb (GtkWidget *widget,
              GtkStyle  *previous_style,
              ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        GdkColor color;
        GtkStyle *font_style;

        font_style = gtk_rc_get_style_by_paths (gtk_widget_get_settings (widget),
                                                "SystemFont",
                                                NULL,
                                                G_TYPE_NONE);
        if (font_style && font_style->font_desc) {
                if (priv->font_desc)
                        pango_font_description_free (priv->font_desc);
                priv->font_desc = pango_font_description_copy (font_style->font_desc);
        }

        if (gtk_style_lookup_color (widget->style,
                                    "ActiveTextColor",
                                    &color)) {
                priv->active_color.red = color.red/(float)G_MAXUINT16;
                priv->active_color.green = color.green/(float)G_MAXUINT16;
                priv->active_color.blue = color.blue/(float)G_MAXUINT16;
        }
}

static void
reset_scroll (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (stop_scroll_anim (self->priv)) {
                priv->scroll_on_click = TRUE;/* priv->scroll_offset; */
                priv->scroll_offset = 0;
                if (priv->scroll_on_click)
                        gtk_widget_show (priv->cut_message);
        }
}

static void
notify_on_current_desktop (GObject      *object,
                           GParamSpec   *unused G_GNUC_UNUSED,
                           ELHomeApplet *self)
{
        gboolean on;

        g_object_get (object, "is-on-current-desktop", &on, NULL);
        if (!on) {
                reset_scroll (self);
                gtk_widget_queue_draw (GTK_WIDGET (self));
        }
}

static gboolean
expose_event (GtkWidget *self, GdkEventExpose *event)
{
        ELHomeAppletPrivate *priv = EL_HOME_APPLET(self)->priv;
        cairo_t *cr;
        cairo_pattern_t *grad;

        cr = gdk_cairo_create (self->window);
        gdk_cairo_region (cr, event->region);
        cairo_clip (cr);

        cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

        /* draw bound box */
        cairo_set_source_rgba (cr, 0.4f, 0.4f, 0.4f, 0.1f);
        cairo_set_line_width (cr, 3.0f);

        rounded_rectangle (cr,
                           CONTENT_OFFSET_X,
                           CONTENT_OFFSET_Y_TOP,
                           BOX_WIDTH - 2*CONTENT_OFFSET_X,
                           BOX_HEIGHT - (CONTENT_OFFSET_Y_TOP + CONTENT_OFFSET_Y_BOTTOM),
                           BOX_RADIOUS,
                           ROUND_CORNER_ALL);

        cairo_close_path (cr);
        cairo_stroke (cr);

        /* draw header */
        cairo_set_line_width (cr, 1.0f);

        cairo_translate (cr, CONTENT_OFFSET_X, CONTENT_OFFSET_Y_TOP);
        rounded_rectangle (cr,
                           0, 0,
                           C_WIDTH, HEADER_HEIGHT,
                           BOX_RADIOUS,
                           ROUND_CORNER_TL | ROUND_CORNER_TR);
        cairo_close_path (cr);

        switch (priv->active) {
        case SELECTED_HEADER:
                cairo_set_source_rgba (cr,
                                       priv->active_color.red,
                                       priv->active_color.green,
                                       priv->active_color.blue,
                                       0.8f);
                break;
        default:
                cairo_set_source_rgba (cr, 0.2f, 0.2f, 0.2f, 0.8f);
        }

        cairo_fill (cr);

        cairo_move_to (cr, 0, HEADER_HEIGHT);
        cairo_line_to (cr, C_WIDTH, HEADER_HEIGHT);
        cairo_set_source_rgba (cr,
                               priv->active_color.red,
                               priv->active_color.green,
                               priv->active_color.blue,
                               1.0f);
        cairo_stroke (cr);

        /* draw body */
        if (!priv->message) {
                rounded_rectangle (cr,
                                   0, HEADER_HEIGHT,
                                   C_WIDTH, C_HEIGHT,
                                   BOX_RADIOUS,
                                   ROUND_CORNER_BL | ROUND_CORNER_BR);
                cairo_close_path (cr);
        }
        else
                cairo_rectangle (cr, 0, HEADER_HEIGHT,
                                 C_WIDTH, MESSAGE_HEIGHT);

        /* draw body filling depending on (in)active state */
        grad = cairo_pattern_create_linear (0, HEADER_HEIGHT,
                                            0, C_HEIGHT - FOOTER_HEIGHT);

        switch (priv->active) {
        case SELECTED_BODY:
                cairo_pattern_add_color_stop_rgba (grad,
                                                   0.5f,
                                                   priv->active_color.red,
                                                   priv->active_color.green,
                                                   priv->active_color.blue,
                                                   0.8f);
                cairo_pattern_add_color_stop_rgba (grad,
                                                   1.0f,
                                                   priv->active_color.red/2,
                                                   priv->active_color.green/2,
                                                   priv->active_color.blue/2,
                                                   0.8f);
                break;
        default:
                cairo_pattern_add_color_stop_rgba (grad, 0.5f,
                                                   0.4f, 0.4f, 0.4f, 0.8f);
                cairo_pattern_add_color_stop_rgba (grad, 1.0f,
                                                   0.2f, 0.2f, 0.2f, 0.8f);
        }

        cairo_set_source (cr, grad);
        cairo_fill (cr);

        cairo_pattern_destroy (grad);

        /* draw avatar */
        if (priv->avatar_pixbuf) {
                rounded_rectangle (cr,
                                   AVATAR_X, -AVATAR_Y,
                                   AVATAR_SIZE, AVATAR_SIZE,
                                   BOX_RADIOUS,
                                   ROUND_CORNER_ALL);
                cairo_close_path (cr);

                gdk_cairo_set_source_pixbuf (cr,
                                             priv->avatar_pixbuf,
                                             AVATAR_X,
                                             -AVATAR_Y);
                cairo_fill_preserve (cr);

                cairo_set_source_rgba (cr,
                                       priv->active_color.red,
                                       priv->active_color.green,
                                       priv->active_color.blue,
                                       1.0f);
                cairo_stroke (cr);
        }
        if (priv->presence_pixbuf) {
                guint x = C_WIDTH - HILDON_ICON_PIXEL_SIZE_XSMALL - HILDON_MARGIN_DEFAULT;
                guint y = (HEADER_HEIGHT - HILDON_ICON_PIXEL_SIZE_XSMALL)/2;

                if (priv->avatar_pixbuf)
                        x -= AVATAR_SIZE + HILDON_MARGIN_DEFAULT;

                cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                gdk_cairo_set_source_pixbuf (cr,
                                             priv->presence_pixbuf,
                                             x,
                                             y);
                cairo_paint (cr);
        }

        if (priv->message) {
                /* draw footer unread part bg */
                rounded_rectangle (cr,
                                   0, C_HEIGHT - FOOTER_HEIGHT,
                                   FOOTER_WIDTH, FOOTER_HEIGHT,
                                   BOX_RADIOUS,
                                   ROUND_CORNER_BL);
                cairo_close_path (cr);

                /* draw body filling depending on (in)active state */
                switch (priv->active) {
                case SELECTED_FOOTER:
                        cairo_set_source_rgba (cr,
                                               priv->active_color.red,
                                               priv->active_color.green,
                                               priv->active_color.blue,
                                               0.8f);
                        break;
                default:
                        cairo_set_source_rgba (cr, 0.1f, 0.1f, 0.1f, 0.9f);
                }
                cairo_fill (cr);

                /* draw footer received part bg */
                rounded_rectangle (cr,
                                   FOOTER_WIDTH, C_HEIGHT - FOOTER_HEIGHT,
                                   C_WIDTH - FOOTER_WIDTH, FOOTER_HEIGHT,
                                   BOX_RADIOUS,
                                   ROUND_CORNER_BR);
                cairo_close_path (cr);

                cairo_set_source_rgba (cr, 0.2f, 0.2f, 0.2f, 0.8f);
                cairo_fill (cr);

                /* draw message */
                if (!priv->message_surface) {
                        gint height;

                        priv->message_surface = draw_text (cr,
                                                           priv->font_desc,
                                                           priv->message,
                                                           MESSAGE_WIDTH,
                                                           &height);

                        priv->hidden_message_height = height - MESSAGE_HEIGHT;
                        priv->scroll_on_click = priv->hidden_message_height > 0;
                        if (priv->scroll_on_click)
                                gtk_widget_show (priv->cut_message);
                }

                cairo_rectangle (cr,
                                 2*CONTENT_OFFSET_X,
                                 TEXT_Y_OFFSET,
                                 MESSAGE_WIDTH,
                                 MESSAGE_HEIGHT);
                cairo_clip (cr);

                cairo_set_source_surface (cr,
                                          priv->message_surface,
                                          2*CONTENT_OFFSET_X,
                                          TEXT_Y_OFFSET - priv->scroll_offset);
                cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
                cairo_paint (cr);
        }

        cairo_destroy (cr);

        return GTK_WIDGET_CLASS (el_home_applet_parent_class)->expose_event (self, event);
}

static void
resize_sender (ELHomeAppletPrivate *priv)
{
        guint width = C_WIDTH;

        if (priv->avatar_pixbuf) {
                width -= AVATAR_SIZE + HILDON_MARGIN_DEFAULT;
        }

        if (priv->presence_pixbuf) {
                width -= HILDON_ICON_PIXEL_SIZE_XSMALL + HILDON_MARGIN_DEFAULT;
        }

        gtk_widget_set_size_request (priv->sender,
                                     width,
                                     HILDON_ICON_PIXEL_SIZE_THUMB);
}

static void
update_presence_pixbuf (ELHomeApplet *self,
                        OssoABookPresence *presence)
{
        ELHomeAppletPrivate *priv = self->priv;
        const gchar *icon_name = osso_abook_presence_get_icon_name (presence);
        gboolean resize = !!priv->presence_pixbuf ^ !!icon_name; /* logical via bit XOR */

        if (priv->presence_pixbuf) {
                g_object_unref (priv->presence_pixbuf);
                priv->presence_pixbuf = NULL;
        }

        if (icon_name)
                priv->presence_pixbuf = gtk_icon_theme_load_icon
                        (gtk_icon_theme_get_default (),
                         icon_name,
                         HILDON_ICON_PIXEL_SIZE_XSMALL,
                         0, NULL);
        if (resize)
                resize_sender (priv);

        gtk_widget_queue_draw (GTK_WIDGET (self));
}

static void
presence_updated (OssoABookPresence *presence,
                  GParamSpec *spec,
                  gpointer *user_data)
{
        ELHomeApplet *self = EL_HOME_APPLET(user_data);

        if (!OSSO_ABOOK_IS_CONTACT(self->priv->contact))
                return;

        update_presence_pixbuf (self, presence);
}

static void
show_contact (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        gtk_label_set_text (GTK_LABEL (priv->sender),
                            osso_abook_contact_get_display_name (priv->contact));
        resize_sender (priv);
        gtk_widget_queue_draw (GTK_WIDGET (self));
}

static void
resolve_contact (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        GList *contacts = NULL;

        if (priv->contact_id) {
                contacts = osso_abook_aggregator_lookup
                        (OSSO_ABOOK_AGGREGATOR (priv->aggregator),
                         priv->contact_id);
        }
        else if (priv->local_id && priv->remote_id) {
                if (g_strcmp0 (priv->local_id, "ring/tel/ring" == 0)) {
                        contacts = osso_abook_aggregator_find_contacts_for_phone_number
                                (OSSO_ABOOK_AGGREGATOR (priv->aggregator),
                                 priv->remote_id,
                                 TRUE);
                }
                else {
                        McAccount *account;
                        account = osso_abook_account_manager_lookup_by_name
                                (NULL,
                                 priv->local_id);
                        if (account) {
                                contacts = osso_abook_aggregator_find_contacts_for_im_contact
                                        (OSSO_ABOOK_AGGREGATOR (priv->aggregator),
                                         priv->remote_id,
                                         account);
                        }
                }
        }

        if (contacts && contacts->data) {
                priv->contact = g_object_ref (OSSO_ABOOK_CONTACT (contacts->data));
                g_signal_connect (priv->contact,
                                  "notify::presence-status",
                                  G_CALLBACK (presence_updated),
                                  self);
                priv->avatar_pixbuf = osso_abook_avatar_get_image_scaled
                        (OSSO_ABOOK_AVATAR (priv->contact),
                         HILDON_ICON_PIXEL_SIZE_THUMB,
                         HILDON_ICON_PIXEL_SIZE_THUMB,
                         TRUE);
                update_presence_pixbuf (self,
                                        OSSO_ABOOK_PRESENCE (priv->contact));
                show_contact (self);
        }
}

static void
contacts_added (OssoABookRoster  *roster,
                OssoABookContact **contacts,
                gpointer          userdata)
{
        ELHomeApplet *self = EL_HOME_APPLET (userdata);
        ELHomeAppletPrivate *priv = self->priv;

        if (!priv->contact)
                resolve_contact (self);
}

static void
reset_contact (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->avatar_pixbuf) {
                g_object_unref (priv->avatar_pixbuf);
                priv->avatar_pixbuf = NULL;
        }

        if (priv->presence_pixbuf) {
                g_object_unref (priv->presence_pixbuf);
                priv->presence_pixbuf = NULL;
        }

        if (priv->contact) {
                g_signal_handlers_disconnect_by_func (priv->contact,
                                                      presence_updated,
                                                      self);
                g_object_unref (priv->contact);
                priv->contact = NULL;
        }

        resize_sender (priv);
}

static void
contacts_removed (OssoABookRoster *roster,
                  const gchar     **ids,
                  gpointer         userdata)
{
        ELHomeApplet *self = EL_HOME_APPLET (userdata);
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->contact) {
                const gchar **contact_id;
                const gchar *uid = osso_abook_contact_get_uid (priv->contact);

                for (contact_id = ids; *contact_id; contact_id++) {
                        if (strcmp (*contact_id, priv->contact_id) == 0) {
                                reset_contact (self);

                                gtk_widget_queue_draw (GTK_WIDGET (self));
                                return;
                        }
                        if (strcmp (*contact_id, uid) == 0) {
                                reset_contact (self);
                                resolve_contact (self);
                                gtk_widget_queue_draw (GTK_WIDGET (self));
                                return;
                        }
                }
        }
}

static void
clean_contact (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        reset_contact (self);

        if (priv->aggregator) {
                if (priv->aggregator_ready_closure){
                        osso_abook_waitable_cancel (OSSO_ABOOK_WAITABLE (priv->aggregator),
                                                    priv->aggregator_ready_closure);
                        priv->aggregator_ready_closure = NULL;
                }
                g_signal_handlers_disconnect_by_func (priv->aggregator,
                                                      contacts_added,
                                                      self);
                g_signal_handlers_disconnect_by_func (priv->aggregator,
                                                      contacts_removed,
                                                      self);
                osso_abook_roster_stop (priv->aggregator);
                g_object_unref (priv->aggregator);
                priv->aggregator = NULL;
        }
}

static void
clean_state (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->message) {
                g_free (priv->message);
                priv->message = NULL;
        }

        if (priv->contact_id) {
                g_free (priv->contact_id);
                priv->contact_id = NULL;
        }
        if (priv->local_id) {
                g_free (priv->local_id);
                priv->local_id = NULL;
        }
        if (priv->remote_id) {
                g_free (priv->remote_id);
                priv->remote_id = NULL;
        }
        if (priv->group_uid) {
                g_free (priv->group_uid);
                priv->group_uid = NULL;
        }
}

static void
dispose (GObject *self)
{
        ELHomeAppletPrivate *priv = EL_HOME_APPLET(self)->priv;

        stop_scroll_anim (priv);
        if (priv->idle_id) {
                g_source_remove (priv->idle_id);
                priv->idle_id = 0;
        }
        if (priv->eventlogger) {
                g_object_unref (priv->eventlogger);
                priv->eventlogger = NULL;
        }
        if (priv->font_desc) {
                pango_font_description_free (priv->font_desc);
                priv->font_desc = NULL;
        }

        clean_state (EL_HOME_APPLET (self));
        clean_contact (EL_HOME_APPLET (self));

        G_OBJECT_CLASS (el_home_applet_parent_class)->dispose (self);
}

static void
finalize (GObject *self)
{
        G_OBJECT_CLASS (el_home_applet_parent_class)->finalize (self);
}

static void
aggregator_ready_cb (OssoABookWaitable *waitable,
                     const GError      *error,
                     gpointer           userdata)
{
        ELHomeApplet *self = EL_HOME_APPLET (userdata);
        ELHomeAppletPrivate *priv = self->priv;

        priv->aggregator_ready_closure = NULL;

        if (error) {
                g_warning ("Failed to create aggregator: %s", error->message);
                return;
        }

        g_signal_connect (priv->aggregator,
                          "contacts-added",
                          G_CALLBACK (contacts_added),
                          self);
        g_signal_connect (priv->aggregator,
                          "contacts-removed",
                          G_CALLBACK (contacts_removed),
                          self);

        resolve_contact (self);
}

static void
start_aggregator (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        EBookQuery *query = NULL;
        GError *error = NULL;

        if (priv->local_id && priv->remote_id) {
                const gchar *vcard = osso_abook_account_manager_get_vcard_field
                        (NULL, priv->local_id);
                if (vcard)
                        query = e_book_query_vcard_field_test (vcard,
                                                               E_BOOK_QUERY_IS,
                                                               priv->remote_id);
                else
                        query = e_book_query_any_field_contains (priv->remote_id);
        }

        if (query) {
                priv->aggregator = osso_abook_aggregator_new_with_query (NULL,
                                                                         query,
                                                                         NULL,
                                                                         1,
                                                                         &error);
                e_book_query_unref (query);
        }
        if (error) {
                g_warning ("Failed to create aggregator: %s", error->message);
                g_error_free (error);
                return;
        }

        if (priv->aggregator) {
                priv->aggregator_ready_closure = osso_abook_waitable_call_when_ready
                        (OSSO_ABOOK_WAITABLE (priv->aggregator),
                         aggregator_ready_cb,
                         self, NULL);

                osso_abook_roster_start (priv->aggregator);
        }
}

static gchar*
format_time (time_t t)
{
        static const guint RESULT_SIZE = 32;

        time_t now;
        struct tm now_tm, t_tm;
        const gchar *format = "%x %X";
        gchar *result = g_malloc0 (RESULT_SIZE);

        now = time (NULL);
        localtime_r (&now, &now_tm);
        localtime_r (&t, &t_tm);

        if ((now_tm.tm_year == t_tm.tm_year) &&
            (now_tm.tm_mon  == t_tm.tm_mon) &&
            (now_tm.tm_mday == t_tm.tm_mday))
                format = "%X";

        strftime (result, RESULT_SIZE, format, &t_tm);

        return result;
}

static void
show_event (ELHomeApplet *self, RTComElIter *it)
{
        ELHomeAppletPrivate *priv = self->priv;
        const gchar *remote = NULL;
        gchar *received = NULL;
        GValueArray *event = NULL;

        if (it && rtcom_el_iter_first (it)) {

                event = rtcom_el_iter_get_valuearray (it,
                                                      "id",
                                                      "start-time",
                                                      "local-uid",
                                                      "remote-uid",
                                                      "remote-name",
                                                      "remote-ebook-uid",
                                                      "free-text",
                                                      "group-uid",
                                                      NULL);
                if (event) {
                        time_t received_t;
#define _VARR_DUP_STR(array, i) g_value_dup_string (g_value_array_get_nth ((array), (i)))

                        priv->event_id = g_value_get_int (g_value_array_get_nth (event, 0));
                        received_t = g_value_get_int (g_value_array_get_nth (event, 1));
                        received = format_time (received_t);
                        priv->local_id = _VARR_DUP_STR (event, 2);
                        priv->remote_id = _VARR_DUP_STR (event, 3);
                        if (priv->remote_id && priv->remote_id[0]) {
                                remote = g_value_get_string (g_value_array_get_nth (event, 4));
                                if (!remote)
                                        remote = priv->remote_id;
                                priv->contact_id = _VARR_DUP_STR (event, 5);
                        }
                        else if (priv->remote_id) {
                                g_free (priv->remote_id);
                                priv->remote_id = NULL;
                        }

                        priv->message = _VARR_DUP_STR (event, 6);
                        priv->group_uid = _VARR_DUP_STR (event, 7);

#undef _VARR_DUP_STR
                }
        }
        else {
                priv->event_id = -1;
        }

        if (priv->message) {
                gtk_widget_hide (priv->empty);
        }
        else {
                gtk_widget_show (priv->empty);
        }

        gtk_label_set_text (GTK_LABEL (priv->received), received);

        if (remote)
                gtk_label_set_text (GTK_LABEL (priv->sender), remote);
        else
                gtk_label_set_text (GTK_LABEL (priv->sender), priv->remote_id);

        stop_scroll_anim (priv);
        priv->scroll_offset = 0;
        if (priv->message_surface) {
                cairo_surface_destroy (priv->message_surface);
                priv->message_surface = NULL;
        }

        if (event)
                g_value_array_free (event);

        gtk_widget_hide (priv->cut_message);
        gtk_widget_queue_draw (GTK_WIDGET (self));
}

static RTComElIter*
make_query (RTComEl *el, gint event_id)
{
        RTComElQuery *query = NULL;
        RTComElIter *it = NULL;

        query = rtcom_el_query_new (el);
        rtcom_el_query_set_limit (query, 1);
        if (event_id >= 0) {
                rtcom_el_query_prepare (query,
                                        "id", event_id, RTCOM_EL_OP_EQUAL,
                                        NULL);
        }
        else {
                rtcom_el_query_prepare (query,
                                        "is-read", FALSE, RTCOM_EL_OP_EQUAL,
                                        "service", conv_services, RTCOM_EL_OP_IN_STRV,
                                        "event-type", conv_event_types, RTCOM_EL_OP_IN_STRV,
                                        NULL);
        }
        it = rtcom_el_get_events (el, query);
        g_object_unref (query);

        return it;
}

static void
update_unread_label (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->unread_count > 0) {
                gchar *text;
                text = g_strdup_printf
                        ("%d<span foreground=\"red\" rise=\"5000\">*</span>",
                         priv->unread_count);

                gtk_label_set_markup (GTK_LABEL (priv->unread), text);
                g_free (text);
        }
        else
                gtk_label_set_text (GTK_LABEL (priv->unread), NULL);
}

static gint
query_unread_events (RTComEl *el)
{
        sqlite3 *db;
        sqlite3_stmt *stmt;
        int ret;
        gint count = 0;

        g_object_get (el, "db", &db, NULL);

        if (sqlite3_prepare_v2 (db,
                                "SELECT SUM(total_events)-SUM(read_events) FROM GroupCache;",
                                -1,
                                &stmt,
                                NULL) != SQLITE_OK) {
                g_error ("%s: can't compile SQL", G_STRFUNC);
                return -1;
        }

        while (SQLITE_BUSY == (ret = sqlite3_step (stmt)));

        if (ret == SQLITE_ROW) {
                count = sqlite3_column_int (stmt, 0);
        }
        else {
                g_error ("%s: error while executing SQL", G_STRFUNC);
        }

        sqlite3_finalize (stmt);

        return count;
}

static gboolean
query_read_events (RTComEl *el, const gchar *service, gint *events, gint *conversations)
{
        sqlite3 *db;
        sqlite3_stmt *stmt;
        int ret;
        gboolean result = TRUE;

        g_object_get (el, "db", &db, NULL);

        if (sqlite3_prepare_v2 (db,
                                "SELECT SUM(total_events), COUNT(group_uid) FROM GroupCache, Services "
                                "WHERE GroupCache.service_id=Services.id AND Services.name=?;",
                                -1,
                                &stmt,
                                NULL) != SQLITE_OK) {
                g_error ("%s: can't compile SQL", G_STRFUNC);
                return FALSE;
        }
        if (sqlite3_bind_text (stmt, 1, service, -1, SQLITE_STATIC) != SQLITE_OK)  {
                g_error ("Failed to bind %s to SQL stmt", service);
                result = FALSE;
                goto DONE;
        }

        while (SQLITE_BUSY == (ret = sqlite3_step (stmt)));

        if (ret == SQLITE_ROW) {
                *events = sqlite3_column_int (stmt, 0);
                *conversations = sqlite3_column_int (stmt, 1);
        }
        else {
                g_error ("%s: error while executing SQL", G_STRFUNC);
                result = FALSE;
                goto DONE;
        }

 DONE:
        sqlite3_finalize (stmt);

        return result;
}

static void
am_ready (OssoABookAccountManager *manager,
          const GError            *error,
          gpointer                 user_data)
{
        ELHomeApplet *self = EL_HOME_APPLET (user_data);
        ELHomeAppletPrivate *priv = self->priv;

        if (!error &&
            priv->local_id &&
            !GTK_WIDGET_VISIBLE (priv->icon)) {
                McAccount *account;

                account = osso_abook_account_manager_lookup_by_name (NULL,
                                                                     priv->local_id);
                if (account) {
                        McProfile *profile = mc_profile_lookup (mc_account_compat_get_profile (account));
                        const gchar *icon_name = mc_profile_get_icon_name (profile);
                        if (icon_name) {
                                gtk_image_set_from_icon_name (GTK_IMAGE (priv->icon),
                                                              icon_name,
                                                              HILDON_ICON_SIZE_XSMALL);
                                gtk_widget_show (priv->icon);
                        }
                }
        }
}

static void
read_event (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        RTComElIter *it = NULL;
        const gchar *icon_name = NULL;
        gchar *remote_id;
        gchar *local_id;

        remote_id = g_strdup (priv->remote_id);
        local_id = g_strdup (priv->local_id);

        clean_state (self);

        it = make_query (priv->eventlogger, -1);
        show_event (self, it);

        if (it) g_object_unref (it);

        if (priv->event_id >= 0) {
                gboolean new_account = g_strcmp0 (priv->local_id, local_id);

                if (g_strcmp0 (priv->remote_id, remote_id) || new_account) {
                        clean_contact (self);
                        start_aggregator (self);
                }
                else {
                        show_contact (self);
                }

                if (new_account) {
                        if (g_strcmp0 (priv->local_id, "ring/tel/ring") == 0) {
                                icon_name = "general_sms";
                        }
                        else {
                                McAccount *account;
                                OssoABookAccountManager *am = osso_abook_account_manager_get_default ();
                                if (!osso_abook_waitable_is_ready (OSSO_ABOOK_WAITABLE (am), NULL)) {
                                        osso_abook_account_manager_call_when_ready  (am,
                                                                                     am_ready,
                                                                                     self,
                                                                                     NULL);
                                }
                                else {
                                        account = osso_abook_account_manager_lookup_by_name (NULL,
                                                                                             priv->local_id);
                                        if (account) {
                                                McProfile *profile = mc_profile_lookup (mc_account_compat_get_profile (account));
                                                icon_name = mc_profile_get_icon_name (profile);
                                        }
                                }
                        }

                        if (icon_name) {
                                gtk_image_set_from_icon_name (GTK_IMAGE (priv->icon),
                                                              icon_name,
                                                              HILDON_ICON_SIZE_XSMALL);
                                gtk_widget_show (priv->icon);
                        }
                        else
                                gtk_widget_hide (priv->icon);
                }
        }
        else {
                gchar *text;
                gint n_sms_events = 0, n_sms_convs = 0;
                gint n_chat_events = 0, n_chat_convs = 0;
                const gchar *fmt = "%d <span size=\"small\">(%d)</span>";

                query_read_events (priv->eventlogger,
                                   "RTCOM_EL_SERVICE_SMS",
                                   &n_sms_events, &n_sms_convs);
                query_read_events (priv->eventlogger,
                                   "RTCOM_EL_SERVICE_CHAT",
                                   &n_chat_events, &n_chat_convs);

                text = g_strdup_printf (fmt, n_sms_convs, n_sms_events);
                gtk_label_set_markup (GTK_LABEL (priv->sms_total), text);
                g_free (text);

                text = g_strdup_printf (fmt, n_chat_convs, n_chat_events);
                gtk_label_set_markup (GTK_LABEL (priv->chat_total), text);
                g_free (text);

                gtk_label_set_text (GTK_LABEL (priv->sender),
                                    dgettext ("rtcom-messaging-ui",
                                              "messaging_ap_conversations"));

                clean_contact (self);
                gtk_widget_hide (priv->icon);
        }

        g_free (local_id);
        g_free (remote_id);
}

static void
remove_notification (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        DBusGConnection *conn;
        GError *error;
        DBusGProxy *proxy;
        GPtrArray *conv_structs;
        GType conv_structs_type;
        GValueArray *account_info;
        GValue value = {0, };
        DBusGProxyCall *call;

        if (!(priv->remote_id && priv->local_id))
                return;

        conn = hd_home_plugin_item_get_dbus_g_connection (HD_HOME_PLUGIN_ITEM (self),
                                                          DBUS_BUS_SESSION,
                                                          &error);
        if (!conn) {
                g_error ("Failed get dbus g connection %s", error->message);
                g_error_free (error);
                return;
        }

        proxy = dbus_g_proxy_new_for_name (conn,
                                           NOTIFICATION_UI_DBUS_NAME,
                                           NOTIFICATION_UI_DBUS_PATH,
                                           NOTIFICATION_UI_DBUS_IFACE);

        conv_structs = g_ptr_array_sized_new (1);
        account_info = g_value_array_new (2);

        g_value_init (&value, G_TYPE_STRING);
        g_value_set_string (&value, priv->local_id);
        g_value_array_append (account_info, &value);
        g_value_unset (&value);

        g_value_init (&value, G_TYPE_STRING);
        g_value_set_string (&value, priv->remote_id);
        g_value_array_append (account_info, &value);
        g_value_unset (&value);

        g_ptr_array_add (conv_structs, account_info);

        conv_structs_type = dbus_g_type_get_collection
                ("GPtrArray",
                 dbus_g_type_get_struct ("GValueArray",
                                         G_TYPE_STRING,
                                         G_TYPE_STRING,
                                         G_TYPE_INVALID));

        call = dbus_g_proxy_begin_call (proxy,
                                        "ClearConversationNotifications",
                                        NULL, NULL, NULL,
                                        conv_structs_type,
                                        conv_structs,
                                        G_TYPE_INVALID);

        g_value_array_free (account_info);
        g_ptr_array_free (conv_structs, TRUE);

        g_object_unref (proxy);
}

static void
mark_as_read (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->event_id >= 0) {
                rtcom_el_set_read_event (priv->eventlogger,
                                         priv->event_id,
                                         TRUE,
                                         NULL);
                remove_notification (self);
        }
}

static void
launch_conversations (ELHomeApplet *self)
{
        DBusConnection *conn;
        DBusMessage *message;
        DBusError error;

        dbus_error_init (&error);
        conn = hd_home_plugin_item_get_dbus_connection (HD_HOME_PLUGIN_ITEM (self),
                                                        DBUS_BUS_SESSION,
                                                        &error);
        if (!conn) {
                if (dbus_error_is_set (&error)) {
                        g_error ("Failed to get dbus connection %s", error.message);
                        dbus_error_free (&error);
                }
                return;
        }

        message = dbus_message_new_method_call (CONVERSATIONS_UI_DBUS_NAME,
                                                CONVERSATIONS_UI_DBUS_PATH,
                                                CONVERSATIONS_UI_DBUS_IFACE,
                                                "top_application");
        dbus_message_set_no_reply (message, TRUE);

        if (dbus_connection_send (conn, message, NULL))
                dbus_connection_flush (conn);
        dbus_message_unref (message);

        dbus_connection_close (conn);
}

static void
open_conversation (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        McAccount *account;
        const gchar *persistent_id = NULL;

        if (!((priv->remote_id || priv->group_uid) && priv->local_id))
                return;

        account = osso_abook_account_manager_lookup_by_name (NULL,
                                                             priv->local_id);
        if (!account)
                return;

        if (priv->group_uid &&
            g_str_has_prefix (priv->group_uid, "group:")) {
                persistent_id = strchr (priv->group_uid, '-');
                if (persistent_id)
                        persistent_id++;
        }

        if (persistent_id && persistent_id[0] != '\0') {
                GHashTable *properties = tp_asv_new
                        (TP_IFACE_CHANNEL ".ChannelType", G_TYPE_STRING,
                         TP_IFACE_CHANNEL_TYPE_TEXT,
                         TP_IFACE_CHANNEL ".TargetHandleType", G_TYPE_UINT,
                         TP_HANDLE_TYPE_NONE,
                         RTCOM_TP_IFACE_CHANNEL_INTERFACE_PERSISTENT ".PersistentID",
                         G_TYPE_STRING, persistent_id,
                         NULL);

                mc_account_channelrequest_ht (account,
                                              properties,
                                              time (NULL),
                                              NULL,
                                              MC_ACCOUNT_CR_FLAG_USE_EXISTING,
                                              NULL, NULL, NULL, NULL);

                g_hash_table_unref (properties);
        }
        else if (priv->remote_id) {
                McAccountChannelrequestData request;

                MC_ACCOUNT_CRD_INIT (&request);
                MC_ACCOUNT_CRD_SET (&request, channel_type, TP_IFACE_QUARK_CHANNEL_TYPE_TEXT);
                MC_ACCOUNT_CRD_SET (&request, target_handle_type, TP_HANDLE_TYPE_CONTACT);
                MC_ACCOUNT_CRD_SET (&request, target_id, priv->remote_id);

                mc_account_channelrequest (account,
                                           &request,
                                           time (NULL),
                                           NULL,
                                           MC_ACCOUNT_CR_FLAG_USE_EXISTING,
                                           NULL, NULL, NULL, NULL);
        }
}

static gboolean
read_new_event (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        read_event (self);
        priv->unread_count = query_unread_events (priv->eventlogger);
        update_unread_label (self);

        priv->idle_id = 0;

        return FALSE;
}

static void
add_new_idle (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->idle_id)
                g_source_remove (priv->idle_id);
        priv->idle_id = g_idle_add ((GSourceFunc)read_new_event,
                                    self);
}

static void
new_event_cb (RTComEl      *backend,
              gint          event_id,
              const gchar  *local_uid,
              const gchar  *remote_uid,
              const gchar  *remote_ebook_uid,
              const gchar  *group_uid,
              const gchar  *service,
              ELHomeApplet *self)
{
        if (service && service[0] != '\0') {
                const gchar** conv_service = conv_services;
                do {
                        if (!g_strcmp0 (*conv_service, service)) {
                                add_new_idle (self);
                                return;
                        }
                }
                while(*++conv_service);
        }
        else
                add_new_idle (self);
}

static gboolean
scroll_anim_cb (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        gboolean to_continue;

        priv->scroll_offset += SCROLL_STEP;
        gtk_widget_queue_draw_area (GTK_WIDGET (self),
                                    3*CONTENT_OFFSET_X,
                                    HEADER_HEIGHT + CONTENT_OFFSET_Y_TOP,
                                    MESSAGE_WIDTH,
                                    MESSAGE_HEIGHT);

        to_continue = priv->scroll_offset <= priv->hidden_message_height;
        if (!to_continue) {
                priv->scroll_anim_id = 0;
                gtk_widget_hide (priv->cut_message);
        }

        return to_continue;
}

static gboolean
button_press_event_cb (GtkWidget      *widget,
                       GdkEventButton *event,
                       ELHomeApplet   *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->event_id >= 0) {
                if (event->y < CONTENT_OFFSET_Y_TOP + HEADER_HEIGHT) {
                        if (priv->aggregator &&
                            osso_abook_waitable_is_ready
                            (OSSO_ABOOK_WAITABLE (priv->aggregator), NULL))
                                priv->active = SELECTED_HEADER;
                }
                else if (event->y > (BOX_HEIGHT - CONTENT_OFFSET_Y_BOTTOM - FOOTER_HEIGHT_PRESS) &&
                         event->x < FOOTER_WIDTH_PRESS)
                        priv->active = SELECTED_FOOTER;
                else
                        priv->active = SELECTED_BODY;
        }
        else {
                priv->active = SELECTED_BODY;
        }

        gtk_widget_queue_draw (widget);

        return TRUE;
}

static GtkWidget*
create_contact_starter_dialog (OssoABookAggregator *aggregator, const gchar *contact_id)
{
        GtkWidget *dialog = NULL;
        GList *contacts = osso_abook_aggregator_lookup (aggregator, contact_id);
        if (contacts && contacts->data) {
                GtkWidget *starter =
                        osso_abook_touch_contact_starter_new_with_contact
                        (NULL,
                         OSSO_ABOOK_CONTACT (contacts->data));
                dialog = osso_abook_touch_contact_starter_dialog_new
                        (NULL,
                         OSSO_ABOOK_TOUCH_CONTACT_STARTER (starter));
                gtk_widget_show_all (starter);
        }

        g_list_free (contacts);

        return dialog;
}

static GtkWidget*
create_temporary_contact_dialog (const gchar *remote_id,
                                 const gchar *account_id)
{
        GtkWidget *dialog = NULL;
        const gchar *vcard = NULL;
        McAccount *account = NULL;

        if (account_id) {
            vcard = osso_abook_account_manager_get_vcard_field (NULL, account_id);
            account = osso_abook_account_manager_lookup_by_name (NULL, account_id);
        }

        if (vcard && account) {
                EVCardAttribute *attribute = e_vcard_attribute_new (NULL, vcard);

                e_vcard_attribute_add_value (attribute, remote_id);
                dialog = osso_abook_temporary_contact_dialog_new
                        (NULL,
                         NULL, /*EBook            *book,*/
                         attribute,
                         account);
                g_signal_connect (dialog,
                                  "response",
                                  G_CALLBACK (gtk_widget_destroy),
                                  NULL);
                e_vcard_attribute_free (attribute);
        }

        return dialog;
}

static gboolean
button_release_event_cb (GtkWidget      *widget,
                         GdkEventButton *event,
                         ELHomeApplet   *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        switch (priv->active) {
        case SELECTED_BODY:
                if (priv->event_id >= 0) {
                        reset_scroll (self);
                        open_conversation (self);
                }
                else
                        launch_conversations (self);
                break;
        case SELECTED_HEADER: {
                GtkWidget *dialog = NULL;

                reset_scroll (self);

                if (priv->aggregator && priv->contact_id)
                        dialog = create_contact_starter_dialog
                                (OSSO_ABOOK_AGGREGATOR (priv->aggregator),
                                 priv->contact_id);
                if (!dialog &&
                    priv->remote_id &&
                    priv->local_id)
                        dialog = create_temporary_contact_dialog (priv->remote_id,
                                                                  priv->local_id);

                if (dialog)
                        gtk_widget_show (dialog);
        }
                break;
        case SELECTED_FOOTER:
                if (priv->scroll_on_click) {
                        priv->scroll_on_click = FALSE;
                        priv->scroll_anim_id = g_timeout_add (SCROLL_PERIOD,
                                                              (GSourceFunc)scroll_anim_cb,
                                                              self);
                }
                else
#ifndef DEBUG_LAYOUT
                        mark_as_read (self);
#endif
                break;
        default:;
        }

        priv->active = SELECTED_NONE;
        gtk_widget_queue_draw (widget);

        return TRUE;
}

static gboolean
leave_notify_event_cb (GtkWidget        *widget,
                       GdkEventCrossing *event,
                       ELHomeApplet     *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        switch (priv->active) {
        case SELECTED_FOOTER:
                stop_scroll_anim (priv);
                /* fall down */
        case SELECTED_HEADER:
        case SELECTED_BODY:
                gtk_widget_queue_draw (widget);
                break;
        default:;
        }

        priv->active = SELECTED_NONE;
        return FALSE;
}

static void
el_home_applet_init (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv;
        GtkWidget *event_box;
        GtkWidget *hbox, *vbox, *align, *footer;
        GtkWidget *w;

        self->priv = EL_HOME_APPLET_GET_PRIVATE (self);
        priv = self->priv;

        gtk_widget_set_app_paintable (GTK_WIDGET (self), TRUE);

        priv->unread = gtk_label_new ("12");
        gtk_misc_set_alignment (GTK_MISC (priv->unread),
                                0.0f,
                                0.5f);
        hildon_helper_set_logical_font (priv->unread, "SmallSystemFont");

        priv->icon = gtk_image_new ();
        gtk_misc_set_alignment (GTK_MISC (priv->icon),
                                0.5f,
                                0.5f);

        priv->sender = gtk_label_new ("asdf asdf asdf asdf asdf");
        gtk_misc_set_alignment (GTK_MISC (priv->sender),
                                0.5f,
                                0.55f);
        gtk_label_set_ellipsize (GTK_LABEL (priv->sender),
                                 PANGO_ELLIPSIZE_END);
        gtk_widget_set_name (priv->sender, "hildon-shadow-label");
        hildon_helper_set_logical_font (priv->sender, "SystemFont");
        gtk_widget_set_size_request (priv->sender,
                                     C_WIDTH,
                                     HILDON_ICON_PIXEL_SIZE_THUMB);

        priv->message = g_strdup ("One two three four five six seven eight nine ten");

        /* construt empty table */
        priv->empty = gtk_fixed_new ();

        w = gtk_image_new_from_icon_name ("general_sms", HILDON_ICON_SIZE_FINGER);
        gtk_fixed_put (GTK_FIXED (priv->empty), w,
                       4*HILDON_MARGIN_DOUBLE,
                       2*HILDON_MARGIN_DOUBLE);

        w = gtk_image_new_from_icon_name ("general_chat", HILDON_ICON_SIZE_FINGER);
        gtk_fixed_put (GTK_FIXED (priv->empty), w,
                       4*HILDON_MARGIN_DOUBLE,
                       3*HILDON_MARGIN_DOUBLE + HILDON_ICON_PIXEL_SIZE_FINGER);

        priv->sms_total = gtk_label_new (NULL);
        gtk_widget_set_name (priv->sms_total, "hildon-shadow-label");
        gtk_fixed_put (GTK_FIXED (priv->empty), priv->sms_total,
                       5*HILDON_MARGIN_DOUBLE  + HILDON_ICON_PIXEL_SIZE_FINGER,
                       2*HILDON_MARGIN_DOUBLE + HILDON_MARGIN_HALF);

        priv->chat_total = gtk_label_new (NULL);
        gtk_widget_set_name (priv->chat_total, "hildon-shadow-label");
        gtk_fixed_put (GTK_FIXED (priv->empty), priv->chat_total,
                       5*HILDON_MARGIN_DOUBLE  + HILDON_ICON_PIXEL_SIZE_FINGER,
                       3*HILDON_MARGIN_DOUBLE + HILDON_MARGIN_HALF + HILDON_ICON_PIXEL_SIZE_FINGER);

        gtk_widget_show_all (GTK_WIDGET (priv->empty));
        gtk_widget_hide (GTK_WIDGET (priv->empty));
        GTK_WIDGET_SET_FLAGS (priv->empty, GTK_NO_SHOW_ALL);

        priv->received = gtk_label_new ("aewf aewf aewf awef");
        gtk_misc_set_alignment (GTK_MISC (priv->received),
                                1.0f,
                                0.5f);
        hildon_helper_set_logical_font (priv->received, "SmallSystemFont");
        gtk_widget_set_name (priv->received, "hildon-shadow-label");


        priv->cut_message = gtk_label_new ("...");
        gtk_misc_set_alignment (GTK_MISC (priv->cut_message),
                                0.5f,
                                0.0f);
        hildon_helper_set_logical_font (priv->cut_message, "SmallSystemFont");
        gtk_widget_set_name (priv->cut_message, "hildon-shadow-label");
        GTK_WIDGET_SET_FLAGS (priv->cut_message, GTK_NO_SHOW_ALL);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), priv->sender, FALSE, FALSE, 0);

        footer = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
        gtk_box_pack_start (GTK_BOX (footer), priv->unread, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (footer), priv->cut_message, TRUE, TRUE, 0);
        gtk_box_pack_end (GTK_BOX (footer), priv->icon, FALSE, FALSE, 0);
        gtk_box_pack_end (GTK_BOX (footer), priv->received, FALSE, FALSE, 0);

        vbox = gtk_vbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (vbox), priv->empty, TRUE, TRUE, 0);
        gtk_box_pack_end (GTK_BOX (vbox), footer, FALSE, FALSE, 0);

        align = gtk_alignment_new (0.5f, 0.0f, 1.0f, 1.0f);
        gtk_alignment_set_padding (GTK_ALIGNMENT (align),
                                   0, 0, HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT);

        gtk_container_set_border_width (GTK_CONTAINER (vbox), HILDON_MARGIN_HALF);

        event_box = gtk_event_box_new ();
        gtk_event_box_set_visible_window (GTK_EVENT_BOX (event_box), FALSE);
        gtk_widget_set_size_request (event_box, BOX_WIDTH, BOX_HEIGHT);

        gtk_container_add (GTK_CONTAINER (align), vbox);
        gtk_container_add (GTK_CONTAINER (event_box), align);
        gtk_container_add (GTK_CONTAINER (self), event_box);

        g_signal_connect (event_box,
                          "button-press-event",
                          G_CALLBACK (button_press_event_cb),
                          self);
        g_signal_connect (event_box,
                          "button-release-event",
                          G_CALLBACK (button_release_event_cb),
                          self);
        g_signal_connect (event_box,
                          "leave-notify-event",
                          G_CALLBACK (leave_notify_event_cb),
                          self);

        g_signal_connect (event_box,
                          "style-set",
                          G_CALLBACK (style_set_cb),
                          self);
        g_signal_connect (self,
                          "notify::is-on-current-desktop",
                          G_CALLBACK (notify_on_current_desktop),
                          self);

        gtk_widget_show_all (GTK_WIDGET (event_box));

#ifndef DEBUG_LAYOUT
        priv->eventlogger = rtcom_el_new ();
        g_signal_connect (priv->eventlogger,
                          "new-event",
                          G_CALLBACK (new_event_cb),
                          self);
        g_signal_connect (priv->eventlogger,
                          "event-updated",
                          G_CALLBACK (new_event_cb),
                          self);

        osso_abook_init_with_name (PACKAGE, NULL);

        add_new_idle (self);
#endif
}

static void
el_home_applet_class_init (ELHomeAppletClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);
        GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

        object_class->dispose = dispose;
        object_class->finalize = finalize;
        widget_class->expose_event = expose_event;
        widget_class->realize = el_home_applet_realize;

        g_type_class_add_private (klass, sizeof (ELHomeAppletPrivate));
}

