/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#include "TextSurfaceManager.h"
#include <QtCore/QDebug>
#include "../display/texture.h"

GLfloat textTexCoords[][8] = { { 180/256.0f, 0.0f, /* Top Right Of The Texture */
		180/256.0f, 180/256.0f, /* Bottom Right Of The Texture */
		0.0f, 0.0f, /* Bottom Left Of The Texture */
		0.0f, 180/256.0f, /* Top Left Of The Texture */
} };

GLfloat** TextSurfaceManager::getTexCoords(TextEntry* textEntry) {
	GLfloat x = textEntry -> width / (GLfloat) nextPowerOfTwo(textEntry -> width);
	GLfloat y = textEntry -> height / (GLfloat) nextPowerOfTwo(textEntry -> height);
	textTexCoords[0][0] = x;
	textTexCoords[0][2] = x;
	textTexCoords[0][3] = y;
	textTexCoords[0][7] = y;
	return (GLfloat**) textTexCoords;
}

GLfloat** TextSurfaceManager::getTexCoordsLandscape(TextEntry* textEntry) {
	GLfloat x = textEntry -> width / (GLfloat) nextPowerOfTwo(textEntry -> width);
	GLfloat y = textEntry -> height / (GLfloat) nextPowerOfTwo(textEntry -> height);
	textTexCoords[0][0] = y;
	textTexCoords[0][2] = y;
	textTexCoords[0][3] = x;
	textTexCoords[0][7] = x;
	return (GLfloat**) textTexCoords;
}

TextSurfaceManager::TextSurfaceManager() {
}

QString TextSurfaceManager::createKey(int fontSize, QString text) {
	QString result;
	result.setNum(fontSize).append("_").append(text);
	return result;
}

TextSurfaceManager* TextSurfaceManager::instance() {
	static TextSurfaceManager* instance = new TextSurfaceManager();
	return instance;
}

TextEntry* TextSurfaceManager::createEntry(int fontSize, QString text) {
	static SDL_Color color = { 255, 255, 255 };
	static SDL_Rect rect;
	TextEntry* entry = new TextEntry();
	SDL_Surface* text_surface = TTF_RenderUTF8_Blended(getFont(fontSize), text.toUtf8().data(), color);
	if (text_surface == NULL) {
		fprintf(stderr, "TTF_RenderUTF8_Blended error: %s\n", TTF_GetError());
		exit(1);
	}
	entry -> texture = 0;
	entry -> width = text_surface -> w;
	entry -> height = text_surface -> h;

	SDL_Surface* tmpSurface = SDL_CreateRGBSurface(SDL_SWSURFACE, nextPowerOfTwo(text_surface -> w), nextPowerOfTwo(text_surface -> h),
			16, 0x00000F000, 0x00000F00, 0x000000F0, 0x000000F);

	if (tmpSurface == NULL) {
		fprintf(stderr, "SDL_CreateRGBSurface error: %s\n", SDL_GetError());
		exit(1);
	}
	rect.x = 0;
	rect.y = 0;
	rect.w = text_surface->w;
	rect.h = text_surface->h;
	SDL_UpperBlit(text_surface, &rect, tmpSurface, &rect);
	entry -> surface = tmpSurface;
	SDL_FreeSurface(text_surface);
	return entry;
}

TTF_Font* TextSurfaceManager::getFont(int fontSize) {
	TTF_Font* result = fonts.value(fontSize);

	if (result == NULL) {
		qDebug() << "Opening font " << options.ttfFont << ", size: " << fontSize;
		result = TTF_OpenFont(options.ttfFont, fontSize);
		if (result == NULL) {
			fprintf(stderr, "TTF_OpenFont error: %s\n", TTF_GetError());
			exit(1);
		}
		fonts.insert(fontSize, result);
	}
	return result;
}

int TextSurfaceManager::getTextWidth(int fontSize, QString text) {
	QString key = createKey(fontSize, text);
	TextEntry* entry = textEntries.value(key);
	if (entry != NULL) {
		return entry -> width;
	}
	int w, h;
	TTF_SizeUTF8(getFont(fontSize), text.toUtf8().data(), &w, &h);
	return w;
}

int TextSurfaceManager::getTextHeight(int fontSize, QString text) {
	QString key = createKey(fontSize, text);
	TextEntry* entry = textEntries.value(key);
	if (entry != NULL) {
		return entry -> height;
	}
	int w, h;
	TTF_SizeUTF8(getFont(fontSize), text.toUtf8().data(), &w, &h);
	return h;
}

TextEntry* TextSurfaceManager::getTextEntry(int fontSize, QString text) {
	QString key = createKey(fontSize, text);
	TextEntry* entry = textEntries.value(key);
	if (entry == NULL) {
		entry = createEntry(fontSize, text);
		textEntries.insert(key, entry);
	}
	if (entry -> texture != 0) {
		return entry;
	}
	entry -> texture = createTexture((GLushort*) entry -> surface -> pixels, entry -> surface -> w, entry -> surface -> h, FALSE);
	return entry;
}

void TextSurfaceManager::deleteUnusedTextures() {
}

TextSurfaceManager::~TextSurfaceManager() {
	// TODO Auto-generated destructor stub
}
