/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */


#include "input.h"

Uint8* keys = NULL;

void initInput() {
	int i;

	for (i = 0; i < MAX_MOUSE_POINTERS; i++) {
		mouse[i].x = 0;
		mouse[i].y = 0;
		mouse[i].button = 0;
		mouse[i].oldx = 0;
		mouse[i].oldy = 0;
		mouse[i].xdelta = 1;
		mouse[i].ydelta = 0;
		mouse[i].oldButton = 0;
		mouse[i].pressed = -1;
		mouse[i].moved = -1;
		mouse[i].processed = false;
	}
	SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEBUTTONUP, SDL_IGNORE);
	SDL_EnableUNICODE(SDL_ENABLE);
	SDL_EnableKeyRepeat(250, 30);
}

void storeMouseOldValues() {
	int i;

	for (i = 0; i < MAX_MOUSE_POINTERS; i++) {
		mouse[i].oldx = mouse[i].x;
		mouse[i].oldy = mouse[i].y;
		mouse[i].oldButton = mouse[i].button;
	}
}

void readMousePositionAndButton() {
#ifdef N950
	int i;

	// true multitouch
	for (i = 0; i < MAX_MOUSE_POINTERS; i++) {
		mouse[i].button = SDL_GetMultiMouseState(i, &mouse[i].x, &mouse[i].y);
	}
#else
	// for debugging treat left mouse button as first pointer and right mouse button as second
	int button, x, y, idx;
	button = SDL_GetMouseState(&x, &y);

	if (keys[SDLK_LSHIFT]) {
		if (button != 0) {
			if (button > 1) {
				idx = 1;
			} else {
				idx = 0;
			}
			mouse[idx].x = x;
			mouse[idx].y = y;
			mouse[idx].button = 1;
		}
	} else {
		mouse[1].button = 0;
		mouse[0].x = x;
		mouse[0].y = y;
		mouse[0].button = button;
	}
#endif
}

void updateMouseValues() {
	int i;

	for (i = 0; i < MAX_MOUSE_POINTERS; i++) {
		if (mouse[i].button == 0) {
			mouse[i].xdelta = 0;
			mouse[i].ydelta = 0;
			mouse[i].draggedDistanceX = 0;
			mouse[i].draggedDistanceY = 0;
		}

		if (mouse[i].button != 0 && mouse[i].oldButton == 0) {
			mouse[i].pressed = nowMillis;
			mouse[i].clickedx = mouse[i].x;
			mouse[i].clickedy = mouse[i].y;
			mouse[i].draggedDistanceX = 0;
			mouse[i].draggedDistanceY = 0;
			mouse[i].processed = false;
		}

		if (mouse[i].oldButton == 1 && mouse[i].button == 1) {
			mouse[i].xdelta = mouse[i].x - mouse[i].oldx;
			mouse[i].ydelta = mouse[i].y - mouse[i].oldy;
			if (mouse[i].xdelta != 0 || mouse[i].ydelta != 0) {
				mouse[i].moved = nowMillis;

				int draggedDistanceX = abs(mouse[i].x - mouse[i].clickedx);
				int draggedDistanceY = abs(mouse[i].y - mouse[i].clickedy);

				if (draggedDistanceX > mouse[i].draggedDistanceX) {
					mouse[i].draggedDistanceX = draggedDistanceX;
				}
				if (draggedDistanceY > mouse[i].draggedDistanceY) {
					mouse[i].draggedDistanceY = draggedDistanceY;
				}
			}
		}
	}
}

void processMouse() {
	storeMouseOldValues();
	readMousePositionAndButton();
	updateMouseValues();
	tileEngineProcessMouse(processUiMouse());
}

BackgroundQueryProvider* getNextProvider(GList* allProviders, BackgroundQueryProvider* currentProvider, int reverseSearch) {
	GList* provider = allProviders;

	while (provider != NULL) {
		BackgroundQueryProvider *backgroundQueryProvider = (BackgroundQueryProvider*) provider -> data;
		if (backgroundQueryProvider == currentProvider) {
			if (reverseSearch) {
				if (provider -> prev != NULL) {
					return (BackgroundQueryProvider*) provider -> prev -> data;
				} else {
					return (BackgroundQueryProvider*) g_list_last(allProviders) -> data;
				}
			} else {
				if (provider -> next != NULL) {
					return (BackgroundQueryProvider*) provider -> next -> data;
				} else {
					return (BackgroundQueryProvider*) allProviders -> data;
				}
			}
			break;
		}
		provider = provider -> next;
	}
}

void addSearchMarkerAtCanvasCenter();
void processKeyDown(SDL_KeyboardEvent key) {
	if (canvas.searchBarActive == 0) {
		switch (key.keysym.sym) {
		case SDLK_b:
			if (key.keysym.mod & KMOD_LSHIFT) {
				clearBuildings();
			} else {
				updateBuildings3d();
			}
			break;
		case SDLK_g:
			options.showGrid = 1 - options.showGrid;
			break;
		case SDLK_h:
			options.showDebugTiles = 1 - options.showDebugTiles;
			break;
		case SDLK_c:
			options.showCoordinates = 1 - options.showCoordinates;
			break;
		case SDLK_SPACE:
			tileEngineGotomypos();
			break;
		case SDLK_x:
			options.accelerometerEnabled = 1 - options.accelerometerEnabled;
			accelerometer.performCalibration = TRUE;
			break;
		case SDLK_z:
			options.offlineMode = 1 - options.offlineMode;
			if (options.offlineMode) {
				addConsoleLine("Off-line mode activated", 0, 1, 0);
			} else {
				addConsoleLine("Off-line mode deactivated", 0, 1, 0);
			}
			break;
		case SDLK_m:
			options.mipmapToggle = 1 - options.mipmapToggle;
			recreateTiles();
			break;
		case SDLK_p:
			if (options.orientation == LANDSCAPE) {
				options.orientation = PORTRAIT;
			} else {
				options.orientation = LANDSCAPE;
			}

			tileEngineChangeOrientation(options.orientation);
			break;
		case SDLK_o:
			addSearchMarkerAtCanvasCenter();
			break;
		case SDLK_v:
			addConsoleLine("CloudGPS ver. " CLOUDGPS_VERSION, 0, 1, 0);
			break;
		case SDLK_KP_ENTER:
		case SDLK_PAGEUP:
			tileEngineZoomIn();
			break;
		case SDLK_PERIOD:
		case SDLK_PAGEDOWN:
			tileEngineZoomOut();
			break;
		case SDLK_BACKSPACE:
			if (canvas.followingMypos) {
				canvas.attraction = canvas.previousCenter;
				canvas.followingMypos = 0;
				canvas.attractionZooming = 0;
				canvas.attractionToPoint = 1;
			}
			break;
		case SDLK_s: {
			options.geocodingProvider = getNextProvider(geocodingProviders, options.geocodingProvider, key.keysym.mod & KMOD_LSHIFT);
			sprintf(strbuf, "New search provider: %s\n", options.geocodingProvider -> name);
			addConsoleLine(strbuf, 0, 1.0, 0);
		}
			break;
		case SDLK_d: {
			canvas.routeFlybyMode = 1;
		}
			break;
		case SDLK_n: {
			GList *provider = tileProviders;

			while (provider != NULL) {
				TileProvider *tileProvider = (TileProvider*) provider -> data;
				if (canvas.provider == tileProvider) {
					if (key.keysym.mod & KMOD_LSHIFT) {
						if (provider -> prev != NULL) {
							canvas.provider = (TileProvider*) provider -> prev -> data;
						} else {
							canvas.provider = (TileProvider*) g_list_last(tileProviders) -> data;
						}
					} else {
						if (provider -> next != NULL) {
							canvas.provider = (TileProvider*) provider -> next -> data;
						} else {
							canvas.provider = (TileProvider*) tileProviders -> data;
						}
					}
					break;
				}
				provider = provider -> next;
			}

			sprintf(strbuf, "New tile provider: %s\n", canvas.provider -> name);
			addConsoleLine(strbuf, 0, 1.0, 0);
			recreateTiles();
		}
			break;
		case SDLK_r: {
			options.routingProvider = getNextProvider(routingProviders, options.routingProvider, key.keysym.mod & KMOD_LSHIFT);
			sprintf(strbuf, "New routing provider: %s\n", options.routingProvider -> name);
			addConsoleLine(strbuf, 0, 1.0, 0);
		}
			break;
		default:
			//			fprintf(stderr, "unhandled key down: key name %s\n", SDL_GetKeyName(key.keysym.sym));
			break;
		}
	} else {
		switch (key.keysym.sym) {
		case SDLK_RETURN:
		case SDLK_KP_ENTER:
			processNewSearchQuery(canvas.searchBar -> getValue().toUtf8().data());
			break;
		case SDLK_LEFT:
			canvas.searchBar -> cursorLeft();
			break;
		case SDLK_RIGHT:
			canvas.searchBar -> cursorRight();
			break;
		case SDLK_BACKSPACE:
			canvas.searchBar -> removeChar();
			break;
		default:
			if (key.keysym.unicode != 0) {
				canvas.searchBar -> addChars(QString::fromUtf16(&key.keysym.unicode, 1));
			}
			break;
		}
	}
}

void processKeyUp(SDL_KeyboardEvent key) {

}

void processKeyboard() {
	if (keys == NULL) {
		keys = SDL_GetKeyState(NULL);
	}

	if (canvas.searchBarActive == 0) {
		if (keys[SDLK_q]) {
			setQuitFlag();
		}
		if (keys[SDLK_LEFT]) {
			if (canvas.viewMode == VIEW_3D) {
				canvas.attractedToRotZ = 0;
				canvas.drotz += .2;
			} else {
				canvas.routeFlybyMode = 0;
				canvas.followingMypos = 0;
				canvas.attractionToPoint = 0;
				canvas.friction = 0.97;
				canvas.dx += 0.2 * sin((canvas.rotz + 90 + 90 * options.orientation) * M_PI / 180.0);
				canvas.dy += 0.2 * cos((canvas.rotz + 90 + 90 * options.orientation) * M_PI / 180.0);
			}
		}
		if (keys[SDLK_RIGHT]) {
			if (canvas.viewMode == VIEW_3D) {
				canvas.attractedToRotZ = 0;
				canvas.drotz -= .2;
			} else {
				canvas.routeFlybyMode = 0;
				canvas.followingMypos = 0;
				canvas.friction = 0.97;

				canvas.attractionToPoint = 0;
				canvas.dx += 0.2 * sin((canvas.rotz - 90 + 90 * options.orientation) * M_PI / 180.0);
				canvas.dy += 0.2 * cos((canvas.rotz - 90 + 90 * options.orientation) * M_PI / 180.0);
			}
		}
		if (keys[SDLK_UP]) {
			canvas.routeFlybyMode = 0;
			canvas.followingMypos = 0;
			canvas.attractionToPoint = 0;
			canvas.friction = 0.97;

			canvas.dx += 0.2 * sin((canvas.rotz + 90 * options.orientation) * M_PI / 180.0);
			canvas.dy += 0.2 * cos((canvas.rotz + 90 * options.orientation) * M_PI / 180.0);
		}
		if (keys[SDLK_DOWN]) {
			canvas.routeFlybyMode = 0;
			canvas.followingMypos = 0;
			canvas.attractionToPoint = 0;
			canvas.friction = 0.97;
			canvas.dx += 0.2 * sin((canvas.rotz + 180 + 90 * options.orientation) * M_PI / 180.0);
			canvas.dy += 0.2 * cos((canvas.rotz + 180 + 90 * options.orientation) * M_PI / 180.0);
		}
		if (keys[SDLK_LSHIFT]) {
			activateZoomBar();
		}
	}
}

void processAccelerometer() {
	//liqaccel_read(&accelerometer.x, &accelerometer.y, &accelerometer.z);

	if (accelerometer.performCalibration) {
		accelerometer.calibrateX = accelerometer.x;
		accelerometer.calibrateY = accelerometer.y;
		accelerometer.calibrateZ = accelerometer.z;
		accelerometer.performCalibration = FALSE;
	}
	tileEngineProcessAccelerometer();
}
