/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <strings.h>
#include <memory.h>
#include <curl/curl.h>
#include <malloc.h>

#include <SDL/SDL.h>
#include <SDL/SDL_image.h>
#include <SDL/SDL_gles.h>
#include <SDL/SDL_ttf.h>
#include <GLES/gl.h>
#include <GLES/glext.h>

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>

#define M_PI		3.14159265358979323846	/* pi */

#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16
#define TILE_SIZE     256

#define PORTRAIT_ROTATION_Y 14
#define LANDSCAPE_ROTATION_X 20

// how many sattelite bars are drawn in status bar
#define MAX_SATTELITES 12

// how many tiles horizontally
#define TILES_X 8

// how many tiles vertically
#define TILES_Y 8

#define MAX_TEXTURES_CREATED_PER_FRAME 1

int tileCoordinateVisibility[TILES_X + 1][TILES_Y + 1];

int quit = FALSE;
long nowMillis = 0, previousMillis = 0, diffMillis = 0;
int downloaded = 0;
int batteryPercentage;
long lastBatteryRefresh = 0, lastGarbageCollection = 0;

volatile int syncLoadedTilesAppearance = FALSE;
int forceGarbageCollection = FALSE;
int texturesCreatedThisFrame = 0;

int currentTilesIdx = 0;
int frame = 0;
GLfloat fps;

typedef enum {
    LANDSCAPE = 0, PORTRAIT = 1
} Orientation;

struct {
    int mipmapToggle;
    int showGrid, showCoordinates, accelerometerEnabled;
    int showDebugTiles;
    int locallyLoadedFadeTime, downloadedFadeTime, zoomChangeFadeTime;

    Orientation orientation;
    char *userAgent;
    char *referer;
} options;

TTF_Font *font = NULL;


typedef struct {
    char* name;
    int minZoom, maxZoom;
    char* filenameFormat;
    char* urlFormat;
} TileProvider;

typedef enum {
    VIEW_2D, VIEW_3D
} ViewMode;

typedef struct {
    double latitude, longitude;
} WorldCoordinate;

typedef struct {
    int tilex, tiley, zoom;
    double x, y;
} TileCoordinate;

struct {
    GLfloat x, y, dx, dy, scale, oldScale, destScale, rotx, roty, rotz, destRotx, destRoty, destRotz, drotz;
    int attractedToRotZ;
    int tilex, tiley;
    int zoom, oldZoom;
    WorldCoordinate center;
    WorldCoordinate previousCenter; // before following mode turned on
    WorldCoordinate attraction; // position to which canvas is attracted (ie current position in following mode)
    TileProvider* provider;
    ViewMode viewMode;
    GLfloat orientationTransition;
    int orientationTransitionLinear;
    int followingMypos;
    int attractionToPoint;
    int rotation2d;
    int searchBarActive;
    int zoomBarActive;
    long zoomBarVisibleMilis;
    long zoomBarLastDragged;
    GLfloat zoomBarAlpha;
    GLfloat zoomKnotPosition;
    GLfloat searchBarY;
    GArray* buildings;
    int buildingsVerticesPrecomputed;
    GLfloat buildingsHeight;
    GLfloat buildingsTileX;
    GLfloat buildingsTileY;
    TileCoordinate boundingTrapezoid[4]; // points in clockwise order
} canvas;

struct {
    double latitude, longitude;
} location;

struct {
    int x, y, oldx, oldy, button, oldButton, xdelta, ydelta;
    long pressed, moved; // milliseconds of last pressed and moved events, -1 if mousebutton is not down
} mouse;

struct {
    int x, y, z;
    int calibrateX, calibrateY, calibrateZ;
    int performCalibration; // if set to TRUE calibration will be performed on next accelerometer read.
} accelerometer;

typedef struct {
    GLfloat r, g, b, a;
} Color;

typedef enum {
    UI_HIDDEN = 0, UI_COMING = 1, UI_SHOWN = 2, UI_GOING = 3
} uiElemStatus;

typedef struct {
    GLuint name;
    int size;
} Texture;

typedef struct {
    GLfloat portraitx, portraity;
    GLfloat landscapex, landscapey;
    GLfloat x, y;
    GLfloat destx, desty;
    long stateChangeMilis;
    uiElemStatus status;
    Texture texture;
    Texture mask;
    Color color;
    void (*handlePressed)();
    void (*handleDragged)();
    GLfloat * texCoords;
} UiElement;

struct {
    char input[100];
    int cursorScreen; // Cursor position on screen
    int cursorInput; // Cursor position in input string
    long lastChange;
} searchBar;

GQueue * downloadQueue;

#include "tile.c"
#include "downloadQueue.c"
GList* uiElems = NULL;

UiElement* pressedUiElem = NULL;

UiElement *crosshair, *zoomIn, *zoomOut, *position, *gotomypos, *view2d, *view3d, *compass, *search, *zoomKnot;

GLfloat modelMatrix[16];
GLfloat projectionMatrix[16];
GLint viewPort[4];

#include "network.c"
#include "texture.c"
#include "tileProviders.c"
#include "glunproject.c"

LocationGPSDevice *device;
LocationGPSDControl *control;

#include "tileengine.c"
#include "glcanvas.c"
#include "accelerometer.c"
#include "geocoder.c"
#include "buildings3d.c"
#include "input.c"
#include "config.c"
#include "location.c"
#include "battery.c"
#include "taskman.c"

int main(int argc, char **argv) {
    SDL_Surface *surface;
    SDL_Event event;

    /* whether or not the window is active */
    int isActive = TRUE;

    /* initialize SDL */
    if (SDL_Init(SDL_INIT_EVERYTHING | SDL_INIT_NOPARACHUTE) < 0) {
        fprintf(stderr, "SDL initialization failed: %s\n", SDL_GetError());
        exit(1);
    }

    if (TTF_Init() == -1) {
        fprintf(stderr, "TTF_Init: %s\n", TTF_GetError());
        exit(2);
    }

    surface = SDL_SetVideoMode(0, 0, 16, SDL_FULLSCREEN);
    SDL_ShowCursor(0);

    SDL_WM_SetCaption("CloudGPS", "cloudgps");

    /* initialize OpenGL */
    initGL();
    loadUI();
    loadConfig();
    initInput();

    initTileEngine();
    initLocation();
    initBattery();
    curl_global_init(CURL_GLOBAL_ALL);

    /* wait for events */
    while (!quit) {
        /* handle the events in the queue */
        previousMillis = nowMillis;
        nowMillis = SDL_GetTicks();
        diffMillis = nowMillis - previousMillis;
        texturesCreatedThisFrame = 0;

        while (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_ACTIVEEVENT:
                    if (event.active.gain == 0) {
                        isActive = FALSE;
                        saveConfig();
                    } else {
                        isActive = TRUE;
                    }
                    break;
                case SDL_QUIT:
                    quit = TRUE;
                    break;
                case SDL_KEYDOWN:
                    processKeyDown(event.key);
                    break;
                case SDL_KEYUP:
                    processKeyUp(event.key);
                    break;
                default:
                    break;
            }
        }
        processMouse();
        processKeyboard();
        processAccelerometer();
        tileEngineUpdateCoordinates();
        setupGlScene();
        tileEngine_computeBoundingTrapezoid();
        tileEngine_handleZoomHelperTiles();
        tileEngine_computeTilesVisibility();
        updateUi();
        updateSyncWaitTiles();

        // TODO detect screen lock
        if (!isActive) {
            // when our window is inactive save system resources by limiting framerate
            SDL_Delay(50);
        }
        drawGLScene();

        // refresh battery percentage and save current view every 10 seconds
        if (nowMillis - lastBatteryRefresh > 10000) {
            refreshBattery();
            lastBatteryRefresh = nowMillis;
            saveConfig();
        }
        if (nowMillis - lastGarbageCollection > 2000 || forceGarbageCollection || g_queue_get_length(allCreatedTiles) > 2 * TILES_X * TILES_Y) {
            processDeferredDeallocation();
            lastGarbageCollection = nowMillis;
            forceGarbageCollection = FALSE;
        }
        frame++;
    }

    /* clean ourselves up and exit */
    saveConfig();
    SDL_GLES_DeleteContext(context);
    shutdownTileEngine();
    shutdownLocation();
    TTF_CloseFont(font);
    TTF_Quit();
    SDL_Quit();
    return 0;
}
