/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <strings.h>
#include <memory.h>
#include <curl/curl.h>

#include <SDL/SDL.h>
#include <SDL/SDL_image.h>
#include <SDL/SDL_gles.h>
#include <GLES/gl.h>

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>

#define M_PI		3.14159265358979323846	/* pi */

#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16
#define TILE_SIZE     256

//#define TRUE  1
//#define FALSE 0

// how many tiles horizontally
#define TILES_X (SCREEN_WIDTH/TILE_SIZE + 2)

// how many tiles vertically
#define TILES_Y (SCREEN_HEIGHT/TILE_SIZE + 2)

//#define DEBUG 0
//#define LOG_INFO 1
//#define LOG_ERROR 2
//
//#define LEVEL 0
//
//SDL_mutex * logMutex = NULL;

//void logmsg(int level, const char* msg, ...) {
////    if (logMutex == NULL) {
////        logMutex = SDL_CreateMutex();
////    }
//    __attribute__ ((msg (printf, 2, 3)));
//    if (level >= LEVEL) {
//        char buf[500];
//        va_list ap;
//        va_start(ap, msg);
//        vsprintf(buf, msg, ap);
//        va_end(ap);
////        SDL_mutexP(logMutex);
//        fprintf(stderr, buf);
////        SDL_mutexV(logMutex);
//    }
//}

int quit = FALSE;
long nowMillis = 0, previousMillis = 0, diffMillis = 0;
int downloaded = 0;
int batteryPercentage;
long lastBatteryRefresh;

int syncLoadedTilesAppearance = 0;

int currentTilesIdx = 0;

typedef enum {
    OSM = 0, CLOUDMADE = 1, GOOGLE = 2, BING = 3, YAHOO = 4
} TileProvider;

typedef enum {
    VIEW_2D, VIEW_3D
} ViewMode;

typedef struct {
    double latitude, longitude;
} WorldCoordinate;

typedef struct {
    int tilex, tiley, zoom;
    double x, y;
} TileCoordinate;

struct {
    GLfloat x, y, dx, dy, scale, oldScale, destScale, rotx, rotz, destRotx, drotz;
    int tilex, tiley;
    int zoom, oldZoom, maxZoom;
    WorldCoordinate center;
    TileProvider provider;
    ViewMode viewMode;
    int showGrid, showCoordinates, accelerometerEnabled;
} canvas;

struct {
    double latitude, longitude;
} location;

struct {
    int x, y, oldx, oldy, button, oldButton, xdelta, ydelta;
    long pressed, moved; // milliseconds of last pressed and moved events, -1 if mousebutton is not down
} mouse;

struct {
    int x, y, z;
    int calibrateX, calibrateY, calibrateZ;
    int performCalibration; // if set to TRUE calibration will be performed on next accelerometer read.
} accelerometer;

typedef struct {
    GLfloat r, g, b, a;
} Color;

typedef enum {
    ANCHOR_LEFT, ANCHOR_UP, ANCHOR_CENTER, ANCHOR_DOWN, ANCHOR_RIGHT
} Anchor;

typedef enum {
    UI_HIDDEN = 0, UI_COMING = 1, UI_SHOWN = 2, UI_GOING = 3
} uiElemStatus;

typedef struct {
    GLuint name;
    int size;
} Texture;

typedef struct {
    GLfloat x, y;
    long stateChangeMilis;
    uiElemStatus status;
    Texture texture;
    Texture mask;
    Color color;
    void (*handlePressed)();
    void (*handleDragged)();
} UiElement;

#include "queue.c"
queue* uiElems;

UiElement* pressedUiElem = NULL;

UiElement *crosshair, *zoomIn, *zoomOut, *position, *gotomypos, *view2d, *view3d;

#include "tile.c"
t_tile* tiles[TILES_X][TILES_Y][2];

#include "network.c"
#include "texture.c"
#include "tileProviders.c"

LocationGPSDevice *device;
LocationGPSDControl *control;

#include "tileengine.c"
#include "glcanvas.c"
#include "input.c"
#include "config.c"
#include "location.c"
#include "battery.c"
#include "taskman.c"
#include "accelerometer.c"

int main(int argc, char **argv) {
    SDL_Surface *surface;
    SDL_Event event;

    /* whether or not the window is active */
    int isActive = TRUE;

    /* initialize SDL */
    if (SDL_Init(SDL_INIT_EVERYTHING | SDL_INIT_NOPARACHUTE) < 0) {
        fprintf(stderr, "SDL initialization failed: %s\n", SDL_GetError());
        exit(1);
    }

    surface = SDL_SetVideoMode(0, 0, 16, SDL_FULLSCREEN);
    SDL_ShowCursor(0);

    SDL_WM_SetCaption("CloudGPS", "cloudgps");

    /* initialize OpenGL */
    initGL();
    loadUI();
    loadConfig();
    initInput();

    initTileEngine();
    initLocation();
    initBattery();
    curl_global_init(CURL_GLOBAL_ALL);
    // TEMPORARY START
    //    runTests();
    //    return 0;
    // TEMPORARY END

    /* wait for events */
    while (!quit) {
        /* handle the events in the queue */
        previousMillis = nowMillis;
        nowMillis = SDL_GetTicks();
        diffMillis = nowMillis - previousMillis;

        while (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_ACTIVEEVENT:
                    if (event.active.gain == 0) {
                        isActive = FALSE;
                        saveConfig();
                    } else {
                        isActive = TRUE;
                    }
                    break;
                case SDL_QUIT:
                    quit = TRUE;
                    break;
                case SDL_USEREVENT:
                    handleGenTextureEvent(&event);
                    break;
                case SDL_KEYDOWN:
                    processKeyDown(event.key);
                    break;
                case SDL_KEYUP:
                    processKeyUp(event.key);
                    break;
                default:
                    break;
            }
        }
        processMouse();
        processKeyboard();
        processAccelerometer();
        tileEngineUpdateCoordinates();
        updateUi();
        updateSyncWaitTiles();
        if (!isActive) {
            // when our window is inactive save system resources by limiting framerate to 10 fps
            SDL_Delay(100);
        }
        drawGLScene();

        if (nowMillis - lastBatteryRefresh > 10000) {
            refreshBattery();
            lastBatteryRefresh = nowMillis;
        }
    }

    /* clean ourselves up and exit */
    // wait for all created threads here
    saveConfig();
    shutdownTileEngine();
    shutdownLocation();
    SDL_GLES_DeleteContext(context);
    SDL_Quit();
    return 0;
}
