/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#define DOWNLOADING_THREADS 2

SDL_Thread* downloadThreads[DOWNLOADING_THREADS];

int downloadThread(void* queue);
void tileEngine_computeBoundingTrapezoid();
void tileEngine_computeTilesVisibility();

int inline getCanvasShiftX() {
    return (TILES_X - (SCREEN_WIDTH / TILE_SIZE)) / 2;
}

int inline getCanvasShiftY() {
    return (TILES_Y - (SCREEN_HEIGHT / TILE_SIZE)) / 2;
}

void toScreenCoordinate(TileCoordinate *tc, GLfloat *x, GLfloat *y) {
    *x = (tc->tilex - canvas.tilex - getCanvasShiftX()) * TILE_SIZE + tc->x;
    *y = (tc->tiley - canvas.tiley - getCanvasShiftY()) * TILE_SIZE + tc->y;
}

void toTileCoordinate(WorldCoordinate* wc, TileCoordinate* result, int zoom) {
    result -> zoom = zoom;

    double p = pow(2.0, zoom);
    double lat = (wc->longitude + 180.) / 360.0 * p;
    result -> tilex = (int) lat;
    result ->x = (lat - result->tilex) * TILE_SIZE;

    double lon = (1.0 - log(tan(wc->latitude * M_PI / 180.0) + 1.0 / cos(wc->latitude * M_PI / 180.0)) / M_PI) / 2.0 * p;
    result->tiley = (int) lon;
    result ->y = (lon - result->tiley) * TILE_SIZE;
}

void toWorldCoordinate(TileCoordinate* tc, WorldCoordinate* result) {
    double p = pow(2.0, tc->zoom);
    double xx = (double) tc->tilex + (tc->x / (double) TILE_SIZE);
    result->longitude = ((xx * 360.0) / p) - 180.0;

    double yy = tc->tiley + tc->y / (double) TILE_SIZE;
    double n = M_PI - 2.0 * M_PI * yy / p;
    double m = 0.5 * (exp(n) - exp(-n));
    double a = atan(m);
    result->latitude = (180.0 / M_PI * a);
}

void canvasCenterToTileCoordinate(TileCoordinate* tc) {
    tc -> zoom = canvas.zoom;
    tc -> tilex = canvas.tilex;
    tc -> tiley = canvas.tiley;
    tc -> x = (TILE_SIZE - canvas.x) + SCREEN_WIDTH / 2;
    tc -> y = (TILE_SIZE - canvas.y) + SCREEN_HEIGHT / 2;
}

t_tile* createTile(int i, int j) {
    t_tile * tile = (t_tile*) calloc(1, sizeof(t_tile));
    if (tile == NULL) {
        fprintf(stderr, "Memory allocation error in createTile()\n");
        exit(1);
    }
    tile->tilex = canvas.tilex + i;
    tile->tiley = canvas.tiley + j;
    tile->zoom = canvas.zoom;
    tile->provider = canvas.provider;
    return tile;
}

void initTileEngine() {
    int i, j;

    downloadQueue = createQueue();
    deallocationQueue = g_queue_new();

    canvas.scale = 1.0;
    canvas.viewMode = VIEW_2D;

    for (i = 0; i < DOWNLOADING_THREADS; i++) {
        //        fprintf(stderr, "before SDL_CreateThread\n");
        downloadThreads[i] = SDL_CreateThread(downloadThread, (void*) downloadQueue);
        //        fprintf(stderr, "after SDL_CreateThread\n");
        SDL_Delay(15);
    }

    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            tiles[i][j][currentTilesIdx] = createTile(i, j);
        }
    }
    tileEngine_computeBoundingTrapezoid();
    tileEngine_computeTilesVisibility();
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
        }
    }
}

int inline roundDown(double x) {
    return x < 0 ? (int) (x - 1.0) : (int) x;
}

int adjustShifts() {
    int horizontalShift = roundDown(canvas.x / (double) TILE_SIZE);
    int verticalShift = roundDown(canvas.y / (double) TILE_SIZE);

    canvas.x -= horizontalShift * TILE_SIZE;
    canvas.y -= verticalShift * TILE_SIZE;

    canvas.tilex += horizontalShift;
    canvas.tiley += verticalShift;
    if (horizontalShift != 0 || verticalShift != 0) {
        return TRUE;
    }
    return FALSE;
}

void recreateTiles() {
    purgeQueue(downloadQueue);
    int i, j;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            if (tiles[i][j][currentTilesIdx] != NULL) {
                deallocateTile(tiles[i][j][currentTilesIdx]);
                tiles[i][j][currentTilesIdx] = NULL;
            }
        }
    }
    // start loading new tile textures:
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            tiles[i][j][currentTilesIdx] = createTile(i, j);
        }
    }
    tileEngine_computeBoundingTrapezoid();
    tileEngine_computeTilesVisibility();
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
        }
    }
}

void handleZoomChange() {
    // compute new tilex, tiley, x and y for canvas using canvas.center
    int i, j;

    TileCoordinate tc;

    toTileCoordinate(&canvas.center, &tc, canvas.zoom);

    if (canvas.zoom - canvas.oldZoom == 1) {
        canvas.tilex = tc.tilex + getCanvasShiftX();
        canvas.tiley = tc.tiley + getCanvasShiftY();
        canvas.x = tc.x - SCREEN_WIDTH / 2;
        canvas.y = tc.y - SCREEN_HEIGHT / 2;
    } else {
        canvas.tilex = tc.tilex - getCanvasShiftX();
        canvas.tiley = tc.tiley - getCanvasShiftY();
        // TODO - where do those number (-16 +32) come from?
        canvas.x = tc.x - 16;
        canvas.y = tc.y + SCREEN_HEIGHT / 2 + 32;
    }

    adjustShifts();

    if (canvas.x != 0.0) {
        canvas.x = TILE_SIZE - canvas.x;
    }
    if (canvas.y != 0.0) {
        canvas.y = TILE_SIZE - canvas.y;
    }

    syncLoadedTilesAppearance = TRUE;

    if (canvas.zoom - canvas.oldZoom == 1) {
        //        fprintf(stderr, "zoomIn 1\n");
        int oldtilex = tiles[0][0][currentTilesIdx] -> tilex;
        int oldtiley = tiles[0][0][currentTilesIdx] -> tiley;

        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                tiles[i][j][1 - currentTilesIdx] = createTile(i, j);

                int oldi = tiles[i][j][1 - currentTilesIdx] -> tilex - (oldtilex * 2);
                int oldj = tiles[i][j][1 - currentTilesIdx] -> tiley - (oldtiley * 2);

                if (oldi >= 0 && oldj >= 0) {

//                    if (tiles[oldi / 2][oldj / 2][currentTilesIdx] != NULL && tiles[oldi / 2][oldj / 2][currentTilesIdx] -> state == STATE_LOADED) {
//                        tiles[i][j][1 - currentTilesIdx] -> state = STATE_SCALED_LOADING;
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureSize = TILE_SIZE;
//
//                        tiles[i][j][1 - currentTilesIdx] -> oldTexture[0] = tiles[oldi / 2][oldj / 2][currentTilesIdx] -> texture;
//                        tiles[i][j][1 - currentTilesIdx] -> oldPixels4444[0] = tiles[oldi / 2][oldj / 2][currentTilesIdx] ->pixels4444;
//
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][0] = (oldi % 2) * 0.5;
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][1] = (oldj % 2) * 0.5;
//
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][2] = ((oldi % 2) + 1) * 0.5;
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][3] = (oldj % 2) * 0.5;
//
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][4] = (oldi % 2) * 0.5;
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][5] = ((oldj % 2) + 1) * 0.5;
//
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][6] = ((oldi % 2) + 1) * 0.5;
//                        tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][7] = ((oldj % 2) + 1) * 0.5;
//                        tiles[i][j][1 - currentTilesIdx] -> transitionTime = 1500;
//                        enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
//                    } else {
                        deallocateTile(tiles[oldi / 2][oldj / 2][currentTilesIdx]);
                        tiles[oldi / 2][oldj / 2][currentTilesIdx] = NULL;
                        tiles[i][j][1 - currentTilesIdx] -> transitionTime = 1500;
                        enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
//                    }
                }
            }
        }
        currentTilesIdx = 1 - currentTilesIdx;
    } else if (canvas.zoom - canvas.oldZoom == -1) {
        int oldtilex = tiles[0][0][currentTilesIdx] -> tilex;
        int oldtiley = tiles[0][0][currentTilesIdx] -> tiley;
        int ii, jj;

        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                tiles[i][j][1 - currentTilesIdx] = createTile(i, j);

                for (ii = 0; ii < 2; ii++) {
                    for (jj = 0; jj < 2; jj++) {
                        int oldi = (tiles[i][j][1 - currentTilesIdx] -> tilex - (oldtilex / 2)) * 2 + ii - (oldtilex % 2);
                        int oldj = (tiles[i][j][1 - currentTilesIdx] -> tiley - (oldtiley / 2)) * 2 + jj - (oldtiley % 2);

                        if (oldi >= 0 && oldi < TILES_X && oldj >= 0 && oldj < TILES_Y) {
                            if (tiles[oldi][oldj][currentTilesIdx] != NULL) {
                                if (tiles[oldi][oldj][currentTilesIdx] -> state == STATE_LOADED) {
                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureSize = TILE_SIZE / 2;
                                    tiles[i][j][1 - currentTilesIdx] -> state = STATE_SCALED_LOADING;

                                    tiles[i][j][1 - currentTilesIdx] -> oldTexture[ii * 2 + jj] = tiles[oldi][oldj][currentTilesIdx] -> texture;
                                    tiles[i][j][1 - currentTilesIdx] -> oldPixels4444[ii * 2 + jj] = tiles[oldi][oldj][currentTilesIdx] -> pixels4444;

                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][0] = 0.0;
                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][1] = 0.0;

                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][2] = 1.0;
                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][3] = 0.0;

                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][4] = 0.0;
                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][5] = 1.0;

                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][6] = 1.0;
                                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][7] = 1.0;
                                }
                            }
                        }
                    }
                }
                tiles[i][j][1 - currentTilesIdx] -> transitionTime = 1500;
                enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
            }
        }
        currentTilesIdx = 1 - currentTilesIdx;
    } else {
        recreateTiles();
    }
}

int inline getLowerDeallocateBound(int size, int shift) {
    return shift < 0 ? 0 : (size - shift) < 0 ? 0 : (size - shift);
}

int inline getUpperDeallocateBound(int size, int shift) {
    return shift < 0 ? (-shift < size ? -shift : size) : size;
}

void deallocateTiles(int horizontalShift, int verticalShift) {

    int i, j;
    for (i = getLowerDeallocateBound(TILES_X, horizontalShift); i < getUpperDeallocateBound(TILES_X, horizontalShift); i++) {
        for (j = 0; j < TILES_Y; j++) {
            deallocateTile(tiles[i][j][currentTilesIdx]);
            tiles[i][j][currentTilesIdx] = NULL;
        }
    }
    for (i = 0; i < TILES_X; i++) {
        for (j = getLowerDeallocateBound(TILES_Y, verticalShift); j < getUpperDeallocateBound(TILES_Y, verticalShift); j++) {
            deallocateTile(tiles[i][j][currentTilesIdx]);
            tiles[i][j][currentTilesIdx] = NULL;
        }
    }
}

void updateTiles(int horizontalShift, int verticalShift) {
    deallocateTiles(horizontalShift, verticalShift);
    int i, j;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            int newPosX = i - horizontalShift;
            int newPosY = j - verticalShift;
            if (newPosX >= 0 && newPosX < TILES_X && newPosY >= 0 && newPosY < TILES_Y) {
                tiles[i][j][1 - currentTilesIdx] = tiles[newPosX][newPosY][currentTilesIdx];
                tiles[newPosX][newPosY][currentTilesIdx] = NULL;
            } else {
                tiles[i][j][1 - currentTilesIdx] = createTile(i, j);
                enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
            }
        }
    }
    currentTilesIdx = 1 - currentTilesIdx;
}

void tileEngineProcessMouse(int uiElemPressed) {
    canvas.dx *= 0.97;
    canvas.dy *= 0.97;

    if (uiElemPressed == FALSE) {
        if (mouse.button) {
            canvas.followingMypos = FALSE;
        }

        if (canvas.viewMode == VIEW_3D) {
            if (options.orientation == PORTRAIT && mouse.x < 150) {
                canvas.drotz -= mouse.ydelta / 18.0;
                return;
            }
            if (options.orientation == LANDSCAPE && mouse.y < 150) {
                canvas.drotz += mouse.xdelta / 25.0;
                return;
            }
        }

        canvas.dx += mouse.ydelta / 10.0 * sin(canvas.rotz * M_PI / 180.0);
        canvas.dy += mouse.ydelta / 10.0 * cos(canvas.rotz * M_PI / 180.0);
        canvas.dx += mouse.xdelta / 10.0 * sin((canvas.rotz + 90) * M_PI / 180.0);
        canvas.dy += mouse.xdelta / 10.0 * cos((canvas.rotz + 90) * M_PI / 180.0);

        if (mouse.pressed > 0 && nowMillis - mouse.pressed > 30) {
            canvas.dx *= 0.97;
            canvas.dy *= 0.97;
        }

        if (mouse.pressed > 0 && nowMillis - mouse.moved > 50) {
            canvas.dx *= 0.2;
            canvas.dy *= 0.2;
        }
    }
}
void tileEngineChangeOrientation(Orientation orientation);

void tileEngineProcessAccelerometer() {
    if (options.accelerometerEnabled == 1) {

        if (canvas.viewMode == VIEW_2D) {
            if (options.orientation == LANDSCAPE) {
                if (abs(accelerometer.x) > 70) {
                    canvas.dx += accelerometer.x / 1000.0;
                }
                if (abs(accelerometer.y) > 70) {
                    canvas.dy += accelerometer.y / 1000.0;
                }
            } else {
                if (abs(accelerometer.y) > 70) {
                    canvas.dx -= accelerometer.y / 1000.0;
                }
                if (abs(accelerometer.x) > 70) {
                    canvas.dy += accelerometer.x / 1000.0;
                }
            }
        } else {
            if (options.orientation == LANDSCAPE) {
                if (abs(accelerometer.y) > 70) {
                    canvas.dx += accelerometer.y / 1000.0 * sin(canvas.rotz * M_PI / 180.0);
                    canvas.dy += accelerometer.y / 1000.0 * cos(canvas.rotz * M_PI / 180.0);
                }
            } else {
                if (abs(accelerometer.x) > 70) {
                    canvas.dx += accelerometer.x / 1000.0 * sin((canvas.rotz + 90) * M_PI / 180.0);
                    canvas.dy += accelerometer.x / 1000.0 * cos((canvas.rotz + 90) * M_PI / 180.0);
                }
            }
            if (options.orientation == LANDSCAPE) {
                canvas.drotz += accelerometer.x / 2000.0;
            } else {
                canvas.drotz -= accelerometer.y / 2000.0;
            }
        }
    }
    if (accelerometer.y + accelerometer.calibrateY < -850) {
        tileEngineChangeOrientation(LANDSCAPE);
    }
    if (accelerometer.x + accelerometer.calibrateX < -850) {
        tileEngineChangeOrientation(PORTRAIT);
    }
}

void tileEngineUpdateCoordinates() {
    //int zoomChange = (int) round(log2((double) canvas.scale));
    int zoomChange = 0;

    canvas.x += canvas.dx;
    canvas.y += canvas.dy;

    if (canvas.followingMypos) {
        if (device -> fix -> fields | LOCATION_GPS_DEVICE_LATLONG_SET) {
            GLfloat x, y;
            WorldCoordinate wc;
            wc.latitude = device -> fix -> latitude;
            wc.longitude = device -> fix -> longitude;
            TileCoordinate tc;
            toTileCoordinate(&wc, &tc, canvas.zoom);
            toScreenCoordinate(&tc, &x, &y);
            //            fprintf(stderr, "x = %f, y = %f\t\ttilex = %d, tiley = %d, tc.x = %f, tc.y = %f \n", x, y, tc.tilex, tc.tiley, tc.x, tc.y);
            canvas.dx = (SCREEN_WIDTH / 2.0 - (canvas.x - TILE_SIZE + x)) / 20.0;
            canvas.dy = (SCREEN_HEIGHT / 2.0 - (canvas.y - TILE_SIZE + y)) / 20.0;
            if (canvas.viewMode == VIEW_3D) {
                if ((device -> fix -> fields | LOCATION_GPS_DEVICE_TRACK_SET) && device -> satellites_in_use > 0 && device -> fix -> speed > 7) {
                    canvas.rotz += (device -> fix -> track - canvas.rotz) * 0.03;
                }
            }
        }
    }

    if (canvas.destScale != 0.0) {
        GLfloat tmp = canvas.destScale - canvas.scale;
        canvas.oldScale = canvas.scale;
        canvas.scale += tmp / 5;

        if (fabs(tmp) < 0.001) {
            canvas.scale = canvas.destScale;
            canvas.destScale = 0.0;
        }
    } else {
        if (canvas.scale < 1.0) {
            zoomChange = -1;
        }
        if (canvas.scale >= 2.0) {
            zoomChange = 1;
        }
    }

    if ((canvas.zoom + zoomChange) < canvas.provider -> minZoom || (canvas.zoom + zoomChange) > canvas.provider -> maxZoom) {
        zoomChange = 0;
    }

    if (zoomChange == 0) {
        int horizontalShift = roundDown(canvas.x / TILE_SIZE);
        int verticalShift = roundDown(canvas.y / TILE_SIZE);

        if (horizontalShift != 0 || verticalShift != 0) {
            canvas.x -= horizontalShift * TILE_SIZE;
            canvas.y -= verticalShift * TILE_SIZE;

            canvas.tilex -= horizontalShift;
            canvas.tiley -= verticalShift;
            updateTiles(horizontalShift, verticalShift);
        }
    }

    TileCoordinate tc;
    canvasCenterToTileCoordinate(&tc);

    int horizontalShift = roundDown(tc.x / (double) TILE_SIZE);
    int verticalShift = roundDown(tc.y / (double) TILE_SIZE);

    tc.x -= horizontalShift * TILE_SIZE;
    tc.y -= verticalShift * TILE_SIZE;

    tc.tilex += horizontalShift;
    tc.tiley += verticalShift;

    toWorldCoordinate(&tc, &canvas.center);

    if (zoomChange != 0 /* && nowMillis - lastZoomChangeMilis > 500 */) {
        canvas.oldZoom = canvas.zoom;
        canvas.zoom += zoomChange;
        handleZoomChange();
        canvas.scale /= pow(2.0, zoomChange);
    }

    canvas.rotx += (canvas.destRotx - canvas.rotx) * 0.03;
    canvas.roty += (canvas.destRoty - canvas.roty) * 0.03;
    canvas.orientationTransition += ((GLfloat) options.orientation - canvas.orientationTransition) * 0.03;
    if (options.orientation == LANDSCAPE) {
        if (canvas.orientationTransitionLinear > 0) {
            canvas.orientationTransitionLinear--;
        }
    } else {
        if (canvas.orientationTransitionLinear < 160) {
            canvas.orientationTransitionLinear++;
        }
    }
    if (canvas.viewMode == VIEW_2D) {
        canvas.rotz += (canvas.destRotz - canvas.rotz) * 0.03;
    } else {
        canvas.rotz += canvas.drotz;
        canvas.drotz *= 0.9;
        if (canvas.rotz > 180) {
            canvas.rotz -= 360;
        }
        if (canvas.rotz < -180) {
            canvas.rotz += 360;
        }
    }
    if (canvas.orientationTransition > 0.999) {
        canvas.orientationTransition = 1.0;
    } else if (canvas.orientationTransition < 0.001) {
        canvas.orientationTransition = 0.0;
    }
}

void tileEngineZoomIn() {
    if (SDL_GetModState() & KMOD_SHIFT) {
        canvas.oldScale = canvas.scale;
        canvas.scale *= 1.005;
    } else {
        if (canvas.destScale == 0.0) {
            if (canvas.scale <= 0.5) {
                canvas.destScale = 1.0;
            } else {
                canvas.destScale = 2.0;
            }
        }
    }
}

void tileEngineZoomOut() {
    if (SDL_GetModState() & KMOD_SHIFT) {
        canvas.oldScale = canvas.scale;
        canvas.scale /= 1.005;
    } else {
        if (canvas.destScale == 0.0) {
            if (canvas.scale >= 2.0) {
                canvas.destScale = 1.0;
            } else {
                canvas.destScale = 0.5;
            }
        }
    }
}

void tileEngineGotomypos() {
    if (mouse.oldButton == 0) {
        canvas.followingMypos = 1 - canvas.followingMypos;
    }
}

void tileEngineViewMode2D() {
    view2d -> status = UI_HIDDEN;
    view3d -> status = UI_SHOWN;
    canvas.viewMode = VIEW_2D;
    if (options.orientation == LANDSCAPE) {
        canvas.destRotx = 0.0;
        canvas.destRoty = 0.0;
        canvas.destRotz = 0.0;
    } else {
        canvas.destRotx = 0.0;
        canvas.destRoty = 0.0;
        canvas.destRotz = -90.0;
    }
}

void tileEngineViewMode3D() {
    view2d -> status = UI_SHOWN;
    view3d -> status = UI_HIDDEN;
    canvas.viewMode = VIEW_3D;
    if (options.orientation == LANDSCAPE) {
        canvas.destRotx = LANDSCAPE_ROTATION_X;
        canvas.destRoty = 0.0;
        canvas.destRotz = 0.0;
    } else {
        canvas.destRotx = 0.0;
        canvas.destRoty = -PORTRAIT_ROTATION_Y;
        canvas.destRotz = 0.0;
    }
}

void tileEngineChangeOrientation(Orientation orientation) {
    options.orientation = orientation;
    if (canvas.viewMode == VIEW_2D) {
        tileEngineViewMode2D();
    } else {
        tileEngineViewMode3D();
    }
}

void shutdownTileEngine() {
    int i;
    for (i = 0; i < DOWNLOADING_THREADS; i++) {
        //        fprintf(stderr, "waiting for download thread %d to finish...\n", SDL_GetThreadID(downloadThreads[i]));
        SDL_WaitThread(downloadThreads[i], NULL);
    }
}

void tileEngine_computeBoundingTrapezoid() {
    GLfloat posx, posy, posz;

    // tile plane z / (far plane z - near plane z)
    // tile plane z / (far plane z - near plane z)
    GLfloat z1 = 350.0 / (600.0 - 50.0);
    GLfloat z = z1 - canvas.roty / 15.0;
    if (z > 1.0) {
        z = 1.0;
    }

    gluUnProject(0, 0, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);
    canvas.boundingTrapezoid[0].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[0].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[0].x = posx;
    canvas.boundingTrapezoid[0].y = posy;
    canvas.boundingTrapezoid[0].zoom = canvas.zoom;

    z = z1;

    gluUnProject(SCREEN_WIDTH - 1, 0, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);

    canvas.boundingTrapezoid[1].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[1].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[1].x = posx;
    canvas.boundingTrapezoid[1].y = posy;
    canvas.boundingTrapezoid[1].zoom = canvas.zoom;

    z = z1 + canvas.rotx / 40.0;
    if (z > 1.0) {
        z = 1.0;
    }

    gluUnProject(SCREEN_WIDTH - 1, SCREEN_HEIGHT - 1, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);

    canvas.boundingTrapezoid[2].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[2].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[2].x = posx;
    canvas.boundingTrapezoid[2].y = posy;
    canvas.boundingTrapezoid[2].zoom = canvas.zoom;

    z = z1 + canvas.rotx / 40.0 - canvas.roty / 15.0;
    if (z > 1.0) {
        z = 1.0;
    }

    gluUnProject(0, SCREEN_HEIGHT - 1, z, modelMatrix, projectionMatrix, viewPort, &posx, &posy, &posz);
    canvas.boundingTrapezoid[3].tilex = canvas.tilex + getCanvasShiftX();
    canvas.boundingTrapezoid[3].tiley = canvas.tiley + getCanvasShiftY();
    canvas.boundingTrapezoid[3].x = posx;
    canvas.boundingTrapezoid[3].y = posy;
    canvas.boundingTrapezoid[3].zoom = canvas.zoom;
}

/*
 * Compute the side of point (x, y) in relation to line segment (x0, y0) - (x1, y1)
 * returns <0 when point is on right side of line segment
 * returns 0 when point is belongs to line
 * return >0 when point is on left side
 */
GLfloat computeSide(GLfloat x0, GLfloat y0, GLfloat x1, GLfloat y1, GLfloat x, GLfloat y) {
    GLfloat result = (y1 - y) * (x1 - x0) - (x1 - x) * (y1 - y0);
    //    fprintf(stderr, "    %f %f %f %f %f %f, result = %f\n", x0, y0, x1, y1, x, y, result);
    return result;
}

int tileEngine_isTileCoordinateInBoundingTrapezoid(int tilex, int tiley) {
    int i;

    for (i = 0; i < 4; i++) {
        if (computeSide(canvas.boundingTrapezoid[i].tilex + canvas.boundingTrapezoid[i].x / (GLfloat) TILE_SIZE, canvas.boundingTrapezoid[i].tiley
                + canvas.boundingTrapezoid[i].y / (GLfloat) TILE_SIZE, canvas.boundingTrapezoid[(i + 1) % 4].tilex + canvas.boundingTrapezoid[(i + 1)
                % 4].x / (GLfloat) TILE_SIZE, canvas.boundingTrapezoid[(i + 1) % 4].tiley + canvas.boundingTrapezoid[(i + 1) % 4].y
                / (GLfloat) TILE_SIZE, tilex, tiley) < 0) {
            //            fprintf(stderr, "    return FALSE\n");
            return FALSE;
        }
    }
    //    fprintf(stderr, "    return TRUE\n");
    return TRUE;
}

void tileEngine_switchTileProvider() {
}

void tileEngine_computeTilesVisibility() {
    int i, j;
    int texturesCreateThisFrame = 0;

    // tiles share coordinates so it is more efficient to precalculate visibility for each unique coordinate.
    for (i = 0; i < TILES_X + 1; i++) {
        for (j = 0; j < TILES_Y + 1; j++) {
            tileCoordinateVisibility[i][j] = tileEngine_isTileCoordinateInBoundingTrapezoid(canvas.tilex + i, canvas.tiley + j);
        }
    }
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            if (tileCoordinateVisibility[i][j] || tileCoordinateVisibility[i + 1][j] || tileCoordinateVisibility[i][j + 1]
                    || tileCoordinateVisibility[i + 1][j + 1]) {
                tiles[i][j][currentTilesIdx] -> visible = TRUE;
                if (tiles[i][j][currentTilesIdx] -> state == STATE_GL_TEXTURE_NOT_CREATED && texturesCreateThisFrame < 1) {
                    tiles[i][j][currentTilesIdx]->texture = createTexture(tiles[i][j][currentTilesIdx] -> pixels4444, TILE_SIZE, TILE_SIZE, TRUE);
                    if (syncLoadedTilesAppearance == 1) {
                        tiles[i][j][currentTilesIdx] -> state = STATE_LOADED_SYNC_WAIT;
                    } else {
                        tiles[i][j][currentTilesIdx] -> state = STATE_LOADED;
                        tiles[i][j][currentTilesIdx] ->stateChangeTime = 0;
                    }
                    texturesCreateThisFrame++;
                }
            } else {
                tiles[i][j][currentTilesIdx] -> visible = FALSE;
                if (tiles[i][j][currentTilesIdx] -> state == STATE_LOADED || tiles[i][j][currentTilesIdx] -> state == STATE_LOADED_SYNC_WAIT) {
                    glDeleteTextures(1, &tiles[i][j][currentTilesIdx] -> texture);
                    tiles[i][j][currentTilesIdx] -> texture = 0;
                    tiles[i][j][currentTilesIdx] -> state = STATE_GL_TEXTURE_NOT_CREATED;
                }
            }
        }
    }
}
