/*!
** @file	flash_button.cpp
**
** @brief	Custom button widget for flash
**
*/
/*---------------------------------------------------------------------------
** Includes
*/
#include "flashbutton.h"
#include "ClassicPrintSettings.h"

/*---------------------------------------------------------------------------
** Defines and Macros
*/

/*---------------------------------------------------------------------------
** Typedefs
*/
//!
//! Structure that contains information about a flash mode
//!
struct FLASHBUTTON_MODE {
	CAMERA_FLASH_MODE	mode;
	const char*			icon_path;
};

/*---------------------------------------------------------------------------
** Local function prototypes
*/

/*---------------------------------------------------------------------------
** Data
*/
static FLASHBUTTON_MODE	flashbutton_modes[] = {
	{ CAMERA_FLASH_MODE_AUTO, "camera_flash_auto.png" },
	{ CAMERA_FLASH_MODE_OFF, "camera_flash_off.png" },
	{ CAMERA_FLASH_MODE_ON, "camera_flash_fill.png" }
};

static const int flashbutton_num_modes = sizeof(flashbutton_modes) / sizeof(FLASHBUTTON_MODE);

//--------------------------------------------------------------------------
/*!
** @brief   	Constructor
*/
FlashButton::FlashButton(QWidget* parent) : QToolButton(parent) {
	m_flash_mode = 0;

	// Set the icon for the current mode
	QString icon(ClassicPrintSettings::config_dir() + QString("/icons/") +
				 flashbutton_modes[m_flash_mode].icon_path);
	setIcon(QIcon(icon));

	// Handle button presses
	connect(this, SIGNAL(clicked(bool)), this, SLOT(clicked(bool)));
}

//--------------------------------------------------------------------------
/*!
** @brief   	Destructor
*/
FlashButton::~FlashButton() {
}
	
//--------------------------------------------------------------------------
/*!
** @brief		Slot that is called when the user selects the button
**
*/
void FlashButton::clicked(bool checked) {
	Q_UNUSED(checked)

	// Loop around the flash modes
	if (++m_flash_mode == flashbutton_num_modes) {
		m_flash_mode = 0;
	}
	// Update the icon
	QString icon(ClassicPrintSettings::config_dir() + QString("/icons/") +
				 flashbutton_modes[m_flash_mode].icon_path);
	setIcon(QIcon(icon));

	// Tell any interested parties about the change
	emit flash_mode_changed(flashbutton_modes[m_flash_mode].mode);
}

//--------------------------------------------------------------------------
/*!
** @brief		Set the button to show the current flash mode
**
** @param[In]	New flash mode
**
*/
void FlashButton::set_flash_mode(CAMERA_FLASH_MODE new_flash_mode) {
	m_flash_mode = new_flash_mode;
	// Update the icon
	QString icon(ClassicPrintSettings::config_dir() + QString("/icons/") +
				 flashbutton_modes[m_flash_mode].icon_path);
	setIcon(QIcon(icon));
}

