#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QtImageFilter>
#include <QtImageFilterFactory>
#include <QFileDialog>
#include <QtConcurrentRun>

#include "ClassicPrintSettings.h"
#include "ClassicPrint.h"
#include "ClassicPrintLens.h"
#include "ClassicPrintFilm.h"
#include "ClassicPrintProcessing.h"

#include "lensconfiguration.h"
#include "filmconfiguration.h"
#include "processingconfiguration.h"

#include "saveimagedialog.h"

#include "LevelsFilter.h"
#include "VignetteFilter.h"
#include "FrameFilter.h"
#include "NoiseFilter.h"
#include "ContrastFilter.h"
#include "ColourLookupFilter.h"
#include "BlendFilter.h"

#define IMAGE_WIDTH     520
#define IMAGE_HEIGHT    370

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    REGISTER_LEVELS_FILTER;
    REGISTER_VIGNETTE_FILTER;
    REGISTER_FRAME_FILTER;
    REGISTER_NOISE_FILTER;
    REGISTER_CONTRAST_FILTER;
    REGISTER_COLOUR_LOOKUP_FILTER;
    REGISTER_BLEND_FILTER;

    // Load the current photo settings from the config folder
	load_settings();

    ui->setupUi(this);

	// Connect the signal from the photo control to the file open slot
	connect(ui->photo, SIGNAL(openPhoto()), SLOT(on_actionOpen_Image_triggered()));

    ui->photo->setText("");

    QIcon film_icon;
    QIcon lens_icon;
    QIcon processing_icon;
    QIcon share_icon;
    QIcon save_icon;

    film_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Film.png", QSize(), QIcon::Normal);
    film_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Film_dimmed.png", QSize(), QIcon::Disabled);
    lens_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Lens.png", QSize(), QIcon::Normal);
    lens_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Lens_dimmed.png", QSize(), QIcon::Disabled);
    processing_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Processing.png", QSize(), QIcon::Normal);
    processing_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Processing_dimmed.png", QSize(), QIcon::Disabled);
    share_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/general_share.png", QSize(), QIcon::Normal);
    share_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/general_share_dimmed.png", QSize(), QIcon::Disabled);
    save_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/notes_save.png", QSize(), QIcon::Normal);
    save_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/notes_save_dimmed.png", QSize(), QIcon::Disabled);

    ui->btn_edit_film->setIcon(film_icon);
    ui->btn_edit_lens->setIcon(lens_icon);
    ui->btn_edit_processing->setIcon(processing_icon);
    ui->btn_share->setIcon(share_icon);
    ui->btn_save->setIcon(save_icon);

	ui->btn_share->hide();

    ui->photo->hide();
    ui->progress->hide();
    ui->progress_processing->hide();
    ui->btn_no_photo->show();

    connect(&m_print, SIGNAL(progress(int)), this, SLOT(progress(int)));
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::update_photo()
{
    if (!m_photo.isNull()) {
        ClassicPrint print;
        QImage photo_processed;

        ui->photo->hide();
        ui->progress->show();
        ui->progress_processing->show();
        ui->btn_no_photo->hide();
        ui->progress->setValue(0);

		m_print.process(m_photo, IMAGE_WIDTH, IMAGE_HEIGHT, photo_processed);

        ui->photo->show();
        ui->progress->hide();
        ui->progress_processing->hide();

        // Draw to screen
        QPixmap pix = QPixmap::fromImage(photo_processed);
        ui->photo->setPixmap(pix);
    }
}

void MainWindow::on_actionOpen_Image_triggered()
{
	QString filename = QFileDialog::getOpenFileName(this, "Choose Photo",
													m_print.photoFolder(),
													"*.jpg *.jpeg *.png *.JPG *.JPEG  *.PNG");
    while (QApplication::activeWindow() != this) {
        QCoreApplication::processEvents();
    }
    if (!filename.isEmpty()) {
		QFileInfo info(filename);
		QString dir = info.absoluteDir().path();
		// Store the filename so we can construct a save file
		m_image_filename = info.baseName();
		// See if we have to update the directory stored in the settings
		if (dir != m_print.photoFolder()) {
			// Set the folder so it gets stored in the settings
			m_print.setPhotoFolder(dir);
			save_settings();
		}

		m_photo_full.load(filename);
		if (!m_photo_full.isNull()) {
			m_photo = m_photo_full.scaled(QSize(IMAGE_WIDTH, IMAGE_HEIGHT), Qt::KeepAspectRatio);
        }
    }
    update_photo();
}

void MainWindow::on_btn_edit_lens_clicked()
{
    LensConfiguration lens_config(&m_print, this);
	if (lens_config.exec() == QDialog::Accepted) {
		save_settings();
		update_photo();
	}
}

void MainWindow::on_btn_edit_film_clicked()
{
    FilmConfiguration film_config(&m_print, this);
	if (film_config.exec() == QDialog::Accepted) {
		save_settings();
		update_photo();
	}
}
void MainWindow::on_btn_edit_processing_clicked()
{
    ProcessingConfiguration proc_config(&m_print, this);
	if (proc_config.exec() == QDialog::Accepted) {
		save_settings();
		update_photo();
	}
}


void MainWindow::on_btn_no_photo_clicked()
{
    on_actionOpen_Image_triggered();
}

void MainWindow::progress(int percent) {
    ui->progress->setValue(percent);
    QCoreApplication::processEvents();
}

void MainWindow::on_btn_share_clicked()
{

}

void MainWindow::on_btn_save_clicked()
{
	SaveImageDialog sid(&m_photo_full, this);
	// Restore the last saved resolution
	if ((m_print.saveWidth() > 0) && (m_print.saveHeight() > 0)) {
		sid.setSelectedSize(m_print.saveWidth(), m_print.saveHeight());
	}
	if (sid.exec() == QDialog::Accepted) {
		QString filename = QFileDialog::getSaveFileName(this, "Save photo",
														m_print.saveFolder() +
														QString("/") +
														m_image_filename + QString("_classic.jpg"),
														"*.jpg");
		/*
		while (QApplication::activeWindow() != this) {
			QCoreApplication::processEvents();
		}
		*/
		if (!filename.isEmpty()) {
			QImage processed;

			// Show the progress bar
			ui->photo->hide();
			ui->progress->show();
			ui->progress_processing->show();
			ui->btn_no_photo->hide();
			ui->progress->setValue(0);

			// Process the photo
			// Process it in a background thread so we can continue to update the UI
			/*
			QFuture<void> future = QtConcurrent::run(&m_print, &ClassicPrint::process, m_photo_full, sid.m_selected_width, sid.m_selected_height, processed);
			while (!future.isFinished()) {
				QCoreApplication::processEvents();
				sleep(1);
			}
			*/
			m_print.process(m_photo_full, sid.m_selected_width, sid.m_selected_height, processed);

			if (!processed.isNull()) {
				processed.save(filename);
			}
			// Save the selected directory and resolution for next time
			QFileInfo info(filename);
			QString dir = info.absoluteDir().path();
			if ((dir != m_print.saveFolder()) ||
				(sid.m_selected_width != m_print.saveWidth()) ||
				(sid.m_selected_height != m_print.saveHeight())) {
				m_print.setSaveFolder(dir);
				m_print.setSaveWidth(sid.m_selected_width);
				m_print.setSaveHeight(sid.m_selected_height);
				save_settings();
			}
			// Put the preview image back
			ui->photo->show();
			ui->progress->hide();
			ui->progress_processing->hide();
		}
	}
}

void MainWindow::load_settings() {
	// Look in the home directory first
	QString dir = QDir::homePath();
	dir += "/.classicprint";

	QFileInfo info_dir(dir);
	dir += "/settings.xml";

	QFileInfo info_file(dir);

	// See if the file exists
	if (info_file.exists()) {
		// Try to load it
		if (m_print.load(dir)) {
			// Done
			return;
		}
		QFile::remove(dir);
	}
	// If here then the file is either invalid or doesn't exist
	m_print.load(ClassicPrintSettings::config_dir() + "/settings.xml");
	// Force a save to create the entry in the home directory
	save_settings();
}

void MainWindow::save_settings() {
	QString dir = QDir::homePath();
	dir += "/.classicprint";
	// Make the directory
	QDir d;
	d.mkpath(dir);

	// Save the file
	dir += "/settings.xml";
	m_print.save(dir);
}
