/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

class IMDbPlugin : Plugin {
	private dynamic DBus.Object server;
	List<MovieSource> sources;
	private weak Osso.Context osso_context;

	public override void hello (Gtk.Window window, Osso.Context context) {
		string filename = Path.build_filename (Environment.get_user_cache_dir(),
		                                       "cinaest", "imdb.db", null);
		stdout.printf ("IMDb Plugin Loaded. Cache Db: %s\n", filename);

		if (!FileUtils.test (filename, FileTest.EXISTS)) {
			var note = new Note.confirmation (window, "There is no local copy of the Internet Movie Database (IMDb). Download it now?\n\nThis can be started later from the Settings dialog.");

			var response = note.run ();
			note.destroy ();

			if (response == ResponseType.OK) {
				download_imdb (window);
			}
		}

		var source = new IMDBSource ();

		sources = new List<MovieSource> ();
		sources.append (source);

		osso_context = context;

		// FIXME - this forces the inclusion of config.h
		(void) Config.GETTEXT_PACKAGE;
	}

	private void download_imdb (Gtk.Window window) {
		var picker = new PickerDialog (window);
		var selector = new TouchSelector.text ();
		string[] mirrors;

		try {
			var conn = DBus.Bus.get (DBus.BusType.SESSION);
			server = conn.get_object (IMDbDownloader.DBUS_SERVICE,
			                          IMDbDownloader.DBUS_OBJECT,
			                          IMDbDownloader.DBUS_IFACE);

			mirrors = server.get_mirrors ();
		} catch (DBus.Error e) {
			warning ("Failed connect to IMDb downloader: %s", e.message);
			return;
		}

		foreach (string mirror in mirrors) {
			selector.append_text (mirror);
		}

		picker.set_title ("Please select a source mirror");
		picker.set_selector (selector);

		int res = picker.run();
		string mirror = selector.get_current_text ();
		picker.destroy();

		if (res == ResponseType.OK) {
			var download = new IMDbDownloadDialog (window);
			download.run (server, mirror);
			download.destroy ();
		}
	}

	public override unowned List<MovieSource> get_sources () {
		return sources;
	}

	public override List<MovieAction> get_actions (Movie movie, Gtk.Window window) {
		List<MovieAction> list = null;

		if (movie.year > 0)
			list.append (new MovieAction (_("IMDb page"), on_visit_imdb, movie, window));
		else
			list.append (new MovieAction (_("Lookup on IMDb"), on_visit_imdb, movie, window));

		return list;
	}

	private void on_visit_imdb (Movie movie, Gtk.Window window) {
		var url = "http://www.imdb.com/find?s=tt&q=" + movie.title.replace(" ", "+");
		if (movie.year > 0)
			url += "+(%d)".printf (movie.year);

		var status = osso_context.rpc_run_with_defaults ("osso_browser", "open_new_window", null, 's', url, 'b', false);
		if (status != Osso.Status.OK) {
			var banner = (Banner) Hildon.Banner.show_information_with_markup (window, null, "Failed to open browser.");
			banner.set_timeout (1500);
		}
	}

	public override void settings_dialog (Gtk.Window window) {
		var dialog = new Gtk.Dialog ();
		dialog.set_transient_for (window);
		dialog.set_title (_("IMDb plugin settings"));

		string updated;
		string filename = Path.build_filename (Environment.get_user_cache_dir(),
		                                       "cinaest", "imdb.db", null);
		var file = File.new_for_path (filename);
		try {
			var info = file.query_info (FILE_ATTRIBUTE_TIME_MODIFIED, FileQueryInfoFlags.NONE, null);
			TimeVal tv;
			info.get_modification_time (out tv);

			var date = Date ();
			date.set_time_val (tv);
			char[] s = new char[64];
			date.strftime (s, "%x");
			updated = (string) s;
		} catch (Error e) {
			updated = _("unknown");
		}

		var download = new Hildon.Button.with_text (SizeType.FINGER_HEIGHT, ButtonArrangement.VERTICAL, _("Download"), _("Last update: ") + updated);

		VBox content = (VBox) dialog.get_content_area ();
		content.pack_start (download, true, true, 0);

		download.clicked.connect (() => {
			download_imdb (window);
		});

		dialog.show_all ();
		dialog.run ();
		dialog.destroy ();
	}

	public override unowned string get_name () {
		return "IMDb";
	}
}

class IMDBSource : MovieSource {
	MovieSource.ReceiveMovieFunction _get_callback;
	public override async void get_movies (MovieFilter filter, MovieSource.ReceiveMovieFunction callback, int limit, Cancellable? cancellable) {
		// IMDb has too many movies
		if (filter.title == "")
			return;
		var sqlite = new IMDbSqlite (Path.build_filename (Environment.get_user_cache_dir (),
		                             "cinaest", "imdb.db", null));

		_get_callback = callback;
		yield sqlite.query (filter, receive_movie, cancellable);
	}

	private void receive_movie (string title, int year, int rating, int genres) {
		Movie movie = new Movie ();
		movie.title = title;
		movie.year = year;
		movie.rating = rating;
		movie.genres.field = genres;
		// TODO - depending on settings, this could be something else, like director info or runtime
		movie.secondary = movie.genres.to_string ();
		_get_callback (movie);
	}

	public override void add_movie (Movie movie) {
	}

	public override void delete_movie (Movie movie) {
	}

	public override unowned string get_name () {
		return "IMDb";
	}

	public override unowned string get_description () {
		return "Movies on IMDb";
	}

	public override bool get_editable () {
		return false;
	}
}

[ModuleInit]
public Type register_plugin () {
	// types are registered automatically
	return typeof (IMDbPlugin);
}
