/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Hildon;
using Gtk;

class SettingsDialog : Gtk.Dialog {
	List<Gtk.Button> buttons;
	Gtk.Window movie_list_window;
	private Hildon.CheckButton download_posters;
	private Hildon.CheckButton start_movies;
	private Hildon.Button default_source;

	public SettingsDialog (Gtk.Window window) {
		movie_list_window = window;
		set_transient_for (window);
	}

	construct {
		set_title (_("Settings"));

		VBox vbox;
		if (CinaestProgram.plugins.length () > 4) {
			vbox = new VBox (true, 0);

			var pannable = new PannableArea ();
			pannable.add_with_viewport (vbox);

			VBox area = (VBox) get_content_area ();
			area.pack_start (pannable, true, true, 0);
			area.set_size_request (-1, 5*70);
		} else {
			vbox = (VBox) get_content_area ();
		}

		download_posters = new Hildon.CheckButton (SizeType.FINGER_HEIGHT);
		download_posters.set_label (_("Download movie posters"));
		vbox.pack_start (download_posters, true, true, 0);

		start_movies = new Hildon.CheckButton (SizeType.FINGER_HEIGHT);
		start_movies.set_label (_("Show movie list on startup"));
		vbox.pack_start (start_movies, true, true, 0);

		default_source = new Hildon.Button.with_text (SizeType.FINGER_HEIGHT, ButtonArrangement.VERTICAL, _("Default movie source"), _("None"));
		default_source.set_style (ButtonStyle.PICKER);
		default_source.set_alignment (0, 0.5f, 0, 0.5f);
		vbox.pack_start (default_source, true, true, 0);

		start_movies.toggled.connect (on_start_movies_toggled);
		default_source.clicked.connect (on_default_source_clicked);

		HBox hbox;
		int i = 0;
		buttons = new List<Hildon.Button> ();
		foreach (Plugin plugin in CinaestProgram.plugins) {
			var button = new Gtk.Button.with_label (plugin.get_name ());

			Hildon.gtk_widget_set_theme_size (button, SizeType.FINGER_HEIGHT);
			button.set_alignment(0, 0.5f);

			if (i++ == 0) {
				hbox = new HBox (true, 0);
				vbox.pack_start (hbox, true, true, 0);
			}
			if (i == 2)
				i = 0;
			hbox.pack_start (button, true, true, 0);

			button.clicked.connect (on_plugin_settings);

			buttons.append (button);
		}

		add_button (_("Done"), ResponseType.ACCEPT);
	}

	public void on_start_movies_toggled () {
		default_source.set_sensitive (start_movies.get_active ());
	}

	public void on_default_source_clicked (Gtk.Button button) {
		var default_source = (Hildon.Button) button;
		var dialog = new SourceDialog (movie_list_window);

		MovieSource source = null;
		dialog.run (ref source);

		if (source != null) {
			default_source.set_value (source.get_name ());
		}
	}

	public void on_plugin_settings (Gtk.Button button) {
		int n = buttons.index (button);

		response (n);
	}

	public new int run () {
		int res = 0;
		var gc = GConf.Client.get_default ();

		try {
			download_posters.set_active (gc.get_bool ("/apps/cinaest/download_posters"));
			start_movies.set_active (gc.get_bool ("/apps/cinaest/start_movies"));
			default_source.set_sensitive (start_movies.get_active ());
			string source = gc.get_string ("/apps/cinaest/default_source");
			if (source != null && source != "")
				default_source.set_value (source);
		} catch (Error e) {
			stdout.printf ("Error getting GConf option: %s\n", e.message);
		}

		show_all ();

		do {
			res = base.run ();
			if (res >= 0) {
				var plugin = CinaestProgram.plugins.nth_data (res);
				plugin.settings_dialog (movie_list_window);
			}
		} while (res >= 0);

		if (res == ResponseType.ACCEPT) try {
			gc.set_bool ("/apps/cinaest/download_posters", download_posters.get_active ());
			gc.set_bool ("/apps/cinaest/start_movies", start_movies.get_active ());
			gc.set_string ("/apps/cinaest/default_source", default_source.get_value ());
		} catch (Error e) {
			stdout.printf ("Error setting GConf option: %s\n", e.message);
		}
		destroy ();

		return res;
	}
}
