#include "charmap.h"

#include <QApplication>
#include <QMessageBox>

// Make sure this section is after the all other #includes.
// The reason is that gucharmap.h uses 'signals' as an
// identifier, upsetting Qt which uses it as a keyword.
// So the first thing is to #undef signals, but that means
// nothing Qt related can be #included after this point.
#undef signals
extern "C" {
#include <gucharmap/gucharmap.h>
}

int main(int argc, char *argv[])
{
    QApplication app(argc, argv);
    MainWindow mainWindow;
    return app.exec();
}

MainWindow::MainWindow()
    : labelCharacter(0)
    , labelUnicode(0)
    , labelDescription(0)
    , edit(0)
{
    ui.setupUi(this);

    QApplication::setApplicationName("Character Map");
    QApplication::setApplicationVersion("1.0.1");
    QApplication::setOrganizationName(QApplication::applicationName());
    QApplication::setOrganizationDomain("https://garage.maemo.org/projects/CharMap");

    setWindowIcon(QIcon("/usr/share/icons/hicolor/64x64/apps/charmap.png"));

#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    setAttribute(static_cast<Qt::WidgetAttribute>(130), true);
#else
    setAttribute(Qt::WA_AutoOrientation, true);
#endif

    connect(ui.buttonCopyPortrait, SIGNAL(clicked()), this, SLOT(buttonCopy_clicked()));
    connect(ui.buttonCopyLandscape, SIGNAL(clicked()), this, SLOT(buttonCopy_clicked()));
    connect(ui.buttonClearPortrait, SIGNAL(clicked()), this, SLOT(buttonClear_clicked()));
    connect(ui.buttonClearLandscape, SIGNAL(clicked()), this, SLOT(buttonClear_clicked()));

#if defined(Q_WS_SIMULATOR)
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showMaximized();
#else
    show();
#endif
}

void MainWindow::resizeEvent(QResizeEvent *event)
{
    QMainWindow::resizeEvent(event);

    removeCharacters();
    addCharacters();

    // Store the previous values for portrait-landscape switch
    // Note the first time after inittialization the old values will be NULL
    const QLabel *oldLabelCharacter = labelCharacter;
    const QLabel *oldLabelUnicode = labelUnicode;
    const QLabel *oldLabelDescription = labelDescription;
    const QLineEdit *oldEdit = edit;

    if (event->size().width() < event->size().height())
    {
        labelCharacter = ui.labelCharacterPortrait;
        labelUnicode = ui.labelUnicodePortrait;
        labelDescription = ui.labelDescriptionPortrait;
        edit = ui.editPortrait;

        ui.framePortrait->show();
        ui.frameLandscape->hide();
    }
    else
    {
        labelCharacter = ui.labelCharacterLandscape;
        labelUnicode = ui.labelUnicodeLandscape;
        labelDescription = ui.labelDescriptionLandscape;
        edit = ui.editLandscape;

        ui.framePortrait->hide();
        ui.frameLandscape->show();
    }

    // Shuffle contents between portrait and landscape
    // Not applicable the first time after initialization
    if (oldLabelCharacter && oldLabelUnicode && oldLabelDescription && oldEdit)
    {
        labelCharacter->setText(oldLabelCharacter->text());
        labelUnicode->setText(oldLabelUnicode->text());
        labelDescription->setText(oldLabelDescription->text());
        edit->setText(oldEdit->text());
        edit->setCursorPosition(oldEdit->cursorPosition());
    }
}

static QString expandCharacter(const QFontMetrics &fm, QChar c)
{
    QString text(c);

    const int spacesNeeded((72 - fm.width(c)) / fm.width(' '));
    for (int i = 0; i < spacesNeeded; ++i)
    {
        if (i % 2 == 0)
            text.prepend(' ');
        else
            text.append(' ');
    }
    return text;
}

void MainWindow::addCharacters()
{
    const QFontMetrics fm(ui.chart->fontMetrics());

    for (uint i = 0x20; i <= 0xffff; ++i)
    {
        switch (gucharmap_unichar_type(i))
        {
        default:
            ui.chart->addItem(expandCharacter(fm, i));
        case G_UNICODE_CONTROL:
        case G_UNICODE_UNASSIGNED:
            break;
        }
    }
}

void MainWindow::removeCharacters()
{
    QListWidgetItem *item;
    while ((item = ui.chart->takeItem(0)) != 0)
        delete item;
}

void MainWindow::on_actionAbout_triggered()
{
    QString title("About");
    title += " ";
    title += QApplication::applicationName();

    QString body("<h3>");
    body += QApplication::applicationName();
    body += "</h3> Version ";
    body += QApplication::applicationVersion();
    body += "<p>A very basic character map for Maemo.</p>"
            "&copy; 2014 Peter Pichler</p>"
            "<p>This program is distributed under <a href=\"http://www.gnu.org/licenses/\">GPL3.</a></p>";

    QMessageBox::about(this, title, body);
}

static QChar validCharacter(QString s)
{
    for (QString::ConstIterator it = s.constBegin(); it != s.constEnd(); ++it)
    {
        const QChar c = *it;
        if (c != ' ')
            return c;
    }
    return ' '; // No non-space characters, space must be the value then
}

void MainWindow::on_chart_currentItemChanged(QListWidgetItem *item, QListWidgetItem *)
{
    if (item)
    {
        const QString character(validCharacter(item->text()));
        labelCharacter->setText(character);

        const uint id(character.toUcs4()[0]);
        labelDescription->setText(gucharmap_get_unicode_name(id));

        QString unicode("U+");
        if (id < 0x1000) unicode.append("0");
        if (id < 0x0100) unicode.append("0");
        if (id < 0x0010) unicode.append("0");
        unicode.append(QString::number(id, 16));
        labelUnicode->setText(unicode);
    }
    else
    {
        labelCharacter->clear();
        labelDescription->clear();
        labelUnicode->clear();
    }
}

void MainWindow::on_chart_itemActivated(QListWidgetItem *item)
{
    const QString character(validCharacter(item->text()));
    edit->insert(character);
}

void MainWindow::buttonCopy_clicked()
{
    edit->selectAll();
    edit->copy();
}

void MainWindow::buttonClear_clicked()
{
    edit->clear();
}
