#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 3 of the
#  License, or (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program. If not, see <http://www.gnu.org/licenses/>.
#


import time, os, edje, ecore
from gauges.screen import ScreenController
from common.carmanconfig import CarmanConfig

WAIT_STOP = 1
WAIT_START = 2
WAIT_FINISH = 3

MESSAGE_WAITING_STOP = "Stop your car"
MESSAGE_WAITING_START = "Go!"
MESSAGE_WAITING_FINISH = "Test in progress"

def _conv_time(time):
    return "%02d:%05.2f" % (time / 60, time % 60)

class Test0_100(ScreenController):

    def __init__(self, filename, scr_edje, scr_name, metric):
        ScreenController.__init__(self, scr_edje)
        self._speed = 0
        self._anim = None
        self._total_time = 0
        self.metric = metric
        self._state = WAIT_STOP
        self._scr_edje = scr_edje
        self._config = CarmanConfig()
        self._message_part = edje.file_data_get(filename, "plugin:" + \
            scr_name + ".message_part")
        self._time_part = edje.file_data_get(filename, "plugin:" + \
            scr_name + ".time_part")
        self.set_state_message()

    def set_state_message(self):
        if self._message_part:
            if self._state == WAIT_STOP:
                self._scr_edje.part_text_set(self._message_part,
                                             MESSAGE_WAITING_STOP)
            elif self._state == WAIT_START:
                self._scr_edje.part_text_set(self._message_part,
                                             MESSAGE_WAITING_START)
            elif self._state == WAIT_FINISH:
                self._scr_edje.part_text_set(self._message_part,
                                             MESSAGE_WAITING_FINISH)

    def sensor_data_cb(self, model, pid, *data):
        ScreenController.sensor_data_cb(self, model, pid, *data)
        if pid == "0D":
            self._speed = data[0]
            if self._speed == 0:
                self._state = WAIT_START
                self.set_state_message()
            elif self._state == WAIT_START and self._speed > 0:
                self._state = WAIT_FINISH
                self._start_time = time.time()
                if self._anim:
                    self._anim.delete()
                self._anim = ecore.timer_add(0.05, self.__time_cb)
                self.set_state_message()
            elif self._state == WAIT_FINISH and\
                    ((self.metric and self._speed >= 100) or\
                    (not self.metric and self._speed >=60)):
                self._state = WAIT_STOP
                self._total_time = time.time() - self._start_time
                self.set_state_message()
                self.save_test()

    def save_test(self):
        lines = []
        rank_file = os.path.join(self._config.get_carman_home_dir(),
            "acceleration_ranking")

        if os.path.exists(rank_file):
            fd = open(rank_file, "r")
            lines = fd.readlines()
            fd.close()

        try:
            lines = [line.split() for line in lines]
            lines = [[int(a), float(b)] for a, b in lines]
        except:
            lines = []

        if len(lines) < 5:
            lines.append([int(round(time.time())), self._total_time])
        else:
            i, idx = 0, 0
            stamp, max_time = lines[0]
            for a, b in lines[1:]:
                i += 1
                if b > max_time or (b == max_time and a < stamp):
                    idx = i
                    stamp, max_time = a, b
            if lines[idx][1] >= self._total_time:
                lines[idx] = [int(round(time.time())), self._total_time]
            lines.sort()

        fd = open(rank_file, "w")
        fd.writelines(["%d %.2f\n" % (a, b) for a, b in lines])
        fd.close()

    def __time_cb(self):
        if self._state == WAIT_FINISH:
            self._scr_edje.part_text_set(self._time_part,
                _conv_time(time.time() - self._start_time))
            return True
        elif self._state == WAIT_STOP:
            self._scr_edje.part_text_set(self._time_part,
                _conv_time(self._total_time))
            return False

    def set_unit_system(self, metric):
        ScreenController.set_unit_system(self, metric)
        self.metric = metric

    def deactivate(self):
        ScreenController.deactivate(self)
        if self._anim:
            self._anim.delete()
            self._anim = None
        self._state = WAIT_STOP
        self.set_state_message()
        self._scr_edje.part_text_set(self._time_part,
                _conv_time(0))

def get_screen(filename, scr_edje, scr_name, metric):
    return Test0_100(filename, scr_edje, scr_name, metric)
