#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{SpeedAlertDlgController}.
"""

from common.carlog import DEBUG
from options.speedalertdlgview import SpeedAlertDlgView
from common.carmanconfig import CarmanConfig, METRIC
from models.kbmodel import KeyBoardModel

class SpeedAlertDlgController(object):
    """
    Controls Speed Alert feature.

    @type   parent: L{GlobalOptionsView}
    @param  parent: Parent window.
    """
    def __init__(self, parent):
        self._parent = parent
        self.metric = CarmanConfig().get_unit() == METRIC
        self.sound = CarmanConfig().get_speed_alert()
        self.speed_value = CarmanConfig().get_max_speed()
        self.speed_value = self.__get_speed_value(self.metric)

        self.view = SpeedAlertDlgView(self.speed_value, self.metric)
        self.view.signal_callback_add("more_speed_bt_clicked", "", \
                                      self.__more_cb)
        self.view.signal_callback_add("less_speed_bt_clicked", "", \
                                      self.__less_cb)
        self.view.signal_callback_add("sound_on_bt_clicked", "", \
                                      self.__sound_on_cb)
        self.view.signal_callback_add("sound_off_bt_clicked", "", \
                                      self.__sound_off_cb)
        self.view.signal_callback_add("confirm_bt_clicked", "", \
                                      self.__confirm_cb)
        self.view.signal_callback_add("cancel_bt_clicked", "", \
                                      self.__cancel_cb)
        self.view.signal_callback_add("hide", "", self.__destroy)

        if self.sound == "ON":
            self.view.signal_emit("enable-on", "")
        else:
            self.view.signal_emit("enable-off", "")

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

    def __get_speed_value(self, metric):
        """
        Returns the speed value.

        @type   metric: boolean
        @param  metric: C{True} if speed unit is metric,
            C{False} if speed unit is imperial.
        @rtype: number
        @return: Speed value.
        """
        if metric:
            return self.speed_value
        else:
            q, r = divmod(((self.speed_value *62)/100),10)
            if r <= 2:
                return q*10
            elif r >= 3 and r <= 7:
                return (q*10)+5
            else:
                return (q+1)*10

    def __key_down_cb(self, obj, event):
        """
        Handles key pressed C{ESC}.

        @type   obj: object
        @param  obj: Not used.
        @type   event: EventKeyDown
        @param  event: Event key down object.
        @rtype: boolean
        @return: C{True} if key pressed is C{ESC},
            C{False} otherwise.
        """
        if event.key == "Escape":
            self.view.signal_emit("hide-anime", "")
            return True
        return False

    def __more_cb(self, *data):
        """
        Called when the C{Increment} icon is clicked.

        @type   data: object
        @param  data: Not used.
        """
        self.speed_value += 5
        self.view.value_set(self.speed_value)

    def __less_cb(self, *data):
        """
        Called when the C{Decrement} icon is clicked.

        @type   data: object
        @param  data: Not used.
        """
        if self.speed_value >= 5:
            self.speed_value -= 5
            self.view.value_set(self.speed_value)

    def __sound_on_cb(self, *data):
        """
        Called when the C{Sound on} icon is clicked.

        @type   data: object
        @param  data: Not used.
        """
        self.view.signal_emit("enable-on", "")
        self.view.signal_emit("disable-off", "")
        self.sound = "ON"

    def __sound_off_cb(self, *data):
        """
        Called when the C{Sound off} icon is clicked.

        @type   data: object
        @param  data: Not used.
        """
        self.view.signal_emit("enable-off", "")
        self.view.signal_emit("disable-on", "")
        self.sound = "OFF"

    def __confirm_cb(self, *data):
        """
        Called when C{OK} button is clicked.

        @type   data: object
        @param  data: Not used.
        """
        self.speed_value = self.__get_speed_value(self.metric)
        CarmanConfig().set_max_speed(self.speed_value)
        CarmanConfig().set_speed_alert(self.sound)
        self.view.hide()
        self._parent.hide(True)

    def __cancel_cb(self, *data):
        """
        Called when C{Cancel} button is clicked.

        @type   data: object
        @param  data: Not used.
        """
        self.view.hide()

    def __destroy(self, *data):
        """
        Removes L{__key_down_cb} callback from L{KeyBoardModel} instance.

        @type   data: object
        @param  data: Not used.
        """
        self.kb_model.del_key_down_cb(self.__key_down_cb)

    def show(self):
        """
        Shows Speed Alert dialog.
        """
        self.view.show()
