# tile_downloader.py - Downloads tiles
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{TileDownloader}.
"""

import os
from common.carlog import ERROR
from common.singleton import Singleton
from models.reposet import RepositoriesSet
from models.download_mng import DownloadManager
from common.carmanconfig import CarmanConfig

def wrapper_cb(filename, data, errmsg=None):
    """
    Callback for initializing the L{DownloadManager} object.
    It can be used as a I{complete} or I{error} callback.

    @type   filename: string
    @param  filename: FIlename path.
    @type   data: tuple
    @param  data: Tile data C{(tile_x, tile_y, zoom, [callback])}.
    @type   errmsg: boolean
    @param  errmsg: C{True} if it is an error callback,
                    C{False} if it is a complete callback.
    """
    x, y, z, cb_list = data
    has_error = lambda msg: None if msg else filename
    for cb in cb_list:
        if not callable(cb):
            continue
        cb(has_error(errmsg), x, y, z)


class TileDownloader(Singleton):
    """
    High level class to download tiles.
    """
    def __init__(self):
        Singleton.__init__(self)

        self._mgr = None
        self._list_repos = []
        self._plug_dir = None
        self._repo = None
        self._cache_dir = None
        self._complete_cb = None
        self._errcb = None
        self.config = CarmanConfig()

    def list_repos(self):
        """
        Returns a list of the avaliable repositories.

        @rtype: list
        @return: List of available repositories.
        """
        return self._list_repos

    def choose_repo(self, name):
        """
        Chooses and initializes a repository with the given name.

        @type   name: string
        @param  name: Name of the repository.

        @rtype: module instance
        @return: Instance of the repository module.
        """
        reposet = RepositoriesSet()
        self._repo = reposet.load_repository(name)
        if self._repo is None:
            ERROR("Could not load repository %s, using default repository." % \
                name)
            default_repo = "default_map_repo/osmrepo"
            self._repo = reposet.load_repository(default_repo)
            self.config.set_repository_default_name(default_repo)
        return self._repo

    def get_plugins_dir(self):
        """
        Returns the plugins dir.

        @rtype: string
        @return: Plugins dir path.
        """
        return self._plug_dir

    def initialize(self):
        """
        Initializes a L{DownloadManager} thread object.

        @rtype: file descriptor
        @return: File descriptor to notify of events.
        """
        self._mgr = DownloadManager()

        return self._mgr.initialize(wrapper_cb, wrapper_cb)

    def get_tile_filename(self, tile_x, tile_y, zoom):
        """
        Returns the tile filename path for the given coordinates.

        @type   tile_x: int
        @param  tile_x: Tile X coordinate.
        @type   tile_y: int
        @param  tile_y: Tile Y coordinate.
        @type   zoom: int
        @param  zoom: Zoom level.

        @rtype: string
        @return: Tile filename path.
        """
        return os.path.join(self.get_cache_dir(), str(zoom), str(tile_y),
            str(tile_x) + self._repo.get_extension())

    def request_tile(self, tile_x, tile_y, zoom, foreground=True, cb=None):
        """
        Requests a tile to the download manager.

        @type   tile_x: int
        @param  tile_x: Tile X coordinate.
        @type   tile_y: int
        @param  tile_y: Tile Y coordinate.
        @type   zoom: int
        @param  zoom: Zoom level.
        @type   foreground: boolean
        @param  foreground: C{True} if the download is set to foreground,
                            C{False} if is set to background.
        @type   cb: callback
        @param  cb: Callback to be called when map tile download finishes.
        @rtype: string
        @return: Tile filename path.
        """
        if self._cache_dir is None:
            raise ValueError("The cache dir must exist.")

        filename = self.get_tile_filename(tile_x, tile_y, zoom)

        if os.path.exists(filename):
            return filename

        data = (tile_x, tile_y, zoom, [cb])

        url = self._repo.build_uri(tile_x, tile_y, zoom)
        self._mgr.add_download(url, filename, data, foreground)

    def process_results(self):
        """
        Process the results from the list of downloads.
        Wrapper for L{DownloadManager}.process_results().
        """
        self._mgr.process_results()

    def change_state(self, tile_x, tile_y, zoom, foreground):
        """
        Start a tile download with the given coordinates.

        @type   tile_x: int
        @param  tile_x: Tile X coordinate.
        @type   tile_y: int
        @param  tile_y: Tile Y coordinate.
        @type   zoom: int
        @param  zoom: Zoom level.
        @type   foreground: boolean
        @param  foreground: C{True} if the download is set to foreground,
                            C{False} if is set to background.

        @raise  Exception: The download manager must be initialized.
        """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        url = self._repo.build_uri(tile_x, tile_y, zoom)
        self._mgr.change_state(url, foreground)

    def has_fg_downloads(self):
        """
        Check if there are any downloads in the foreground.
        Wrapper for L{DownloadManager}.has_fg_downloads().

        @rtype: boolean
        @return: C{True} if there are downloads in the foreground,
                C{False} otherwise.
        """
        return self._mgr.has_fg_downloads()

    def has_bg_downloads(self):
        """
        Check if there are any downloads in the background.
        Wrapper for L{DownloadManager}.has_bg_downloads().

        @rtype: boolean
        @return: C{True} if there are downloads in the background,
                C{False} otherwise.
        """
        return self._mgr.has_bg_downloads()

    def has_downloads(self):
        """
        Check if there are any downloads.
        Wrapper for L{DownloadManager}.has_downloads().

        @rtype: boolean
        @return: C{True} if there are any downloads,
                C{False} otherwise.
        """
        return self._mgr.has_downloads()

    def get_cache_dir(self):
        """
        Returns the cache dir path.

        @rtype: string
        @return: Cache dir path.
        """
        return self._cache_dir

    def set_cache_dir(self, cache_dir):
        """
        Sets the cache dir path.

        @type   cache_dir: string
        @param  cache_dir: Cache dir path.
        """
        path = os.path.join(cache_dir, self._repo.get_cache_name())

        if not os.path.exists(path):
            os.makedirs(path)

        self._cache_dir = path

    def stop_all_downloads(self):
        """
        Stop all downloads.
        Wrapper for L{DownloadManager}.stop_all_downloads().

        @raise  Exception: The download manager must be initialized.
        """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        self._mgr.stop_all_downloads()

    def clear_fg_queue(self):
        """
        Clear the foreground download queue.
        Wrapper for L{DownloadManager}.clear_fg_queue().
        """
        self._mgr.clear_fg_queue()

    def clear_bg_queue(self):
        """
        Clear the background download queue.
        Wrapper for L{DownloadManager}.clear_bg_queue().
        """
        self._mgr.clear_bg_queue()

    def clear_queue(self):
        """
        Clear the download queue.
        Wrapper for L{DownloadManager}.clear_queues().

        @raise  ValueError: The download manager must be initialized.
        """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        self._mgr.clear_queues()

    def cancel_download(self, tile_x, tile_y, zoom):
        """
        Cancel a download with the given coordinates.

        @type   tile_x: int
        @param  tile_x: Tile X coordinate.
        @type   tile_y: int
        @param  tile_y: Tile Y coordinate.
        @type   zoom: int
        @param  zoom: Zoom level.
        @raise  ValueError: The download manager must be initialized.
        """
        if self._mgr is None:
            raise ValueError("The download manager must be initialized.")

        url = self._repo.build_uri(tile_x, tile_y, zoom)
        self._mgr.cancel_download(url)

    def close(self):
        """
        Close the download manager.
        """
        self._mgr.close()
        self._mgr.join()
