#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{EventView}.

@var    __THRESHOLD__: Mouse position threshold.
"""

__THRESHOLD__ = 15

import evas

class EventView(evas.SmartObject):
    """
    Common class to handle mouse events.

    @type   canvas: L{evas.Canvas}
    @param  canvas: Carman canvas
    @type   controller: L{MapTileController}
    @param  controller: L{MapTileController} object.
    """
    def __init__(self, canvas, controller):
        evas.SmartObject.__init__(self, canvas)
        self.dragged = False
        self.button_down = False
        self.controller = controller
        self.area = self.Rectangle(color=(0, 0, 0, 0))
        self.area.on_mouse_move_add(self.__on_mouse_move)
        self.area.on_mouse_down_add(self.__on_mouse_down)
        self.area.on_mouse_up_add(self.__on_mouse_up)

    def __on_mouse_move(self, obj, event):
        """
        Called when the mouse changes position while
        a button is pressed.

        @type   obj: object
        @param  obj: Not used.
        @type   event: EventMouseMove
        @param  event: Event mouse move object.
        """
        if self.button_down:
            dx = self.prev_position_x - event.position.output.x
            dy = self.prev_position_y - event.position.output.y
            if abs(dx) > __THRESHOLD__ or abs(dy) > __THRESHOLD__:
                self.dragged = True
                self.prev_position_x = event.position.output.x
                self.prev_position_y = event.position.output.y
                self.controller.move(dx, dy)

    def __on_mouse_down(self, obj, event):
        """
        Called when the mouse is moved down.

        @type   obj: object
        @param  obj: Not used.
        @type   event: EventMouseDown
        @param  event: Event mouse move object.
        """
        self.dragged = False
        self.button_down = True
        self.prev_position_x = event.position.output.x
        self.prev_position_y = event.position.output.y

    def __on_mouse_up(self, obj, event):
        """
        Called when the mouse is moved up or dragged while
        a button is pressed.

        @type   obj: object
        @param  obj: Not used.
        @type   event: EventMouseUp
        @param  event: Event mouse move object.
        """
        if self.button_down and not self.dragged:
            posx, posy = self.pos
            dx = event.position.output.x - posx - self.w / 2
            dy = event.position.output.y - posy - self.h / 2
            self.controller.move(dx, dy)
        self.button_down = False

    def resize(self, w, h):
        """
        Overrides evas.SmartObject resize method.

        @type   w: int
        @param  w: Width.
        @type   h: int
        @param  h: Height.
        """
        self.w = w
        self.h = h
        self.area.resize(w, h)

    def clip_set(self, obj):
        """
        Clips object into area.

        @type   obj: C{evas.Rectangle}
        @param  obj: Object to clip.
        """
        self.area.clip_set(obj)

    def clip_unset(self):
        """
        Unclips all objects inside area.
        """
        self.area.clip_unset()

    def show(self):
        """
        Shows the C{evas.Rectangle} object.
        """
        self.area.show()

    def hide(self):
        """
        Hides the C{evas.Rectangle} object.
        """
        self.area.hide()

