#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{TrackModel}.
"""

from rtree import Rtree

class TrackModel(object):
    """
    Implements GPS tracking feature(stores GPS position history).
    """
    def __init__(self):
        self.clear()
        self.selected_track = None

    def clear(self):
        """
        Clear all stored data.
        """
        self.models = []
        self.current_model = None
        self.empty = True

    def clear_selected_track(self):
        """
        Clear the current selected track.
        """
        self.selected_track = None

    def new_track(self):
        """
        Adds a new track.
        """
        if not self.empty or self.current_model is None:
            self.current_model = Rtree()
            self.empty = True
            self.models.append(self.current_model)

    def add_point(self, point):
        """
        Adds a point to the current model.

        @type   point: tuple
        @param  point: Point C{(x,y)} to be added to the current model.
        """
        self.current_model.add(point)
        self.empty = False

    def get_track_area(self, area):
        """
        Return the track for the given I{area}.

        @type   area: tuple
        @param  area: Area C{(x1, y1, x2, y2)} to be tracked.

        @rtype: list
        @return: List of points that intersect the area.
        """
        track = []
        for model in self.models:
            intersec = model.intersection(area, sort=True, region=True)
            if intersec:
                prev_id, point  = intersec[0]
                point = int(point[0]), int(point[1])
                seg = [point]
                for id, point in intersec:
                    point = int(point[0]), int(point[1])
                    if id - prev_id == 1:
                        seg.append(point)
                    else:
                        if len(seg) > 1:
                            track.append(seg)
                        seg = [point]
                    prev_id = id

                if len(seg) > 1:
                    track.append(seg)
        return track

    def set_models(self, models, trackfile):
        """
        Sets the models and the current selected track.

        @type   models: list
        @param  models: List of models.
        @type   trackfile: string
        @param  trackfile: Track filename path.
        """
        self.empty = True
        self.models = models
        self.current_model = None
        self.selected_track = trackfile

    def get_selected_track(self):
        """
        Returns the current selected track.

        @rtype: string
        @return: Track filename path.
        """
        return self.selected_track

    def finalize(self):
        """
        Finalize the GPS Tracking model.
        """
        pass

